<?php

namespace Modules\Tasks\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Tasks\Models\TimeLog;
use Modules\Tasks\Models\Task;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Exception;

class TimeLogController extends AdminController
{
    /**
     * Check if current user is a full admin
     */
    protected function isFullAdmin(): bool
    {
        return $this->admin->is_admin == 1;
    }
    
    public function index()
    {
        try {
            // Staff can only see their own time logs
            $isAdmin = $this->isFullAdmin();
            
            $stats = [
                'total_entries' => $isAdmin ? TimeLog::count() : TimeLog::byAdmin($this->admin->id)->count(),
                'my_entries' => TimeLog::byAdmin($this->admin->id)->count(),
                'total_hours' => $isAdmin 
                    ? round(TimeLog::sum('duration_minutes') / 60, 2) 
                    : round(TimeLog::byAdmin($this->admin->id)->sum('duration_minutes') / 60, 2),
                'billable_hours' => $isAdmin 
                    ? round(TimeLog::billable()->sum('duration_minutes') / 60, 2)
                    : round(TimeLog::byAdmin($this->admin->id)->billable()->sum('duration_minutes') / 60, 2),
            ];

            return $this->View('tasks::timelogs.index', compact('stats'));

        } catch (Exception $e) {
            Log::error('TimeLogController@index Error: ' . $e->getMessage());
            return back()->with('error', 'Failed to load time logs');
        }
    }

    public function dataTable(Request $request): JsonResponse
    {
        try {
            $query = TimeLog::with(['task.project', 'admin']);
            
            // Staff can only see their own time logs
            if (!$this->isFullAdmin()) {
                $query->where('admin_id', $this->admin->id);
            }

            // Search filter
            if ($search = $request->input('search')) {
                $query->where(function($q) use ($search) {
                    $q->whereHas('task', function($sub) use ($search) {
                        $sub->where('title', 'LIKE', "%{$search}%");
                    })->orWhereHas('admin', function($sub) use ($search) {
                        $sub->where('name', 'LIKE', "%{$search}%");
                    });
                });
            }

            // Advanced filters
            if ($filters = $request->input('filters')) {
                $decoded = is_array($filters) ? $filters : json_decode($filters, true);
                foreach ($decoded ?? [] as $key => $value) {
                    if ($value !== '' && $value !== null) {
                        if ($key === 'is_billable') {
                            $query->where('is_billable', $value);
                        } elseif ($key === 'is_billed') {
                            $query->where('is_billed', $value);
                        } elseif ($key === 'admin_id' && $this->isFullAdmin()) {
                            // Only admins can filter by other users
                            $query->where('admin_id', $value);
                        } elseif ($key === 'is_running') {
                            $query->where('is_running', $value);
                        }
                    }
                }
            }

            // Sorting
            $query->orderBy(
                $request->input('sort', 'id'),
                $request->input('dir', 'desc')
            );

            // Pagination
            $data = $query->paginate($request->input('per_page', 15));

            // Map items with action URLs and location data
            $items = collect($data->items())->map(function ($item) {
                $item->_show_url = route('admin.timelogs.show', $item->id);
                
                // Add formatted start location data for display
                $item->_has_location = $item->has_location;
                $item->_location_string = $item->location_string;
                $item->_google_maps_url = $item->google_maps_url;
                
                // Add formatted end location data for display
                $item->_has_end_location = $item->has_end_location;
                
                // Add accurate duration in seconds
                $item->_working_seconds = $item->working_seconds;
                $item->_formatted_duration = $item->formatted_duration;
                
                return $item;
            });

            return response()->json([
                'data' => $items,
                'total' => $data->total(),
                'current_page' => $data->currentPage(),
                'last_page' => $data->lastPage(),
            ]);

        } catch (Exception $e) {
            Log::error('TimeLogController@dataTable Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to load time log data',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
 * Get time log statistics for AJAX refresh
 */
public function stats(): JsonResponse
{
    try {
        $stats = [
            'total_entries' => TimeLog::count(),
            'my_entries' => TimeLog::where('admin_id', $this->admin->id)->count(),
            'total_hours' => round(TimeLog::sum('duration_minutes') / 60, 2),
            'billable_hours' => round(TimeLog::where('is_billable', true)->sum('duration_minutes') / 60, 2),
        ];

        return response()->json($stats);

    } catch (Exception $e) {
        Log::error('TimeLogController@stats Error: ' . $e->getMessage());
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to load stats'
        ], 500);
    }
}

    /**
     * Display single time log with GPS location details
     */
    public function show($id)
    {
        try {
            $timeLog = TimeLog::with(['task.project', 'admin', 'invoice'])
                ->findOrFail($id);
            
            // Staff can only view their own time logs
            if (!$this->isFullAdmin() && $timeLog->admin_id != $this->admin->id) {
                return redirect()->route('admin.timelogs.index')
                    ->with('error', 'You do not have permission to view this time log.');
            }

            return $this->View('tasks::timelogs.show', compact('timeLog'));

        } catch (Exception $e) {
            Log::error('TimeLogController@show Error: ' . $e->getMessage(), [
                'time_log_id' => $id
            ]);

            return redirect()->route('admin.timelogs.index')
                ->with('error', 'Time log not found');
        }
    }

    public function destroy($id)
    {
        try {
            $timeLog = TimeLog::findOrFail($id);
            
            // Staff can only delete their own time logs
            if (!$this->isFullAdmin() && $timeLog->admin_id != $this->admin->id) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false, 
                        'message' => 'You do not have permission to delete this time log.'
                    ], 403);
                }
                return redirect()->route('admin.timelogs.index')
                    ->with('error', 'You do not have permission to delete this time log.');
            }
            
            // Prevent deletion of billed time logs
            if ($timeLog->is_billed) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false, 
                        'message' => 'Cannot delete billed time log'
                    ], 400);
                }

                return back()->with('error', 'Cannot delete billed time log');
            }

            // Prevent deletion of running timers
            if ($timeLog->is_running) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false, 
                        'message' => 'Cannot delete running timer. Please stop it first.'
                    ], 400);
                }

                return back()->with('error', 'Cannot delete running timer');
            }

            $timeLog->delete();

            Log::info('Time log deleted', [
                'time_log_id' => $id,
                'deleted_by' => $this->admin->id
            ]);

            if (request()->ajax()) {
                return response()->json([
                    'success' => true, 
                    'message' => 'Time log deleted successfully!'
                ]);
            }

            return redirect()->route('admin.timelogs.index')
                ->with('success', 'Time log deleted successfully!');

        } catch (Exception $e) {
            Log::error('TimeLogController@destroy Error: ' . $e->getMessage(), [
                'time_log_id' => $id
            ]);

            if (request()->ajax()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Failed to delete time log'
                ], 500);
            }

            return back()->with('error', 'Failed to delete time log');
        }
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        try {
            $ids = $request->input('ids', []);
            
            if (empty($ids)) {
                return response()->json([
                    'success' => false, 
                    'message' => 'No time logs selected'
                ], 400);
            }
            
            // Staff can only delete their own time logs
            if (!$this->isFullAdmin()) {
                $notOwnedCount = TimeLog::whereIn('id', $ids)
                    ->where('admin_id', '!=', $this->admin->id)
                    ->count();
                    
                if ($notOwnedCount > 0) {
                    return response()->json([
                        'success' => false, 
                        'message' => 'You can only delete your own time logs.'
                    ], 403);
                }
            }

            // Check for billed or running timers
            $billedCount = TimeLog::whereIn('id', $ids)->where('is_billed', true)->count();
            $runningCount = TimeLog::whereIn('id', $ids)->where('is_running', true)->count();

            if ($billedCount > 0) {
                return response()->json([
                    'success' => false, 
                    'message' => "Cannot delete {$billedCount} billed time log(s)"
                ], 400);
            }

            if ($runningCount > 0) {
                return response()->json([
                    'success' => false, 
                    'message' => "Cannot delete {$runningCount} running timer(s). Please stop them first."
                ], 400);
            }

            $deleted = TimeLog::whereIn('id', $ids)->delete();

            Log::info('Bulk time log deletion', [
                'count' => $deleted,
                'deleted_by' => $this->admin->id
            ]);

            return response()->json([
                'success' => true, 
                'message' => "{$deleted} time log(s) deleted successfully!"
            ]);

        } catch (Exception $e) {
            Log::error('TimeLogController@bulkDelete Error: ' . $e->getMessage());

            return response()->json([
                'success' => false, 
                'message' => 'Failed to delete time logs'
            ], 500);
        }
    }
}