<?php

namespace Modules\Tasks\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class TimeLog extends Model
{
    protected $table = 'module_task_time_logs';

    protected $fillable = [
        'task_id',
        'admin_id',
        'start_time',
        'end_time',
        'duration_minutes',
        'description',
        'is_billable',
        'is_billed',
        'is_running',
        'is_paused',  // NEW: Track if currently paused
        'hourly_rate',
        'invoice_id',
        // GPS & Location Tracking - START
        'ip_address',
        'latitude',
        'longitude',
        'location_accuracy',
        'location_address',
        // GPS & Location Tracking - END
        'end_latitude',
        'end_longitude',
        'end_location_accuracy',
        'end_location_address',
        'end_ip_address',
        // Pause tracking
        'total_pause_seconds',  // NEW: Total time spent paused
    ];

    protected $casts = [
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'duration_minutes' => 'integer',
        'is_billable' => 'boolean',
        'is_billed' => 'boolean',
        'is_running' => 'boolean',
        'is_paused' => 'boolean',
        'hourly_rate' => 'decimal:2',
        // GPS Fields
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'location_accuracy' => 'decimal:2',
        'end_latitude' => 'decimal:8',
        'end_longitude' => 'decimal:8',
        'end_location_accuracy' => 'decimal:2',
        'total_pause_seconds' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function task()
    {
        return $this->belongsTo(Task::class, 'task_id');
    }

    public function admin()
    {
        return $this->belongsTo('App\Models\Admin', 'admin_id');
    }

    public function invoice()
    {
        if (class_exists('App\Models\Invoice')) {
            return $this->belongsTo('App\Models\Invoice', 'invoice_id');
        }
        return $this->belongsTo(TimeLog::class, 'invoice_id')->whereRaw('1 = 0');
    }

    /**
     * All pauses for this time log
     */
    public function pauses()
    {
        return $this->hasMany(TimeLogPause::class, 'time_log_id')->orderBy('paused_at', 'asc');
    }

    /**
     * Get the currently active pause (if paused)
     */
    public function activePause()
    {
        return $this->hasOne(TimeLogPause::class, 'time_log_id')
            ->whereNull('resumed_at')
            ->latest('paused_at');
    }

    // ==================== SCOPES ====================

    public function scopeRunning(Builder $query): Builder
    {
        return $query->where('is_running', true);
    }

    public function scopePaused(Builder $query): Builder
    {
        return $query->where('is_paused', true);
    }

    public function scopeActive(Builder $query): Builder
    {
        // Running or paused (not finished)
        return $query->whereNull('end_time');
    }

    public function scopeStopped(Builder $query): Builder
    {
        return $query->where('is_running', false)->where('is_paused', false);
    }

    public function scopeCompleted(Builder $query): Builder
    {
        return $query->whereNotNull('end_time');
    }

    public function scopeBillable(Builder $query): Builder
    {
        return $query->where('is_billable', true);
    }

    public function scopeUnbilled(Builder $query): Builder
    {
        return $query->where('is_billed', false)->where('is_billable', true);
    }

    public function scopeBilled(Builder $query): Builder
    {
        return $query->where('is_billed', true);
    }

    public function scopeByTask(Builder $query, $taskId): Builder
    {
        return $query->where('task_id', $taskId);
    }

    public function scopeByProject(Builder $query, $projectId): Builder
    {
        return $query->whereHas('task', function ($q) use ($projectId) {
            $q->where('project_id', $projectId);
        });
    }

    public function scopeByAdmin(Builder $query, $adminId): Builder
    {
        return $query->where('admin_id', $adminId);
    }

    public function scopeDateRange(Builder $query, $startDate, $endDate): Builder
    {
        return $query->whereBetween('start_time', [$startDate, $endDate]);
    }

    // ==================== ACCESSORS ====================

    public function getDurationHoursAttribute(): float
    {
        // Use working_seconds for accuracy
        $seconds = $this->working_seconds;
        return round($seconds / 3600, 4);
    }

    public function getBillableAmountAttribute(): float
    {
        if (!$this->is_billable || !$this->hourly_rate) {
            return 0;
        }
        // Calculate from actual working time for accuracy
        $hours = $this->working_seconds / 3600;
        return round($hours * $this->hourly_rate, 2);
    }

    /**
     * Get total pause time in minutes
     */
    public function getTotalPauseMinutesAttribute(): float
    {
        return round(($this->total_pause_seconds ?? 0) / 60, 2);
    }

    /**
     * Get number of pauses
     */
    public function getPauseCountAttribute(): int
    {
        return $this->pauses()->count();
    }

    /**
     * Formatted duration HH:MM:SS
     * Calculates from start_time and end_time for accuracy including seconds
     */
    public function getFormattedDurationAttribute(): string
    {
        // If timer has ended, calculate from actual times
        if ($this->start_time && $this->end_time) {
            $totalSeconds = $this->start_time->diffInSeconds($this->end_time);
            $pauseSeconds = $this->total_pause_seconds ?? 0;
            $workingSeconds = max(0, $totalSeconds - $pauseSeconds);
            
            $hours = floor($workingSeconds / 3600);
            $mins = floor(($workingSeconds % 3600) / 60);
            $secs = $workingSeconds % 60;
            return sprintf('%02d:%02d:%02d', $hours, $mins, $secs);
        }
        
        // Fallback to duration_minutes (legacy)
        $minutes = $this->duration_minutes ?? 0;
        $hours = floor($minutes / 60);
        $mins = $minutes % 60;
        return sprintf('%02d:%02d:00', $hours, $mins);
    }
    
    /**
     * Get working seconds (total - pauses)
     */
    public function getWorkingSecondsAttribute(): int
    {
        if ($this->start_time && $this->end_time) {
            $totalSeconds = $this->start_time->diffInSeconds($this->end_time);
            $pauseSeconds = $this->total_pause_seconds ?? 0;
            return max(0, $totalSeconds - $pauseSeconds);
        }
        
        // Fallback to duration_minutes
        return ($this->duration_minutes ?? 0) * 60;
    }

    /**
     * Get current elapsed time (for running timers)
     */
    public function getCurrentElapsedAttribute(): int
    {
        if (!$this->start_time) {
            return 0;
        }

        if ($this->end_time) {
            // Timer finished - return working seconds (accurate with pauses)
            return $this->working_seconds;
        }

        // Calculate current elapsed
        $elapsed = $this->start_time->diffInSeconds(now());
        
        // Subtract total pause time
        $pauseSeconds = $this->total_pause_seconds ?? 0;
        
        // If currently paused, also subtract time since pause started
        if ($this->is_paused) {
            $activePause = $this->activePause;
            if ($activePause && $activePause->paused_at) {
                $pauseSeconds += $activePause->paused_at->diffInSeconds(now());
            }
        }
        
        return max(0, $elapsed - $pauseSeconds);
    }

    /**
     * Generate Google Maps URL for START location
     */
    public function getGoogleMapsUrlAttribute(): string
    {
        if (!$this->latitude || !$this->longitude) {
            return '#';
        }
        return "https://www.google.com/maps?q={$this->latitude},{$this->longitude}";
    }

    /**
     * Generate Google Maps URL for END location
     */
    public function getEndGoogleMapsUrlAttribute(): string
    {
        if (!$this->end_latitude || !$this->end_longitude) {
            return '#';
        }
        return "https://www.google.com/maps?q={$this->end_latitude},{$this->end_longitude}";
    }

    /**
     * Check if has START location
     */
    public function getHasLocationAttribute(): bool
    {
        return !is_null($this->latitude) && !is_null($this->longitude);
    }

    /**
     * Check if has END location
     */
    public function getHasEndLocationAttribute(): bool
    {
        return !is_null($this->end_latitude) && !is_null($this->end_longitude);
    }

    /**
     * Get formatted START location string
     */
    public function getLocationStringAttribute(): string
    {
        if (!$this->has_location) {
            return 'No location data';
        }
        
        $coords = number_format($this->latitude, 6) . ', ' . number_format($this->longitude, 6);
        
        if ($this->location_accuracy) {
            $coords .= ' (±' . round($this->location_accuracy) . 'm)';
        }
        
        return $coords;
    }

    /**
     * Get formatted END location string
     */
    public function getEndLocationStringAttribute(): string
    {
        if (!$this->has_end_location) {
            return 'No location data';
        }
        
        $coords = number_format($this->end_latitude, 6) . ', ' . number_format($this->end_longitude, 6);
        
        if ($this->end_location_accuracy) {
            $coords .= ' (±' . round($this->end_location_accuracy) . 'm)';
        }
        
        return $coords;
    }

    /**
     * Get timer status label
     */
    public function getStatusLabelAttribute(): string
    {
        if ($this->end_time) {
            return 'Completed';
        }
        if ($this->is_paused) {
            return 'Paused';
        }
        if ($this->is_running) {
            return 'Running';
        }
        return 'Unknown';
    }

    // ==================== METHODS ====================

    /**
     * Pause the timer
     */
    public function pauseTimer(array $location = []): TimeLogPause
    {
        // Create pause record
        $pause = $this->pauses()->create([
            'paused_at' => now(),
            'pause_latitude' => $location['latitude'] ?? null,
            'pause_longitude' => $location['longitude'] ?? null,
            'pause_accuracy' => $location['accuracy'] ?? null,
            'pause_ip_address' => $location['ip_address'] ?? null,
        ]);

        // Update timer status
        $this->is_running = false;
        $this->is_paused = true;
        $this->save();

        return $pause;
    }

    /**
     * Resume the timer
     */
    public function resumeTimer(array $location = []): void
    {
        // Find active pause and resume it
        $activePause = $this->pauses()->whereNull('resumed_at')->first();
        
        if ($activePause) {
            $activePause->resume([
                'latitude' => $location['latitude'] ?? null,
                'longitude' => $location['longitude'] ?? null,
                'accuracy' => $location['accuracy'] ?? null,
                'ip_address' => $location['ip_address'] ?? null,
            ]);

            // Update total pause time
            $this->total_pause_seconds = ($this->total_pause_seconds ?? 0) + $activePause->pause_duration_seconds;
        }

        // Update timer status
        $this->is_running = true;
        $this->is_paused = false;
        $this->save();
    }

    /**
     * Stop/Finish the timer
     */
    public function finishTimer(array $location = []): void
    {
        // If paused, first resume to calculate final pause duration
        if ($this->is_paused) {
            $activePause = $this->pauses()->whereNull('resumed_at')->first();
            if ($activePause) {
                $activePause->resume([
                    'latitude' => $location['latitude'] ?? null,
                    'longitude' => $location['longitude'] ?? null,
                    'accuracy' => $location['accuracy'] ?? null,
                    'ip_address' => $location['ip_address'] ?? null,
                ]);
                $this->total_pause_seconds = ($this->total_pause_seconds ?? 0) + $activePause->pause_duration_seconds;
            }
        }

        // Set end time and location
        $this->end_time = now();
        $this->is_running = false;
        $this->is_paused = false;

        if (!empty($location)) {
            $this->end_latitude = $location['latitude'] ?? null;
            $this->end_longitude = $location['longitude'] ?? null;
            $this->end_location_accuracy = $location['accuracy'] ?? null;
            $this->end_ip_address = $location['ip_address'] ?? null;
        }

        // Calculate final working duration
        // Total time - Pause time = Working time
        $totalSeconds = $this->start_time->diffInSeconds($this->end_time);
        $pauseSeconds = $this->total_pause_seconds ?? 0;
        $workingSeconds = max(0, $totalSeconds - $pauseSeconds);
        
        $this->duration_minutes = (int) floor($workingSeconds / 60);
        
        $this->save();
    }

    /**
     * Calculate duration (legacy method)
     */
    public function calculateDuration(): void
    {
        if ($this->start_time && $this->end_time) {
            $totalSeconds = $this->start_time->diffInSeconds($this->end_time);
            $pauseSeconds = $this->total_pause_seconds ?? 0;
            $workingSeconds = max(0, $totalSeconds - $pauseSeconds);
            $this->duration_minutes = (int) floor($workingSeconds / 60);
            $this->save();
        }
    }
}