<?php

namespace Modules\AdvancedEmail\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Illuminate\Http\Request;
use Modules\AdvancedEmail\Models\EmailActivityLog;
use App\Models\Admin\Staff;

class ActivityLogsController extends AdminController
{
    /**
     * Display activity logs
     */
    public function index(Request $request)
    {
        $query = EmailActivityLog::with(['staff', 'targetStaff'])->orderBy('created_at', 'desc');

        // Filters
        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        if ($request->filled('staff_id')) {
            $query->where('staff_id', $request->staff_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->filled('search')) {
            $query->where('description', 'like', '%' . $request->search . '%');
        }

        $logs = $query->paginate(20);

        // Stats
        $stats = [
            'total' => EmailActivityLog::count(),
            'settings_created' => EmailActivityLog::where('action', 'settings_created')->count(),
            'settings_updated' => EmailActivityLog::where('action', 'settings_updated')->count(),
            'test_emails' => EmailActivityLog::whereIn('action', ['test_email_sent', 'test_email_success'])->count(),
        ];

        // Staff list for filter dropdown
        $staffList = Staff::where('status', 1)->orderBy('first_name')->get();

        return view('advancedemail::activity.index', compact('logs', 'stats', 'staffList'));
    }

    /**
     * Clear all activity logs
     */
    public function clearAll()
    {
        EmailActivityLog::truncate();
        return redirect()->back()->with('success', 'All activity logs cleared.');
    }

    /**
     * Clear logs older than X days
     */
    public function clearOld($days)
    {
        EmailActivityLog::where('created_at', '<', now()->subDays($days))->delete();
        return redirect()->back()->with('success', "Activity logs older than {$days} days cleared.");
    }
}
