<?php

namespace Modules\AdvancedEmail\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Illuminate\Http\Request;
use Modules\AdvancedEmail\Models\EmailSendLog;
use App\Models\Admin\Staff;

class EmailLogsController extends AdminController
{
    /**
     * Display email logs
     */
    public function index(Request $request)
    {
        $query = EmailSendLog::with('staff')->orderBy('created_at', 'desc');

        // Filters
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('email_type')) {
            $query->where('email_type', $request->email_type);
        }

        if ($request->filled('staff_id')) {
            $query->where('staff_id', $request->staff_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('from_email', 'like', "%{$search}%")
                  ->orWhere('to_email', 'like', "%{$search}%")
                  ->orWhere('subject', 'like', "%{$search}%");
            });
        }

        $logs = $query->paginate(20);

        // Stats
        $stats = [
            'sent' => EmailSendLog::where('status', 'sent')->count(),
            'failed' => EmailSendLog::where('status', 'failed')->count(),
        ];

        // Staff list for filter dropdown
        $staffList = Staff::where('status', 1)->orderBy('first_name')->get();

        // Email types for filter dropdown
        $emailTypes = [
            'invoice' => 'Invoice',
            'proposal' => 'Proposal',
            'estimate' => 'Estimate',
            'task' => 'Task',
            'project' => 'Project',
            'general' => 'General',
        ];

        return view('advancedemail::logs.index', compact('logs', 'stats', 'staffList', 'emailTypes'));
    }

    /**
     * Show single log entry
     */
    public function show($id)
    {
        $log = EmailSendLog::with('staff')->findOrFail($id);
        return response()->json($log);
    }

    /**
     * Delete log entry
     */
    public function destroy($id)
    {
        $log = EmailSendLog::findOrFail($id);
        $log->delete();

        return redirect()->back()->with('success', 'Log entry deleted.');
    }

    /**
     * Bulk delete logs
     */
    public function bulkDelete(Request $request)
    {
        $ids = $request->input('ids', []);
        
        if (!empty($ids)) {
            EmailSendLog::whereIn('id', $ids)->delete();
            return response()->json(['success' => true, 'message' => count($ids) . ' logs deleted.']);
        }

        return response()->json(['success' => false, 'message' => 'No logs selected.']);
    }

    /**
     * Clear all logs
     */
    public function clearAll(Request $request)
    {
        $retentionDays = $request->input('keep_days', 0);

        if ($retentionDays > 0) {
            EmailSendLog::where('created_at', '<', now()->subDays($retentionDays))->delete();
            $message = "Logs older than {$retentionDays} days cleared.";
        } else {
            EmailSendLog::truncate();
            $message = 'All email logs cleared.';
        }

        return redirect()->back()->with('success', $message);
    }

    /**
     * Export logs to CSV
     */
    public function export(Request $request)
    {
        $query = EmailSendLog::with('staff')->orderBy('created_at', 'desc');

        // Apply same filters as index
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('email_type')) {
            $query->where('email_type', $request->email_type);
        }

        if ($request->filled('staff_id')) {
            $query->where('staff_id', $request->staff_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->get();

        $filename = 'email_logs_' . date('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function () use ($logs) {
            $file = fopen('php://output', 'w');
            
            // Header row
            fputcsv($file, ['Date', 'Staff', 'From', 'To', 'Subject', 'Type', 'Status', 'Error']);

            foreach ($logs as $log) {
                $staffName = $log->staff ? trim(($log->staff->first_name ?? '') . ' ' . ($log->staff->last_name ?? '')) : 'Unknown';
                fputcsv($file, [
                    $log->created_at->format('Y-m-d H:i:s'),
                    $staffName,
                    $log->from_email,
                    $log->to_email,
                    $log->subject,
                    $log->email_type,
                    $log->status,
                    $log->error_message ?? '',
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
