<?php

namespace Modules\AdvancedEmail\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Illuminate\Http\Request;
use Modules\AdvancedEmail\Models\StaffEmailSetting;
use Modules\AdvancedEmail\Models\StaffEmailPermission;
use Modules\AdvancedEmail\Models\EmailActivityLog;
use Modules\AdvancedEmail\Services\AdvancedEmailService;

class EmailSettingsController extends AdminController
{
    protected $emailService;

    public function __construct(AdvancedEmailService $emailService)
    {
        $this->emailService = $emailService;
    }

    /**
     * Get current staff ID
     */
    protected function getStaffId()
    {
        $user = auth()->guard('admin')->user();
        
        // Try staffid directly (Perfex CRM pattern)
        if (isset($user->staffid)) {
            return $user->staffid;
        }
        
        // Try staff relationship (Laravel pattern)
        if ($user->staff) {
            return $user->staff->id ?? $user->staff->staffid;
        }
        
        return $user->id;
    }

    /**
     * Check if current user is admin
     */
    protected function checkIsAdmin(): bool
    {
        $user = auth()->guard('admin')->user();
        return $user->is_admin ?? $user->hasRole('super-admin') ?? false;
    }

    /**
     * Display the email settings form
     */
    public function index()
    {
        $staffId = $this->getStaffId();
        $settings = StaffEmailSetting::getByStaffId($staffId);
        $permission = StaffEmailPermission::getByStaffId($staffId);
        $staff = auth()->guard('admin')->user();

        // Check if user can configure email
        $canConfigure = $this->checkIsAdmin() || ($permission && $permission->can_configure_email);

        return view('advancedemail::settings.index', compact('settings', 'permission', 'staff', 'canConfigure'));
    }

    /**
     * Store or update email settings
     */
    public function store(Request $request)
    {
        $staffId = $this->getStaffId();
        $permission = StaffEmailPermission::getByStaffId($staffId);

        // Check permission
        if (!$this->checkIsAdmin() && (!$permission || !$permission->can_configure_email)) {
            return redirect()->back()->with('error', 'You do not have permission to configure email settings.');
        }

        $validated = $request->validate([
            'email' => 'required|email',
            'from_name' => 'required|string|max:255',
            'mail_driver' => 'required|in:smtp,sendmail,mail',
            'smtp_host' => 'required_if:mail_driver,smtp|nullable|string|max:255',
            'smtp_port' => 'required_if:mail_driver,smtp|nullable|integer|min:1|max:65535',
            'smtp_encryption' => 'nullable|in:,ssl,tls',
            'smtp_username' => 'required_if:mail_driver,smtp|nullable|string|max:255',
            'smtp_password' => 'nullable|string',
            'email_charset' => 'nullable|string|max:50',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->has('is_active');

        $this->emailService->saveSettings($staffId, $validated);

        return redirect()->route('admin.advancedemail.settings.index')
            ->with('success', 'Email settings saved successfully.');
    }

    /**
     * Send test email
     */
    public function sendMyTestEmail(Request $request)
    {
        $request->validate([
            'test_email' => 'required|email',
        ]);

        $staffId = $this->getStaffId();
        $result = $this->emailService->sendTestEmail($staffId, $request->test_email);

        if ($request->ajax()) {
            return response()->json($result);
        }

        if ($result['success']) {
            return redirect()->back()->with('success', $result['message']);
        }

        return redirect()->back()->with('error', $result['message']);
    }

    /**
     * Toggle active status
     */
    public function toggleActive(Request $request)
    {
        $staffId = $this->getStaffId();
        $status = $request->boolean('is_active');

        $this->emailService->toggleActive($staffId, $status);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Status updated successfully',
            ]);
        }

        return redirect()->back()->with('success', 'Email settings status updated.');
    }

    /**
     * Update email type permissions
     */
    public function updateTypePermissions(Request $request)
    {
        $staffId = $this->getStaffId();
        $permission = StaffEmailPermission::getByStaffId($staffId);

        if (!$permission) {
            $permission = StaffEmailPermission::create([
                'staff_id' => $staffId,
                'can_configure_email' => true,
            ]);
        }

        $permission->update([
            'use_own_email_for_invoices' => $request->boolean('use_own_email_for_invoices'),
            'use_own_email_for_proposals' => $request->boolean('use_own_email_for_proposals'),
            'use_own_email_for_estimates' => $request->boolean('use_own_email_for_estimates'),
            'use_own_email_for_tasks' => $request->boolean('use_own_email_for_tasks'),
            'use_own_email_for_projects' => $request->boolean('use_own_email_for_projects'),
        ]);

        return redirect()->back()->with('success', 'Email type preferences updated.');
    }
}
