<?php

namespace Modules\AdvancedEmail\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Illuminate\Http\Request;
use Modules\AdvancedEmail\Models\StaffEmailSetting;
use Modules\AdvancedEmail\Models\StaffEmailPermission;
use Modules\AdvancedEmail\Models\EmailActivityLog;
use Modules\AdvancedEmail\Services\AdvancedEmailService;
use App\Models\Admin\Staff;

class StaffEmailController extends AdminController
{
    protected $emailService;

    public function __construct(AdvancedEmailService $emailService)
    {
        $this->emailService = $emailService;
    }

    /**
     * Display list of all staff with email status
     */
    public function index()
    {
        // Get all active staff
        $staffList = Staff::where('status', 1)
            ->orderBy('first_name')
            ->get()
            ->map(function ($staff) {
                $staffId = $staff->id ?? $staff->staffid;
                $settings = StaffEmailSetting::getByStaffId($staffId);
                $permission = StaffEmailPermission::getByStaffId($staffId);

                return [
                    'id' => $staffId,
                    'staffid' => $staffId,
                    'name' => trim(($staff->first_name ?? $staff->firstname ?? '') . ' ' . ($staff->last_name ?? $staff->lastname ?? '')),
                    'email' => $staff->email,
                    'profile_image' => $staff->profile_image ?? null,
                    'has_settings' => (bool) $settings,
                    'is_configured' => $settings && $settings->smtp_host && $settings->smtp_username,
                    'is_active' => $settings ? $settings->is_active : false,
                    'is_verified' => $settings ? $settings->is_verified : false,
                    'can_configure' => $permission ? $permission->can_configure_email : false,
                    'custom_email' => $settings ? $settings->email : null,
                    'is_admin' => $staff->is_admin ?? false,
                    'default_email' => $staff->email,
                ];
            });

        // Stats
        $stats = [
            'total' => $staffList->count(),
            'configured' => $staffList->where('is_configured', true)->count(),
            'active' => $staffList->where('is_active', true)->count(),
            'with_permission' => $staffList->where('can_configure', true)->count(),
        ];

        return view('advancedemail::staff.index', compact('staffList', 'stats'));
    }

    /**
     * Show edit form for staff email settings
     */
    public function edit($id)
    {
        $staff = Staff::findOrFail($id);
        $staffId = $staff->id ?? $staff->staffid;
        $settings = StaffEmailSetting::getByStaffId($staffId);
        $permission = StaffEmailPermission::getByStaffId($staffId);

        return view('advancedemail::staff.edit', compact('staff', 'settings', 'permission'));
    }

    /**
     * Update staff email settings
     */
    public function update(Request $request, $id)
    {
        $staff = Staff::findOrFail($id);
        $staffId = $staff->id ?? $staff->staffid;

        $validated = $request->validate([
            'email' => 'required|email',
            'from_name' => 'required|string|max:255',
            'mail_driver' => 'required|in:smtp,sendmail,mail',
            'smtp_host' => 'required_if:mail_driver,smtp|nullable|string|max:255',
            'smtp_port' => 'required_if:mail_driver,smtp|nullable|integer|min:1|max:65535',
            'smtp_encryption' => 'nullable|in:,ssl,tls',
            'smtp_username' => 'required_if:mail_driver,smtp|nullable|string|max:255',
            'smtp_password' => 'nullable|string',
            'email_charset' => 'nullable|string|max:50',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->has('is_active');

        $this->emailService->saveSettings($staffId, $validated);

        $staffName = trim(($staff->first_name ?? $staff->firstname ?? '') . ' ' . ($staff->last_name ?? $staff->lastname ?? ''));
        EmailActivityLog::logActivity('settings_updated_by_admin', "Email settings updated for {$staffName}", $staffId);

        return redirect()->route('admin.advancedemail.staff.index')
            ->with('success', "Email settings for {$staffName} updated successfully.");
    }

    /**
     * Delete staff email settings
     */
    public function destroy($id)
    {
        $staff = Staff::findOrFail($id);
        $staffId = $staff->id ?? $staff->staffid;
        
        $this->emailService->deleteSettings($staffId);

        $staffName = trim(($staff->first_name ?? $staff->firstname ?? '') . ' ' . ($staff->last_name ?? $staff->lastname ?? ''));

        return redirect()->route('admin.advancedemail.staff.index')
            ->with('success', "Email settings for {$staffName} deleted.");
    }

    /**
     * Toggle permission to configure email
     */
    public function togglePermission(Request $request, $id)
    {
        $staff = Staff::findOrFail($id);
        $staffId = $staff->id ?? $staff->staffid;
        $canConfigure = $request->boolean('can_configure');

        $adminId = auth()->guard('admin')->user()->id ?? null;

        if ($canConfigure) {
            StaffEmailPermission::grantPermission($staffId, $adminId);
            $action = 'permission_granted';
            $message = 'Permission granted';
        } else {
            StaffEmailPermission::revokePermission($staffId);
            $action = 'permission_revoked';
            $message = 'Permission revoked';
        }

        $staffName = trim(($staff->first_name ?? $staff->firstname ?? '') . ' ' . ($staff->last_name ?? $staff->lastname ?? ''));
        EmailActivityLog::logActivity($action, "{$message} for {$staffName}", $staffId);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => $message,
            ]);
        }

        return redirect()->back()->with('success', $message);
    }

    /**
     * Toggle active status
     */
    public function toggleActive(Request $request, $id)
    {
        $staff = Staff::findOrFail($id);
        $staffId = $staff->id ?? $staff->staffid;
        $status = $request->boolean('is_active');

        $this->emailService->toggleActive($staffId, $status);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Status updated',
            ]);
        }

        return redirect()->back()->with('success', 'Status updated');
    }

    /**
     * Update email type permissions
     */
    public function updateTypePermissions(Request $request, $id)
    {
        $staff = Staff::findOrFail($id);
        $staffId = $staff->id ?? $staff->staffid;
        $permission = StaffEmailPermission::getByStaffId($staffId);

        if (!$permission) {
            $permission = StaffEmailPermission::create([
                'staff_id' => $staffId,
                'can_configure_email' => true,
            ]);
        }

        $permission->update([
            'use_own_email_for_invoices' => $request->boolean('use_own_email_for_invoices'),
            'use_own_email_for_proposals' => $request->boolean('use_own_email_for_proposals'),
            'use_own_email_for_estimates' => $request->boolean('use_own_email_for_estimates'),
            'use_own_email_for_tasks' => $request->boolean('use_own_email_for_tasks'),
            'use_own_email_for_projects' => $request->boolean('use_own_email_for_projects'),
        ]);

        $staffName = trim(($staff->first_name ?? $staff->firstname ?? '') . ' ' . ($staff->last_name ?? $staff->lastname ?? ''));
        EmailActivityLog::logActivity('type_permissions_updated', "Email type permissions updated for {$staffName}", $staffId);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Email type permissions updated',
            ]);
        }

        return redirect()->back()->with('success', 'Email type permissions updated.');
    }

    /**
     * Send test email for staff
     */
    public function sendStaffTestEmail(Request $request, $id)
    {
        $request->validate([
            'test_email' => 'required|email',
        ]);

        $staff = Staff::findOrFail($id);
        $staffId = $staff->id ?? $staff->staffid;
        $result = $this->emailService->sendTestEmail($staffId, $request->test_email);


        return response()->json($result);
                if ($request->ajax()) {
                    return response()->json($result);
                }

                if ($result['success']) {
                    return redirect()->back()->with('success', $result['message']);
                }

                return redirect()->back()->with('error', $result['message']);
    }

    /**
     * Get settings via AJAX
     */
    public function getSettings($id)
    {
        $staff = Staff::findOrFail($id);
        $staffId = $staff->id ?? $staff->staffid;
        $settings = StaffEmailSetting::getByStaffId($staffId);
        $permission = StaffEmailPermission::getByStaffId($staffId);

        return response()->json([
            'success' => true,
            'settings' => $settings,
            'permission' => $permission, 
        ]);
    }



/**
 * Test email using your existing MailService
 */
public function testStaffEmailSimple($id, $email = null)
{
    $staff = Staff::findOrFail($id);
    $staffId = $staff->id ?? $staff->staffid;
    
    // Get staff email settings
    $settings = StaffEmailSetting::getByStaffId($staffId);
    
    echo "<h1>Testing Email for Staff: {$staff->first_name} {$staff->last_name}</h1>";
    echo "<h3>Staff Email: {$staff->email}</h3>";
    
    if ($settings) {
        echo "<h3>Staff Custom Email Settings:</h3>";
        echo "<pre>";
        print_r($settings->toArray());
        echo "</pre>";
        
        // Test using your existing MailService from AdminController
        if ($email) {
            echo "<h3>Testing with Email: {$email}</h3>";
            
            try {
                // Use your existing sendTestEmail method from AdminController
                $fromEmail = $settings->email ?? 'test@echopx.com';
                $fromName = $settings->from_name ?? 'Test Sender';
                
                echo "<h4>Sending using mail() function...</h4>";
                
                // Prepare email content
                $subject = "Test Email from Staff {$staff->first_name}";
                $body = "<html>
                    <body>
                        <h2>Test Email</h2>
                        <p>This is a test email sent from staff member: <strong>{$staff->first_name} {$staff->last_name}</strong></p>
                        <p>Using email: {$fromEmail}</p>
                        <p>Sent at: " . date('d/m/Y h:i A') . "</p>
                    </body>
                </html>";
                
                // Prepare headers
                $headers = [
                    'From: ' . $fromName . ' <' . $fromEmail . '>',
                    'Reply-To: ' . $fromEmail,
                    'MIME-Version: 1.0',
                    'Content-Type: text/html; charset=UTF-8',
                    'X-Mailer: PHP/' . phpversion(),
                ];
                
                $result = mail($email, $subject, $body, implode("\r\n", $headers));
                
                if ($result) {
                    echo "<p style='color: green;'><strong>✓ Email sent successfully!</strong></p>";
                } else {
                    $error = error_get_last();
                    echo "<p style='color: red;'><strong>✗ Failed to send email</strong></p>";
                    echo "<pre>Error: " . print_r($error, true) . "</pre>";
                }
                
            } catch (\Exception $e) {
                echo "<p style='color: red;'><strong>Error: " . $e->getMessage() . "</strong></p>";
            }
        }
        
    } else {
        echo "<p>No custom email settings found for this staff.</p>";
    }
    
    // Form to test
    echo "<hr>";
    echo "<h3>Send Test Email</h3>";
    echo "<form method='get' action=''>";
    echo "<input type='email' name='email' placeholder='test@example.com' required>";
    echo "<input type='submit' value='Send Test Email'>";
    echo "</form>";
    
    echo "<p><a href='" . url()->previous() . "'>Go Back</a></p>";
}



/**
 * Test staff email using the main MailService
 */
public function testUsingMailService($id)
{
    $staff = Staff::findOrFail($id);
    $staffId = $staff->id ?? $staff->staffid;
    
    // Get staff email settings
    $settings = StaffEmailSetting::getByStaffId($staffId);
    
    echo "<h1>Test Staff Email with Main MailService</h1>";
    echo "<h3>Staff: {$staff->first_name} {$staff->last_name}</h3>";
    
    if (!$settings) {
        echo "<p>No custom email settings found.</p>";
        echo "<p><a href='" . route('admin.advancedemail.staff.edit', $id) . "'>Configure Settings</a></p>";
        return;
    }
    
    echo "<h3>Current Settings:</h3>";
    echo "<pre>";
    print_r([
        'email' => $settings->email,
        'from_name' => $settings->from_name,
        'smtp_host' => $settings->smtp_host,
        'smtp_port' => $settings->smtp_port,
        'smtp_username' => $settings->smtp_username,
        'smtp_encryption' => $settings->smtp_encryption,
        'is_active' => $settings->is_active,
    ]);
    echo "</pre>";
    
    // Test email addresses
    $testEmails = ['echopx@gmail.com'];
    
    echo "<h3>Testing with Multiple Recipients:</h3>";
    
    foreach ($testEmails as $testEmail) {
        echo "<h4>Sending to: {$testEmail}</h4>";
        
        try {
            // Prepare email using staff settings
            $subject = "Test Email from Staff - " . date('d/m/Y H:i');
            $body = "
                <div style='font-family: Arial, sans-serif; padding: 20px;'>
                    <h2>Test Email from Staff</h2>
                    <p><strong>Staff:</strong> {$staff->first_name} {$staff->last_name}</p>
                    <p><strong>From Email:</strong> {$settings->email}</p>
                    <p><strong>From Name:</strong> {$settings->from_name}</p>
                    <p><strong>SMTP Host:</strong> {$settings->smtp_host}</p>
                    <p><strong>SMTP Port:</strong> {$settings->smtp_port}</p>
                    <p><strong>Time:</strong> " . date('d/m/Y h:i A') . "</p>
                </div>
            ";
            
            // Use mail() function directly (the same way AdminController does)
            $headers = [
                'From: ' . $settings->from_name . ' <' . $settings->email . '>',
                'Reply-To: ' . $settings->email,
                'MIME-Version: 1.0',
                'Content-Type: text/html; charset=UTF-8',
                'X-Mailer: PHP/' . phpversion(),
                'X-Priority: 3',
            ];
            
            $result = mail($testEmail, $subject, $body, implode("\r\n", $headers));
            
            if ($result) {
                echo "<p style='color: green;'>✓ Sent successfully to {$testEmail}</p>";
            } else {
                $error = error_get_last();
                echo "<p style='color: red;'>✗ Failed to send to {$testEmail}</p>";
                if ($error) {
                    echo "<pre>Error: " . $error['message'] . "</pre>";
                }
            }
            
        } catch (\Exception $e) {
            echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
        }
        
        echo "<hr>";
    }
    
    echo "<h3>Debug Info:</h3>";
    echo "<pre>";
    echo "PHP mail() function exists: " . (function_exists('mail') ? 'Yes' : 'No') . "\n";
    echo "Server OS: " . PHP_OS . "\n";
    echo "PHP Version: " . PHP_VERSION . "\n";
    echo "SMTP Host: " . ini_get('SMTP') . "\n";
    echo "SMTP Port: " . ini_get('smtp_port') . "\n";
    echo "</pre>";
    
    echo "<p><a href='" . url()->previous() . "'>Go Back</a> | ";
    echo "<a href='" . route('admin.advancedemail.staff.edit', $id) . "'>Edit Settings</a></p>";
}




/**
 * Get Hostinger-recommended settings
 */
public function hostingerSettings()
{
    echo "<h1>Hostinger Email Settings</h1>";
    
    echo "<h3>Recommended SMTP Settings:</h3>";
    echo "<pre>";
    echo "SMTP Host: <strong>smtp.hostinger.com</strong>\n";
    echo "SMTP Port: <strong>465</strong> (for SSL) or <strong>587</strong> (for TLS)\n";
    echo "SMTP Username: <strong>Your full email address</strong>\n";
    echo "SMTP Password: <strong>Your email account password</strong>\n";
    echo "Encryption: <strong>ssl</strong> (for port 465) or <strong>tls</strong> (for port 587)\n";
    echo "</pre>";
    
    echo "<h3>OR Use Hostinger's mail() function workaround:</h3>";
    echo "<pre>";
    echo "In php.ini or .htaccess:\n";
    echo "php_value sendmail_path \"/usr/sbin/sendmail -t -i -f your-email@your-domain.com\"\n";
    echo "</pre>";
    
    echo "<h3>Quick Test:</h3>";
    echo "<p>Update staff settings to:</p>";
    echo "<ul>";
    echo "<li>SMTP Host: <code>smtp.hostinger.com</code></li>";
    echo "<li>Port: <code>465</code></li>";
    echo "<li>Encryption: <code>ssl</code></li>";
    echo "<li>Username: <code>googleteam2@echopx.com</code></li>";
    echo "<li>Password: <code>[your email password]</code></li>";
    echo "</ul>";
}



private function sendViaHostingerSMTP($settings, $testEmail)
{
    // Use Hostinger's SMTP
    $host = 'smtp.hostinger.com';
    $port = 465;
    $encryption = 'ssl';
    
    try {
        $transport = (new \Swift_SmtpTransport($host, $port, $encryption))
            ->setUsername($settings->email) // Use the staff's email
            ->setPassword($settings->smtp_password);
        
        $mailer = new \Swift_Mailer($transport);
        
        $message = (new \Swift_Message('Test Email from ' . config('app.name')))
            ->setFrom([$settings->email => $settings->from_name])
            ->setTo($testEmail)
            ->setBody('Test email body', 'text/html');
        
        $result = $mailer->send($message);
        
        return [
            'success' => $result > 0,
            'message' => $result > 0 ? 'Email sent' : 'Failed',
            'sent' => $result
        ];
        
    } catch (\Exception $e) {
        return [
            'success' => false,
            'message' => 'SMTP Error: ' . $e->getMessage()
        ];
    }
}
}
