<?php

namespace Modules\AdvancedEmail\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\Admin\Staff;

class EmailSendLog extends Model
{
    protected $table = 'email_send_logs';

    protected $fillable = [
        'staff_id',
        'from_email',
        'to_email',
        'cc_email',
        'bcc_email',
        'subject',
        'email_type',
        'related_id',
        'status',
        'error_message',
    ];

    /**
     * Relationship to staff
     */
    public function staff()
    {
        return $this->belongsTo(Staff::class, 'staff_id', 'id');
    }

    /**
     * Log a sent email
     */
    public static function logEmail($data)
    {
        $user = auth()->guard('admin')->user();
        $staffId = $data['staff_id'] ?? null;
        
        if (!$staffId && $user) {
            $staffId = $user->staffid ?? ($user->staff ? $user->staff->id : $user->id);
        }

        return static::create([
            'staff_id' => $staffId,
            'from_email' => $data['from_email'],
            'to_email' => $data['to_email'],
            'cc_email' => $data['cc_email'] ?? null,
            'bcc_email' => $data['bcc_email'] ?? null,
            'subject' => $data['subject'],
            'email_type' => $data['email_type'] ?? 'general',
            'related_id' => $data['related_id'] ?? null,
            'status' => $data['status'] ?? 'pending',
            'error_message' => $data['error_message'] ?? null,
        ]);
    }

    /**
     * Mark as sent
     */
    public function markAsSent()
    {
        $this->update(['status' => 'sent']);
    }

    /**
     * Mark as failed
     */
    public function markAsFailed($errorMessage = null)
    {
        $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage,
        ]);
    }

    /**
     * Get status badge class
     */
    public function getStatusBadgeClassAttribute()
    {
        return match($this->status) {
            'sent' => 'module-badge-success',
            'failed' => 'module-badge-danger',
            'pending' => 'module-badge-warning',
            default => 'module-badge-info',
        };
    }

    /**
     * Get email type label
     */
    public function getEmailTypeLabelAttribute()
    {
        $labels = config('advancedemail.email_types', []);
        return $labels[$this->email_type] ?? ucfirst($this->email_type);
    }

    /**
     * Scope by email type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('email_type', $type);
    }

    /**
     * Scope by status
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for current staff
     */
    public function scopeForCurrentStaff($query)
    {
        $user = auth()->guard('admin')->user();
        $staffId = $user->staffid ?? ($user->staff ? $user->staff->id : $user->id);
        return $query->where('staff_id', $staffId);
    }
}
