<?php

namespace Modules\AdvancedEmail\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\Admin\Staff;

class StaffEmailPermission extends Model
{
    protected $table = 'staff_email_permissions';

    protected $fillable = [
        'staff_id',
        'can_configure_email',
        'use_own_email_for_invoices',
        'use_own_email_for_proposals',
        'use_own_email_for_estimates',
        'use_own_email_for_tasks',
        'use_own_email_for_projects',
        'granted_by',
    ];

    protected $casts = [
        'can_configure_email' => 'boolean',
        'use_own_email_for_invoices' => 'boolean',
        'use_own_email_for_proposals' => 'boolean',
        'use_own_email_for_estimates' => 'boolean',
        'use_own_email_for_tasks' => 'boolean',
        'use_own_email_for_projects' => 'boolean',
    ];

    /**
     * Relationship to staff
     */
    public function staff()
    {
        return $this->belongsTo(Staff::class, 'staff_id', 'id');
    }

    /**
     * Relationship to staff who granted permission
     */
    public function grantedByStaff()
    {
        return $this->belongsTo(Staff::class, 'granted_by', 'id');
    }

    /**
     * Get permissions for a staff member
     */
    public static function getByStaffId($staffId)
    {
        return static::firstOrCreate(
            ['staff_id' => $staffId],
            [
                'can_configure_email' => false,
                'use_own_email_for_invoices' => true,
                'use_own_email_for_proposals' => true,
                'use_own_email_for_estimates' => true,
                'use_own_email_for_tasks' => true,
                'use_own_email_for_projects' => true,
            ]
        );
    }

    /**
     * Check if staff can configure email
     */
    public static function canConfigureEmail($staffId)
    {
        // Admin can always configure
        $user = auth()->guard('admin')->user();
        if ($user && ($user->is_admin ?? $user->hasRole('super-admin') ?? false)) {
            return true;
        }

        $permission = static::where('staff_id', $staffId)->first();
        return $permission ? $permission->can_configure_email : false;
    }

    /**
     * Check if staff should use own email for a specific type
     */
    public static function shouldUseOwnEmail($staffId, $emailType)
    {
        $permission = static::getByStaffId($staffId);
        
        $field = "use_own_email_for_{$emailType}";
        
        if (isset($permission->$field)) {
            return $permission->$field;
        }

        return true;
    }

    /**
     * Grant permission to configure email
     */
    public static function grantPermission($staffId, $grantedBy = null)
    {
        $user = auth()->guard('admin')->user();
        $grantedById = $grantedBy;
        
        if (!$grantedById && $user) {
            $grantedById = $user->staffid ?? ($user->staff ? $user->staff->id : $user->id);
        }

        $permission = static::getByStaffId($staffId);
        $permission->update([
            'can_configure_email' => true,
            'granted_by' => $grantedById,
        ]);

        EmailActivityLog::logActivity('permission_granted', 'Email configuration permission granted', $staffId);

        return $permission;
    }

    /**
     * Revoke permission to configure email
     */
    public static function revokePermission($staffId)
    {
        $permission = static::getByStaffId($staffId);
        $permission->update([
            'can_configure_email' => false,
        ]);

        EmailActivityLog::logActivity('permission_revoked', 'Email configuration permission revoked', $staffId);

        return $permission;
    }
}
