<?php

namespace Modules\AdvancedEmail\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Crypt;
use App\Models\Admin\Staff;

class StaffEmailSetting extends Model
{
    protected $table = 'staff_email_settings';

    protected $fillable = [
        'staff_id',
        'email',
        'from_name',
        'mail_driver',
        'smtp_host',
        'smtp_port',
        'smtp_encryption',
        'smtp_username',
        'smtp_password',
        'email_charset',
        'is_active',
        'is_verified',
        'last_verified_at',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_verified' => 'boolean',
        'last_verified_at' => 'datetime',
        'smtp_port' => 'integer',
    ];

    protected $hidden = [
        'smtp_password',
    ];

    /**
     * Encrypt the password before saving
     */
    public function setSmtpPasswordAttribute($value)
    {
        if (!empty($value)) {
            $this->attributes['smtp_password'] = Crypt::encryptString($value);
        }
    }

    /**
     * Decrypt the password when retrieving
     */
    public function getSmtpPasswordDecryptedAttribute()
    {
        $attributes = $this->getAttributes();
        $encrypted = $attributes['smtp_password'] ?? null;
        
        if (empty($encrypted)) {
            return null;
        }
        
        try {
            return Crypt::decryptString($encrypted);
        } catch (\Exception $e) {
            // If decryption fails, check if it's a plain password (legacy)
            if (strpos($encrypted, 'eyJ') !== 0) {
                return $encrypted; // Return as-is if not encrypted
            }
            \Log::warning("SMTP password decryption failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Relationship to staff
     */
    public function staff()
    {
        // Try staffid first (Perfex CRM pattern), then id (Laravel pattern)
        return $this->belongsTo(Staff::class, 'staff_id', 'id');
    }

    /**
     * Get email settings for a specific staff member
     */
    public static function getByStaffId($staffId)
    {
        return static::where('staff_id', $staffId)->first();
    }

    /**
     * Get active email settings for a staff member
     */
    public static function getActiveSettings($staffId)
    {
        return static::where('staff_id', $staffId)
            ->where('is_active', true)
            ->first();
    }

    /**
     * Check if staff has configured email
     */
    public static function hasConfiguredEmail($staffId)
    {
        $settings = static::getByStaffId($staffId);
        return $settings && $settings->is_active && $settings->smtp_host && $settings->smtp_username;
    }

    /**
     * Get mailer configuration array with SSL verification disabled
     * This fixes issues with hosting proxies (like Hostinger) intercepting SMTP
     */
    public function getMailerConfig()
    {
        return [
            'transport' => $this->mail_driver ?? 'smtp',
            'host' => $this->smtp_host,
            'port' => $this->smtp_port ?? 587,
            'encryption' => $this->smtp_encryption ?: null,
            'username' => $this->smtp_username,
            'password' => $this->smtp_password_decrypted,
            'timeout' => null,
            'local_domain' => env('MAIL_EHLO_DOMAIN'),
            // SSL verification bypass for hosting proxies
            'stream' => [
                'ssl' => [
                    'allow_self_signed' => true,
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                ],
            ],
        ];
    }

    /**
     * Get from address configuration
     */
    public function getFromConfig()
    {
        return [
            'address' => $this->email,
            'name' => $this->from_name,
        ];
    }
}
