<?php

namespace Modules\AdvancedEmail\Services;

use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Config;
use Illuminate\Mail\Mailable;
use Modules\AdvancedEmail\Models\StaffEmailSetting;
use Modules\AdvancedEmail\Models\StaffEmailPermission;
use Modules\AdvancedEmail\Models\EmailSendLog;
use Modules\AdvancedEmail\Models\EmailActivityLog;

class AdvancedEmailService
{
    protected $originalMailConfig;
    protected $originalFromConfig;

    /**
     * Module configuration for installation
     */
    public static function config(): array
    {
        return [
            'name' => 'AdvancedEmail',
            'alias' => 'advancedemail',
            'description' => 'Advanced Email System - Configure individual SMTP settings for each staff member',
            'version' => '1.0.0',
            'is_core' => false,
            'sort_order' => 50,
        ];
    }

    /**
     * Menu configuration for installation
     */
    public static function menus(): array
    {
        return [
            // Main Menu - Email Settings
            [
                'menu_name' => 'Email Settings',
                'slug' => 'email-settings',
                'icon' => 'envelope-open-text',
                'route' => null,
                'category' => 'core',
                'sort_order' => 50,
                'actions' => ['read', 'create', 'edit', 'delete'],
                'children' => [
                    [
                        'menu_name' => 'My Email Settings',
                        'slug' => 'my-email-settings',
                        'icon' => 'cog',
                        'route' => 'admin.advancedemail.settings.index',
                        'actions' => ['read', 'edit'],
                    ],
                    [
                        'menu_name' => 'Manage Staff Emails',
                        'slug' => 'manage-staff-emails',
                        'icon' => 'users-cog',
                        'route' => 'admin.advancedemail.staff.index',
                        'actions' => ['read', 'create', 'edit', 'delete'],
                    ],
                    [
                        'menu_name' => 'Email Logs',
                        'slug' => 'email-logs',
                        'icon' => 'list-alt',
                        'route' => 'admin.advancedemail.logs.index',
                        'actions' => ['read', 'delete'],
                    ],
                    [
                        'menu_name' => 'Activity Log',
                        'slug' => 'email-activity-log',
                        'icon' => 'history',
                        'route' => 'admin.advancedemail.activity.index',
                        'actions' => ['read', 'delete'],
                    ],
                ],
            ],
        ];
    }

    /**
     * Called when module is uninstalled
     */
    public static function onUninstall(): void
    {
        // Clean up if needed
    }

    /**
     * Check if module is installed
     */
    public static function isInstalled(): bool
    {
        return class_exists(\Modules\AdvancedEmail\Models\StaffEmailSetting::class);
    }

    /**
     * Get current staff ID
     */
    protected function getCurrentStaffId()
    {
        $user = auth()->guard('admin')->user();
        if (!$user) return null;
        
        return $user->staffid ?? ($user->staff ? $user->staff->id : $user->id);
    }

    /**
     * Get email settings for current user or specified staff
     */
    public function getEmailSettings($staffId = null)
    {
        $staffId = $staffId ?? $this->getCurrentStaffId();
        return StaffEmailSetting::getByStaffId($staffId);
    }

    /**
     * Check if staff has configured and active email settings
     */
    public function hasActiveEmailSettings($staffId = null)
    {
        $staffId = $staffId ?? $this->getCurrentStaffId();
        return StaffEmailSetting::hasConfiguredEmail($staffId);
    }

    /**
     * Configure mailer with staff's email settings
     * Uses sendmail transport to bypass Hostinger SMTP interception
     */
    public function configureMailer($staffId = null, $emailType = 'general')
    {
        $staffId = $staffId ?? $this->getCurrentStaffId();

        // Check if staff should use their own email for this type
        if (!StaffEmailPermission::shouldUseOwnEmail($staffId, $emailType)) {
            return false;
        }

        $settings = StaffEmailSetting::getActiveSettings($staffId);

        if (!$settings) {
            return false;
        }

        // Store original config
        $this->originalMailConfig = Config::get('mail.mailers.sendmail');
        $this->originalFromConfig = Config::get('mail.from');

        // Use sendmail transport (works on Hostinger)
        Config::set('mail.mailers.sendmail', [
            'transport' => 'sendmail',
            'path' => '/usr/sbin/sendmail -t -i',
        ]);
        
        Config::set('mail.from', $settings->getFromConfig());
        Config::set('mail.default', 'sendmail');

        // Purge the mailer instance to use new config
        Mail::purge('sendmail');

        return true;
    }

    /**
     * Restore original mailer configuration
     */
    public function restoreMailer()
    {
        if ($this->originalMailConfig) {
            Config::set('mail.mailers.sendmail', $this->originalMailConfig);
        }

        if ($this->originalFromConfig) {
            Config::set('mail.from', $this->originalFromConfig);
        }

        Mail::purge('sendmail');
    }

    /**
     * Send email using staff's email settings
     * Uses native mail() function for Hostinger compatibility
     */
    public function sendEmail($mailable, $to, $options = [])
    {
        $staffId = $options['staff_id'] ?? $this->getCurrentStaffId();
        $emailType = $options['email_type'] ?? 'general';
        $relatedId = $options['related_id'] ?? null;

        // Get settings
        $settings = StaffEmailSetting::getActiveSettings($staffId);
        
        // Check if should use own email
        $shouldUseOwnEmail = StaffEmailPermission::shouldUseOwnEmail($staffId, str_replace('s', '', $emailType));

        // Log entry
        $logData = [
            'staff_id' => $staffId,
            'from_email' => $settings && $shouldUseOwnEmail ? $settings->email : config('mail.from.address'),
            'to_email' => is_array($to) ? implode(', ', $to) : $to,
            'cc_email' => $options['cc'] ?? null,
            'bcc_email' => $options['bcc'] ?? null,
            'subject' => $mailable->subject ?? 'No Subject',
            'email_type' => $emailType,
            'related_id' => $relatedId,
            'status' => 'pending',
        ];

        $emailLog = EmailSendLog::logEmail($logData);

        try {
            // Configure mailer with staff settings
            $configured = $this->configureMailer($staffId, str_replace('s', '', $emailType));

            // Send email using Laravel Mail with sendmail
            if (is_array($to)) {
                Mail::to($to)->send($mailable);
            } else {
                Mail::to($to)->send($mailable);
            }

            // Mark as sent
            $emailLog->markAsSent();

            // Restore original mailer
            if ($configured) {
                $this->restoreMailer();
            }

            return [
                'success' => true,
                'message' => 'Email sent successfully',
                'log_id' => $emailLog->id,
            ];

        } catch (\Exception $e) {
            // Mark as failed
            $emailLog->markAsFailed($e->getMessage());

            // Restore original mailer
            $this->restoreMailer();

            return [
                'success' => false,
                'message' => $e->getMessage(),
                'log_id' => $emailLog->id,
            ];
        }
    }

    /**
     * Send raw email using native PHP mail() function
     * This bypasses all framework layers and works reliably on Hostinger
     */
    public function sendRawEmail($to, $subject, $body, $options = [])
    {
        $staffId = $options['staff_id'] ?? $this->getCurrentStaffId();
        $settings = StaffEmailSetting::getActiveSettings($staffId);
        
        if (!$settings) {
            return ['success' => false, 'message' => 'No active email settings found'];
        }

        $fromEmail = $settings->email;
        $fromName = $settings->from_name ?? '';
        $isHtml = $options['is_html'] ?? true;
        
        // Build headers
        $headers = [];
        $headers[] = "From: " . ($fromName ? "\"{$fromName}\" <{$fromEmail}>" : $fromEmail);
        $headers[] = "Reply-To: " . ($options['reply_to'] ?? $fromEmail);
        $headers[] = "Return-Path: {$fromEmail}";
        $headers[] = "MIME-Version: 1.0";
        $headers[] = "X-Mailer: PHP/" . phpversion();
        $headers[] = "X-Priority: 3";
        
        if ($isHtml) {
            $headers[] = "Content-Type: text/html; charset=UTF-8";
        } else {
            $headers[] = "Content-Type: text/plain; charset=UTF-8";
        }
        
        // Add CC
        if (!empty($options['cc'])) {
            $cc = is_array($options['cc']) ? implode(', ', $options['cc']) : $options['cc'];
            $headers[] = "Cc: {$cc}";
        }
        
        // Add BCC
        if (!empty($options['bcc'])) {
            $bcc = is_array($options['bcc']) ? implode(', ', $options['bcc']) : $options['bcc'];
            $headers[] = "Bcc: {$bcc}";
        }
        
        $headersString = implode("\r\n", $headers);
        
        // Send
        $toList = is_array($to) ? $to : [$to];
        $allSent = true;
        
        foreach ($toList as $recipient) {
            $result = mail($recipient, $subject, $body, $headersString);
            if (!$result) {
                $allSent = false;
                \Log::error("[AdvancedEmail] Failed to send to {$recipient}");
            }
        }
        
        return [
            'success' => $allSent,
            'message' => $allSent ? 'Email sent successfully' : 'Some emails failed to send',
        ];
    }

    // /**
    //  * Send test email to verify SMTP settings
    //  * Uses sendmail/native mail() function - works on Hostinger
    //  */
    // public function sendTestEmail($staffId, $testEmail)
    // {
    //     $settings = StaffEmailSetting::getByStaffId($staffId);

    //     if (!$settings) {
    //         return [
    //             'success' => false,
    //             'message' => 'Email settings not found',
    //         ];
    //     }

    //     if (!$settings->email) {
    //         return [
    //             'success' => false,
    //             'message' => 'Email address is not set.',
    //         ];
    //     }

    //     EmailActivityLog::logActivity('test_email_sent', "Test email sent to {$testEmail}", $staffId);

    //     try {
    //         $fromEmail = $settings->email;
    //         $fromName = $settings->from_name ?? 'System';
    //         $subject = 'SMTP Test Email - Advanced Email System';
    //         $htmlBody = $this->getTestEmailHtml($settings);

    //         // Build headers like the working MailService
    //         $headers = [];
    //         $headers[] = "From: \"{$fromName}\" <{$fromEmail}>";
    //         $headers[] = "Reply-To: {$fromEmail}";
    //         $headers[] = "Return-Path: {$fromEmail}";
    //         $headers[] = "MIME-Version: 1.0";
    //         $headers[] = "X-Mailer: PHP/" . phpversion();
    //         $headers[] = "X-Priority: 3";
    //         $headers[] = "Content-Type: text/html; charset=UTF-8";
            
    //         $headersString = implode("\r\n", $headers);

    //         // Use PHP's native mail() function
    //         $result = mail($testEmail, $subject, $htmlBody, $headersString);

    //         if ($result) {
    //             // Update verification status
    //             $settings->update([
    //                 'is_verified' => true,
    //                 'last_verified_at' => now(),
    //             ]);

    //             EmailActivityLog::logActivity('test_email_success', "Test email sent successfully to {$testEmail}", $staffId);

    //             return [
    //                 'success' => true,
    //                 'message' => 'Test email sent successfully! Please check your inbox.',
    //             ];
    //         } else {
    //             $error = error_get_last();
    //             $errorMsg = $error ? $error['message'] : 'Unknown error';
                
    //             EmailActivityLog::logActivity('test_email_failed', "Test email failed: {$errorMsg}", $staffId);

    //             return [
    //                 'success' => false,
    //                 'message' => 'Failed to send test email: ' . $errorMsg,
    //             ];
    //         }

    //     } catch (\Exception $e) {
    //         EmailActivityLog::logActivity('test_email_failed', "Test email failed: {$e->getMessage()}", $staffId);
    //         return [
    //             'success' => false,
    //             'message' => 'Failed to send test email: ' . $e->getMessage(),
    //         ];
    //     }
    // }

    /**
     * Send SMTP command (kept for compatibility)
     */
    protected function smtpSendCommand($socket, $command)
    {
        fwrite($socket, $command . "\r\n");
    }

    /**
     * Get SMTP response (kept for compatibility)
     */
    protected function smtpGetResponse($socket)
    {
        $response = '';
        while ($line = fgets($socket, 515)) {
            $response .= $line;
            if (substr($line, 3, 1) === ' ') {
                break;
            }
        }
        return trim($response);
    }

    /**
     * Get HTML content for test email
     */
    protected function getTestEmailHtml($settings)
    {
        $host = $settings->smtp_host;
        $port = $settings->smtp_port;
        $encryption = $settings->smtp_encryption;
        $email = $settings->email;
        $name = $settings->from_name;
        $date = date('Y-m-d H:i:s');

        return "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;'>
            <div style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 30px; border-radius: 10px 10px 0 0;'>
                <h1 style='color: white; margin: 0; text-align: center;'>✉️ SMTP Test Successful!</h1>
            </div>
            <div style='background: #f8f9fa; padding: 30px; border-radius: 0 0 10px 10px;'>
                <p style='color: #28a745; font-size: 18px; text-align: center;'>
                    <strong>✓ Your email settings are configured correctly!</strong>
                </p>
                <hr style='border: none; border-top: 1px solid #dee2e6; margin: 20px 0;'>
                <table style='width: 100%; font-size: 14px; color: #495057;'>
                    <tr><td style='padding: 8px 0;'><strong>From Email:</strong></td><td>{$email}</td></tr>
                    <tr><td style='padding: 8px 0;'><strong>From Name:</strong></td><td>{$name}</td></tr>
                    <tr><td style='padding: 8px 0;'><strong>SMTP Host:</strong></td><td>{$host}</td></tr>
                    <tr><td style='padding: 8px 0;'><strong>SMTP Port:</strong></td><td>{$port}</td></tr>
                    <tr><td style='padding: 8px 0;'><strong>Encryption:</strong></td><td>{$encryption}</td></tr>
                    <tr><td style='padding: 8px 0;'><strong>Sent At:</strong></td><td>{$date}</td></tr>
                </table>
                <hr style='border: none; border-top: 1px solid #dee2e6; margin: 20px 0;'>
                <p style='color: #6c757d; font-size: 12px; text-align: center;'>
                    This is an automated test email from the Advanced Email System.
                </p>
            </div>
        </div>";
    }

    /**
     * Save email settings
     */
    public function saveSettings($staffId, $data)
    {
        $settings = StaffEmailSetting::getByStaffId($staffId);

        $settingsData = [
            'staff_id' => $staffId,
            'email' => $data['email'],
            'from_name' => $data['from_name'],
            'mail_driver' => $data['mail_driver'] ?? 'smtp',
            'smtp_host' => $data['smtp_host'],
            'smtp_port' => $data['smtp_port'] ?? 587,
            'smtp_encryption' => $data['smtp_encryption'] ?? 'tls',
            'smtp_username' => $data['smtp_username'],
            'email_charset' => $data['email_charset'] ?? 'utf-8',
            'is_active' => $data['is_active'] ?? false,
        ];

        // Only update password if provided
        if (!empty($data['smtp_password'])) {
            $settingsData['smtp_password'] = $data['smtp_password'];
        }

        if ($settings) {
            $settings->update($settingsData);
            EmailActivityLog::logActivity('settings_updated', 'Email settings updated', $staffId);
        } else {
            $settings = StaffEmailSetting::create($settingsData);
            EmailActivityLog::logActivity('settings_created', 'Email settings created', $staffId);
        }

        return $settings;
    }

    /**
     * Toggle active status
     */
    public function toggleActive($staffId, $status)
    {
        $settings = StaffEmailSetting::getByStaffId($staffId);

        if ($settings) {
            $settings->update(['is_active' => $status]);
            
            $action = $status ? 'settings_activated' : 'settings_deactivated';
            EmailActivityLog::logActivity($action, 'Email settings ' . ($status ? 'activated' : 'deactivated'), $staffId);

            return true;
        }

        return false;
    }

    /**
     * Delete settings
     */
    public function deleteSettings($staffId)
    {
        $settings = StaffEmailSetting::getByStaffId($staffId);

        if ($settings) {
            $settings->delete();
            EmailActivityLog::logActivity('settings_deleted', 'Email settings deleted', $staffId);
            return true;
        }

        return false;
    }

    /**
     * Get staff list with email settings status
     */
    public function getStaffWithEmailStatus()
    {
        $staff = \App\Models\Staff::select('staffid', 'firstname', 'lastname', 'email', 'is_admin')
            ->where('active', 1)
            ->get();

        return $staff->map(function ($member) {
            $settings = StaffEmailSetting::getByStaffId($member->staffid);
            $permission = StaffEmailPermission::getByStaffId($member->staffid);

            return [
                'staffid' => $member->staffid,
                'name' => $member->firstname . ' ' . $member->lastname,
                'default_email' => $member->email,
                'has_settings' => (bool) $settings,
                'is_configured' => $settings && $settings->smtp_host && $settings->smtp_username,
                'is_active' => $settings ? $settings->is_active : false,
                'is_verified' => $settings ? $settings->is_verified : false,
                'can_configure' => $permission->can_configure_email || $member->is_admin,
                'custom_email' => $settings ? $settings->email : null,
                'is_admin' => $member->is_admin,
            ];
        });
    }




    // In AdvancedEmailService.php
public function sendTestEmail($staffId, $testEmail)
{
    $settings = StaffEmailSetting::getByStaffId($staffId);
    
    if (!$settings) {
        return ['success' => false, 'message' => 'No settings found'];
    }
    
    // If using localhost on Hostinger, switch to Hostinger SMTP
    if ($settings->smtp_host === 'localhost') {
        return $this->sendViaHostingerSMTP($settings, $testEmail);
    }
    
    return $this->sendViaSMTP($settings, $testEmail);
}

}
