<?php

namespace Modules\Attendance\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use App\Traits\DataTable;
use Illuminate\Http\Request;
use Modules\Attendance\Models\Attendance;
use Modules\Attendance\Models\AttendanceLog;
use Modules\Attendance\Models\AttendanceSetting;
use Modules\Attendance\Models\Shift;
use Modules\Attendance\Models\DayOff;
use Modules\Attendance\Models\LeaveApplication;
use App\Models\Admin\Staff;
use App\Models\Admin;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AttendanceController extends AdminController
{
    use DataTable;

    /**
     * Check if current user should track attendance
     */
    protected function shouldTrackAttendance($user = null): bool
    {
        $user = $user ?? auth()->guard('admin')->user();
        
        // 1. Check if super-admin and tracking is disabled
        if ($user->hasRole('super-admin')) {
            $trackSuperAdmin = AttendanceSetting::get('track_super_admin', '0');
            if ($trackSuperAdmin !== '1') {
                return false;
            }
        }

        // 2. Check exempt roles
        $exemptRoles = json_decode(AttendanceSetting::get('exempt_roles', '[]'), true) ?: [];
        if (!empty($exemptRoles)) {
            foreach ($user->roles as $role) {
                if (in_array($role->name, $exemptRoles)) {
                    return false;
                }
            }
        }

        // 3. Check if admin has staff profile
        $staff = $user->staff ?? null;
        if (!$staff) {
            // Admin without staff - check if we should track them
            $trackAdminOnly = AttendanceSetting::get('track_admin_only', '0');
            return $trackAdminOnly === '1';
        }

        // 4. Check if staff is active
        if (!$staff->status) {
            return false;
        }

        // 5. Check exit date
        if ($staff->exit_date && $staff->exit_date < today()) {
            return false;
        }

        return true;
    }

    /**
     * Get or create staff for admin (for check-in)
     */
    protected function getOrCreateStaffForAdmin($user): ?Staff
    {
        // Already has staff
        if ($user->staff) {
            return $user->staff;
        }

        // Check if we should auto-create staff for admins
        $trackAdminOnly = AttendanceSetting::get('track_admin_only', '0');
        if ($trackAdminOnly !== '1') {
            return null;
        }

        // Parse name
        $nameParts = explode(' ', $user->name, 2);
        $firstName = $nameParts[0] ?? 'Admin';
        $lastName = $nameParts[1] ?? '';

        // Generate employee code
        $year = date('Y');
        $lastStaff = Staff::where('employee_code', 'like', "ADM{$year}%")
            ->orderBy('employee_code', 'desc')
            ->first();
        
        if ($lastStaff && preg_match('/ADM' . $year . '(\d+)/', $lastStaff->employee_code, $matches)) {
            $sequence = intval($matches[1]) + 1;
        } else {
            $sequence = 1;
        }
        $employeeCode = 'ADM' . $year . str_pad($sequence, 4, '0', STR_PAD_LEFT);

        // Create staff profile
        try {
            $staff = Staff::create([
                'admin_id' => $user->id,
                'employee_code' => $employeeCode,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'email' => $user->email,
                'join_date' => today(),
                'status' => $user->is_active,
            ]);

            // Refresh relationship
            $user->load('staff');
            
            return $staff;
        } catch (\Exception $e) {
            Log::error('Failed to create staff for admin', [
                'admin_id' => $user->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    // =====================
    // INDEX - Monthly Attendance Grid (Livewire)
    // =====================
    public function index(Request $request)
    {
        $pageTitle = 'Attendance';
        return view('attendance::attendance.index', compact('pageTitle'));
    }

    // =====================
    // DAY DETAIL - AJAX endpoint for modal
    // =====================
    public function dayDetail($staffId, $date)
    {
        $staff = Staff::find($staffId);
        if (!$staff) {
            return response()->json(['error' => 'Staff not found'], 404);
        }
        
        $dateObj = Carbon::parse($date);
        
        // Check for holiday
        $holiday = DayOff::whereDate('off_date', $date)->first();
        if ($holiday) {
            return response()->json([
                'status' => 'Holiday',
                'message' => $holiday->name ?? 'Public Holiday',
                'icon' => '🎉'
            ]);
        }
        
        // Check for leave
        $leave = LeaveApplication::where('staff_id', $staffId)
            ->where('status', 'approved')
            ->whereDate('from_date', '<=', $date)
            ->whereDate('to_date', '>=', $date)
            ->with('leaveType')
            ->first();
        
        if ($leave) {
            return response()->json([
                'status' => 'On Leave',
                'message' => $leave->leaveType->name ?? 'Leave',
                'icon' => '🏖️'
            ]);
        }
        
        // Check for shift
        $shift = Shift::with('shiftType')
            ->where('staff_id', $staffId)
            ->where('status', true)
            ->where('from_date', '<=', $date)
            ->where(function($q) use ($date) {
                $q->whereNull('to_date')
                  ->orWhere('to_date', '>=', $date);
            })
            ->first();
        
        if (!$shift) {
            return response()->json([
                'status' => 'No Shift Assigned',
                'message' => 'Staff has no shift for this date',
                'icon' => '📋'
            ]);
        }
        
        // Check for day off
        if ($shift->shiftType) {
            $daysOff = $shift->shiftType->days_off;
            if (is_string($daysOff)) {
                $daysOff = json_decode($daysOff, true) ?: [];
            }
            if (is_array($daysOff) && in_array($dateObj->format('l'), $daysOff)) {
                return response()->json([
                    'status' => 'Day Off',
                    'message' => $dateObj->format('l') . ' is a day off',
                    'icon' => '😴'
                ]);
            }
        }
        
        // Get attendance record
        $attendance = Attendance::where('staff_id', $staffId)
            ->whereDate('attendance_date', $date)
            ->first();
        
        if (!$attendance) {
            // Check if future
            if ($dateObj->isFuture()) {
                return response()->json([
                    'status' => 'Upcoming',
                    'message' => 'Future date',
                    'icon' => '📅'
                ]);
            }
            return response()->json([
                'status' => 'Absent',
                'message' => 'No attendance record',
                'icon' => '❌'
            ]);
        }
        
        // Get logs
        $logs = AttendanceLog::where('attendance_id', $attendance->id)
            ->whereIn('action', ['check_in', 'check_out'])
            ->orderBy('action_time', 'asc')
            ->get();
        
        if ($logs->isEmpty()) {
            return response()->json([
                'status' => 'Present',
                'message' => 'Checked in at ' . Carbon::parse($attendance->check_in)->format('h:i A'),
                'icon' => '✅'
            ]);
        }
        
        // Calculate times
        $totalSeconds = 0;
        $breakSeconds = 0;
        $lastCheckIn = null;
        $lastCheckOut = null;
        $firstIn = null;
        $lastOut = null;
        
        $logData = [];
        
        foreach ($logs as $log) {
            $time = Carbon::parse($log->action_time);
            
            if ($log->action === 'check_in') {
                if (!$firstIn) $firstIn = $time->format('h:i A');
                
                // Calculate break if there was a previous checkout
                if ($lastCheckOut) {
                    $breakSeconds += $lastCheckOut->diffInSeconds($time);
                }
                $lastCheckIn = $time;
                $lastCheckOut = null;
                
                $logData[] = [
                    'action' => 'check_in',
                    'time' => $time->format('h:i A'),
                    'duration' => null,
                    'location' => $log->location,
                ];
            } else {
                $lastOut = $time->format('h:i A');
                
                if ($lastCheckIn) {
                    $workedSecs = $lastCheckIn->diffInSeconds($time);
                    $totalSeconds += $workedSecs;
                    
                    // Format duration
                    $hours = floor($workedSecs / 3600);
                    $mins = floor(($workedSecs % 3600) / 60);
                    $duration = $hours > 0 ? sprintf('%dh %02dm', $hours, $mins) : sprintf('%dm', $mins);
                    
                    $logData[] = [
                        'action' => 'check_out',
                        'time' => $time->format('h:i A'),
                        'duration' => $duration,
                        'location' => $log->location,
                    ];
                    
                    $lastCheckIn = null;
                } else {
                    $logData[] = [
                        'action' => 'check_out',
                        'time' => $time->format('h:i A'),
                        'duration' => null
                    ];
                }
                $lastCheckOut = $time;
            }
        }
        
        // Format totals
        $workedHours = floor($totalSeconds / 3600);
        $workedMins = floor(($totalSeconds % 3600) / 60);
        $workedTime = $workedHours > 0 ? sprintf('%dh %02dm', $workedHours, $workedMins) : sprintf('%dm', $workedMins);
        
        $breakHours = floor($breakSeconds / 3600);
        $breakMins = floor(($breakSeconds % 3600) / 60);
        $breakTime = $breakHours > 0 ? sprintf('%dh %02dm', $breakHours, $breakMins) : sprintf('%dm', $breakMins);
        
        return response()->json([
            'logs' => $logData,
            'worked_time' => $workedTime,
            'break_time' => $breakTime,
            'first_in' => $firstIn,
            'last_out' => $lastOut
        ]);
    }

    // =====================
    // EXPORT ATTENDANCE
    // =====================
 public function export(Request $request)
    {
        $month = $request->get('month', now()->format('Y-m'));
        $monthDate = Carbon::parse($month . '-01');
        $startDate = $monthDate->copy()->startOfMonth();
        $endDate = $monthDate->copy()->endOfMonth();
        $daysInMonth = $monthDate->daysInMonth;
        
        // Get staff
        $staffQuery = Staff::where('status', true)
            ->whereNotNull('join_date')
            ->where('join_date', '<=', $endDate->format('Y-m-d'));
        
        if ($request->get('department')) {
            $staffQuery->where('department_id', $request->get('department'));
        }
        
        $staffList = $staffQuery->orderBy('first_name')->get();
        $staffIds = $staffList->pluck('id');
        
        // Get attendance data
        $attendances = Attendance::whereBetween('attendance_date', [$startDate, $endDate])
            ->whereIn('staff_id', $staffIds)
            ->get()
            ->groupBy('staff_id');
        
        // Get holidays for the month
        $holidays = DayOff::whereMonth('off_date', $monthDate->month)
            ->whereYear('off_date', $monthDate->year)
            ->pluck('off_date')
            ->map(fn($d) => Carbon::parse($d)->format('Y-m-d'))
            ->toArray();
        
        // Get shifts for all staff
        $shifts = Shift::whereIn('staff_id', $staffIds)
            ->where('status', true)
            ->where('from_date', '<=', $endDate)
            ->where(function($q) use ($startDate) {
                $q->whereNull('to_date')
                  ->orWhere('to_date', '>=', $startDate);
            })
            ->with('shiftType')
            ->get()
            ->groupBy('staff_id');
        
        // Get approved leaves for the month
        $leaves = LeaveApplication::whereIn('staff_id', $staffIds)
            ->where('status', 'approved')
            ->where(function($q) use ($startDate, $endDate) {
                $q->whereBetween('from_date', [$startDate, $endDate])
                  ->orWhereBetween('to_date', [$startDate, $endDate])
                  ->orWhere(function($q2) use ($startDate, $endDate) {
                      $q2->where('from_date', '<=', $startDate)
                         ->where('to_date', '>=', $endDate);
                  });
            })
            ->get()
            ->groupBy('staff_id');
        
        // Build CSV
        $filename = 'attendance_' . $month . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($staffList, $attendances, $monthDate, $daysInMonth, $holidays, $shifts, $leaves, $startDate, $endDate) {
            $file = fopen('php://output', 'w');
            
            // Header row
            $header = ['#', 'Employee Code', 'Staff Name'];
            for ($day = 1; $day <= $daysInMonth; $day++) {
                $date = $monthDate->copy()->day($day);
                $header[] = $date->format('d') . ' (' . $date->format('D') . ')';
            }
            $header[] = 'Total Present';
            $header[] = 'Total Absent';
            $header[] = 'Total Leave';
            $header[] = 'Total Holiday';
            $header[] = 'Total Weekend';
            $header[] = 'Total Hours';
            $header[] = 'Overtime Hours';
            fputcsv($file, $header);
            
            // Data rows
            foreach ($staffList as $index => $staff) {
                $row = [
                    $index + 1, 
                    $staff->employee_code ?? '-',
                    trim($staff->first_name . ' ' . $staff->last_name)
                ];
                
                $totalPresent = 0;
                $totalAbsent = 0;
                $totalLeave = 0;
                $totalHoliday = 0;
                $totalWeekend = 0;
                $totalHours = 0;
                $totalOvertime = 0;
                
                // Get staff's shift
                $staffShift = ($shifts[$staff->id] ?? collect())->first();
                $shiftType = $staffShift?->shiftType;
                
                // Get weekend days from shift type
                $weekendDays = ['Saturday', 'Sunday'];
                if ($shiftType) {
                    $daysOff = $shiftType->days_off ?? $shiftType->weekend_days;
                    if ($daysOff) {
                        $weekendDays = is_array($daysOff) ? $daysOff : (json_decode($daysOff, true) ?: ['Saturday', 'Sunday']);
                    }
                }
                // Normalize weekend days to title case for comparison
                $weekendDays = array_map(fn($d) => ucfirst(strtolower($d)), $weekendDays);
                
                // Get staff's leaves expanded to dates
                $staffLeaveDates = [];
                if (isset($leaves[$staff->id])) {
                    foreach ($leaves[$staff->id] as $leave) {
                        $leaveStart = Carbon::parse($leave->from_date);
                        $leaveEnd = Carbon::parse($leave->to_date);
                        for ($d = $leaveStart->copy(); $d->lte($leaveEnd); $d->addDay()) {
                            $staffLeaveDates[$d->format('Y-m-d')] = $leave->leaveType->name ?? 'Leave';
                        }
                    }
                }
                
                // Shift date range
                $shiftFromDate = $staffShift ? Carbon::parse($staffShift->from_date) : null;
                $shiftToDate = ($staffShift && $staffShift->to_date) ? Carbon::parse($staffShift->to_date) : null;
                
                for ($day = 1; $day <= $daysInMonth; $day++) {
                    $date = $monthDate->copy()->day($day);
                    $dateStr = $date->format('Y-m-d');
                    $dayName = $date->format('l'); // Full day name like "Monday"
                    
                    $att = ($attendances[$staff->id] ?? collect())->firstWhere('attendance_date', $dateStr);
                    
                    // Check if date is before staff joined or after exit
                    if ($staff->join_date && $date->lt(Carbon::parse($staff->join_date))) {
                        $row[] = 'N/J'; // Not Joined
                        continue;
                    }
                    if ($staff->exit_date && $date->gt(Carbon::parse($staff->exit_date))) {
                        $row[] = 'EXT'; // Exited
                        continue;
                    }
                    
                    // Check if shift was active on this day
                    $hasShiftOnThisDay = false;
                    if ($shiftFromDate) {
                        $hasShiftOnThisDay = $date->gte($shiftFromDate) && 
                            ($shiftToDate === null || $date->lte($shiftToDate));
                    }
                    
                    // Future date
                    if ($date->isFuture()) {
                        $row[] = '-';
                        continue;
                    }
                    
                    // Holiday check
                    if (in_array($dateStr, $holidays)) {
                        $row[] = 'H';
                        $totalHoliday++;
                        continue;
                    }
                    
                    // Leave check
                    if (isset($staffLeaveDates[$dateStr])) {
                        $row[] = 'L';
                        $totalLeave++;
                        continue;
                    }
                    
                    // Weekend check
                    if (in_array($dayName, $weekendDays)) {
                        $row[] = 'W';
                        $totalWeekend++;
                        continue;
                    }
                    
                    // No shift assigned
                    if (!$hasShiftOnThisDay) {
                        $row[] = 'NS'; // No Shift
                        continue;
                    }
                    
                    // Check attendance
                    if ($att && $att->check_in) {
                        $totalPresent++;
                        $hours = $att->worked_hours ?? 0;
                        $overtime = $att->overtime_hours ?? 0;
                        $totalHours += $hours;
                        $totalOvertime += $overtime;
                        
                        // Show status with hours
                        $status = 'P';
                        if ($att->status === 'late') {
                            $status = 'LT'; // Late
                        } elseif ($att->status === 'early_leave') {
                            $status = 'EL'; // Early Leave
                        } elseif ($att->status === 'half_day') {
                            $status = 'HD'; // Half Day
                        }
                        
                        if ($hours > 0) {
                            $row[] = $status . ' (' . round($hours, 1) . 'h)';
                        } else {
                            $row[] = $status;
                        }
                    } else {
                        // Absent
                        $row[] = 'A';
                        $totalAbsent++;
                    }
                }
                
                $row[] = $totalPresent;
                $row[] = $totalAbsent;
                $row[] = $totalLeave;
                $row[] = $totalHoliday;
                $row[] = $totalWeekend;
                $row[] = round($totalHours, 1);
                $row[] = round($totalOvertime, 1);
                
                fputcsv($file, $row);
            }
            
            // Add legend at the bottom
            fputcsv($file, []); // Empty row
            fputcsv($file, ['Legend:']);
            fputcsv($file, ['P = Present', 'A = Absent', 'L = Leave', 'H = Holiday', 'W = Weekend']);
            fputcsv($file, ['LT = Late', 'EL = Early Leave', 'HD = Half Day', 'NS = No Shift', 'N/J = Not Joined', 'EXT = Exited']);
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }








    // =====================
    // MARK ATTENDANCE (Admin)
    // =====================
    public function mark(Request $request)
    {
        $pageTitle = 'Mark Attendance';
        $date = $request->get('date', today()->format('Y-m-d'));
        
        // Get all active staff (same logic as index)
        $staffQuery = Staff::where('status', true)
            ->whereNotNull('join_date')
            ->where('join_date', '<=', $date)
            ->where(function($q) use ($date) {
                $q->whereNull('exit_date')
                  ->orWhere('exit_date', '>=', $date);
            });

        // Exclude exempt roles
        $exemptRoles = json_decode(AttendanceSetting::get('exempt_roles', '[]'), true) ?: [];
        if (!empty($exemptRoles)) {
            $staffQuery->whereHas('admin', function($q) use ($exemptRoles) {
                $q->whereDoesntHave('roles', function($rq) use ($exemptRoles) {
                    $rq->whereIn('name', $exemptRoles);
                });
            });
        }

        // Exclude is_admin=1 users if tracking is disabled
        $trackIsAdmin = AttendanceSetting::get('track_is_admin', '0');
        if ($trackIsAdmin !== '1') {
            $staffQuery->whereHas('admin', function($q) {
                $q->where('is_admin', '!=', 1);
            });
        }

        $staffList = $staffQuery->orderBy('first_name')->get();
        
        // Get existing attendance
        $attendances = Attendance::whereDate('attendance_date', $date)
            ->whereIn('staff_id', $staffList->pluck('id'))
            ->get()
            ->keyBy('staff_id');
        
        return view('attendance::attendance.mark', compact('pageTitle', 'staffList', 'attendances', 'date'));
    }

    public function markStore(Request $request)
    {
        $date = $request->input('date', today()->format('Y-m-d'));
        $attendanceData = $request->input('attendance', []);
        $adminStaffId = auth()->guard('admin')->user()->staff->id ?? null;

        foreach ($attendanceData as $staffId => $data) {
            $staff = Staff::find($staffId);
            if (!$staff) continue;
            
            // Validate against join_date and exit_date
            if ($staff->join_date && Carbon::parse($date)->lt($staff->join_date)) continue;
            if ($staff->exit_date && Carbon::parse($date)->gt($staff->exit_date)) continue;

            $checkIn = $data['check_in'] ?? null;
            $checkOut = $data['check_out'] ?? null;
            $status = $data['status'] ?? 'present';
            
            if (!$checkIn && $status === 'present') {
                $status = 'absent';
            }

            $attendance = Attendance::updateOrCreate(
                [
                    'staff_id' => $staffId,
                    'attendance_date' => $date,
                ],
                [
                    'check_in' => $checkIn ?: null,
                    'check_out' => $checkOut ?: null,
                    'status' => $status,
                    'is_manual' => true,
                    'marked_by' => $adminStaffId,
                    'remarks' => $data['remarks'] ?? null,
                ]
            );

            // Calculate hours if both check_in and check_out
            if ($checkIn && $checkOut) {
                $attendance->update([
                    'worked_hours' => $attendance->calculateWorkedHours(),
                    'overtime_hours' => $attendance->calculateOvertimeHours(),
                ]);
            }
        }

        return redirect()->route('admin.attendance.index', ['date' => $date])
            ->with('success', 'Attendance marked successfully');
    }

    // =====================
    // SELF CHECK-IN (Staff/Admin)
    // =====================
    public function checkIn(Request $request)
    {
        $admin = auth()->guard('admin')->user();
        $staffId = $request->input('staff_id');
        
        // If staff_id is 'self' or empty, check-in for current user
        if (!$staffId || $staffId === 'self') {
            return $this->checkInSelf($request);
        }
        
        // Otherwise, admin is marking for another staff
        return $this->checkInForStaff($request, $staffId);
    }

    protected function checkInSelf(Request $request)
    {
        $admin = auth()->guard('admin')->user();
        $staff = Staff::where('admin_id', $admin->id)->first();
        
        // Check if staff has active shift for today
        if ($staff) {
            $hasShift = \Modules\Attendance\Models\Shift::where('staff_id', $staff->id)
                ->where('status', true)
                ->where('from_date', '<=', today())
                ->where(function($q) {
                    $q->whereNull('to_date')
                      ->orWhere('to_date', '>=', today());
                })
                ->exists();
            
            if (!$hasShift) {
                return response()->json([
                    'success' => false,
                    'message' => 'No shift assigned. Please contact admin.'
                ], 400);
            }
        }
        
        // Check if already checked in
        $existing = Attendance::getTodayForCurrentUser();
        if ($existing && $existing->check_in) {
            return response()->json([
                'success' => false,
                'message' => 'Already checked in at ' . $existing->check_in_time
            ], 400);
        }

        // Perform check-in
        $attendance = Attendance::checkInCurrentUser([
            'location' => $request->input('location'),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Checked in successfully',
            'time' => $attendance->check_in_time,
        ]);
    }

    protected function checkInForStaff(Request $request, $staffId)
    {
        $staff = Staff::find($staffId);
        if (!$staff) {
            return response()->json([
                'success' => false,
                'message' => 'Staff not found'
            ], 400);
        }

        // Check if staff has active shift for today
        $hasShift = \Modules\Attendance\Models\Shift::where('staff_id', $staffId)
            ->where('status', true)
            ->where('from_date', '<=', today())
            ->where(function($q) {
                $q->whereNull('to_date')
                  ->orWhere('to_date', '>=', today());
            })
            ->exists();
        
        if (!$hasShift) {
            return response()->json([
                'success' => false,
                'message' => $staff->first_name . ' has no shift assigned for today'
            ], 400);
        }

        // Check if already checked in
        $existing = Attendance::getTodayForStaff($staffId);
        if ($existing && $existing->check_in) {
            return response()->json([
                'success' => false,
                'message' => $staff->first_name . ' already checked in at ' . $existing->check_in_time
            ], 400);
        }

        // Perform check-in
        $attendance = Attendance::checkIn($staffId, [
            'location' => $request->input('location'),
        ]);

        return response()->json([
            'success' => true,
            'message' => $staff->first_name . ' checked in successfully',
            'time' => $attendance->check_in_time,
        ]);
    }

    public function checkOut(Request $request)
    {
        $admin = auth()->guard('admin')->user();
        $staffId = $request->input('staff_id');
        
        // If staff_id is 'self' or empty, check-out for current user
        if (!$staffId || $staffId === 'self') {
            return $this->checkOutSelf($request);
        }
        
        // Otherwise, admin is marking for another staff
        return $this->checkOutForStaff($request, $staffId);
    }

    protected function checkOutSelf(Request $request)
    {
        $attendance = Attendance::getTodayForCurrentUser();
        
        if (!$attendance || !$attendance->check_in) {
            return response()->json([
                'success' => false,
                'message' => 'You have not checked in today'
            ], 400);
        }

        if ($attendance->check_out) {
            return response()->json([
                'success' => false,
                'message' => 'Already checked out at ' . $attendance->check_out_time
            ], 400);
        }

        $attendance = Attendance::checkOutCurrentUser([
            'location' => $request->input('location'),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Checked out successfully',
            'time' => $attendance->check_out_time,
            'worked_hours' => number_format($attendance->worked_hours, 1),
        ]);
    }

    protected function checkOutForStaff(Request $request, $staffId)
    {
        $staff = Staff::find($staffId);
        if (!$staff) {
            return response()->json([
                'success' => false,
                'message' => 'Staff not found'
            ], 400);
        }

        $attendance = Attendance::getTodayForStaff($staffId);
        
        if (!$attendance || !$attendance->check_in) {
            return response()->json([
                'success' => false,
                'message' => $staff->first_name . ' has not checked in today'
            ], 400);
        }

        if ($attendance->check_out) {
            return response()->json([
                'success' => false,
                'message' => $staff->first_name . ' already checked out at ' . $attendance->check_out_time
            ], 400);
        }

        $attendance = Attendance::checkOut($staffId, [
            'location' => $request->input('location'),
        ]);

        return response()->json([
            'success' => true,
            'message' => $staff->first_name . ' checked out successfully',
            'time' => $attendance->check_out_time,
            'worked_hours' => number_format($attendance->worked_hours, 1),
        ]);
    }

    // Get staff status for dropdown change
    public function staffStatus(Request $request)
    {
        $staffId = $request->input('staff_id');
        
        if (!$staffId || $staffId === 'self') {
            $attendance = Attendance::getTodayForCurrentUser();
        } else {
            $attendance = Attendance::getTodayForStaff($staffId);
        }

        return response()->json([
            'success' => true,
            'is_checked_in' => $attendance && $attendance->check_in,
            'is_checked_out' => $attendance && $attendance->check_out,
            'check_in_time' => $attendance?->check_in_time,
            'check_out_time' => $attendance?->check_out_time,
            'worked_hours' => $attendance ? number_format($attendance->worked_hours, 1) : '0',
        ]);
    }

    public function myStatus()
    {
        $user = auth()->guard('admin')->user();
        
        // Check if user should track attendance
        $shouldTrack = $this->shouldTrackAttendance($user);
        
        if (!$shouldTrack) {
            return response()->json([
                'success' => true,
                'should_track' => false,
                'attendance' => null,
                'is_checked_in' => false,
                'is_checked_out' => false,
            ]);
        }

        // Get today's attendance for current user (works for both staff and admin-only)
        $attendance = Attendance::getTodayForCurrentUser();

        return response()->json([
            'success' => true,
            'should_track' => true,
            'attendance' => $attendance,
            'is_checked_in' => $attendance && $attendance->check_in,
            'is_checked_out' => $attendance && $attendance->check_out,
            'check_in_time' => $attendance?->check_in_time,
            'check_out_time' => $attendance?->check_out_time,
            'worked_hours' => $attendance ? number_format($attendance->worked_hours, 1) : '0',
            'worked_duration' => $attendance?->worked_duration ?? '0h 0m',
        ]);
    }

    // =====================
    // VIEW SINGLE ATTENDANCE
    // =====================
    public function show($id)
    {
        $attendance = Attendance::with(['staff', 'admin', 'shift', 'logs.performedByAdmin'])->findOrFail($id);
        $pageTitle = 'Attendance Details';
        
        return view('attendance::attendance.show', compact('attendance', 'pageTitle'));
    }

    // =====================
    // EDIT ATTENDANCE
    // =====================
    public function edit($id)
    {
        $attendance = Attendance::with(['staff', 'admin'])->findOrFail($id);
        $pageTitle = 'Edit Attendance';
        
        return view('attendance::attendance.edit', compact('attendance', 'pageTitle'));
    }

    public function update(Request $request, $id)
    {
        $attendance = Attendance::findOrFail($id);
        $adminId = auth()->guard('admin')->id();

        $validated = $request->validate([
            'check_in' => 'nullable|date_format:H:i',
            'check_out' => 'nullable|date_format:H:i',
            'status' => 'required|in:present,absent,half_day,late,early_leave,on_leave,holiday,weekend',
            'remarks' => 'nullable|string|max:500',
        ]);

        $attendance->update([
            'check_in' => $validated['check_in'] ?? $attendance->check_in,
            'check_out' => $validated['check_out'] ?? $attendance->check_out,
            'status' => $validated['status'],
            'remarks' => $validated['remarks'],
            'is_manual' => true,
            'marked_by' => $adminId,
        ]);

        // Recalculate hours
        if ($attendance->check_in && $attendance->check_out) {
            $attendance->update([
                'worked_hours' => $attendance->calculateWorkedHours(),
                'overtime_hours' => $attendance->calculateOvertimeHours(),
            ]);
        }

        // Log the manual edit
        $attendance->logs()->create([
            'action' => 'manual_edit',
            'action_time' => now()->format('H:i:s'),
            'ip_address' => request()->ip(),
            'device_info' => request()->userAgent(),
            'performed_by' => $adminId,
            'notes' => 'Manual edit by admin',
        ]);

        return redirect()->route('admin.attendance.index', ['date' => $attendance->attendance_date->format('Y-m-d')])
            ->with('success', 'Attendance updated successfully');
    }

    // =====================
    // DELETE ATTENDANCE
    // =====================
    public function destroy($id)
    {
        $attendance = Attendance::findOrFail($id);
        $date = $attendance->attendance_date->format('Y-m-d');
        $attendance->delete();

        return response()->json([
            'success' => true,
            'message' => 'Attendance record deleted',
            'redirect' => route('admin.attendance.index', ['date' => $date])
        ]);
    }

    // =====================
    // MY ATTENDANCE (Staff/Admin View)
    // =====================
    public function myAttendance(Request $request)
    {
        $pageTitle = 'My Attendance';
        $user = auth()->guard('admin')->user();
        
        // Check if should track attendance
        if (!$this->shouldTrackAttendance($user)) {
            return redirect()->route('admin.dashboard')
                ->with('info', 'Attendance tracking is not enabled for your account');
        }

        $month = $request->get('month', now()->month);
        $year = $request->get('year', now()->year);

        // Get staff and shift info
        $staff = Staff::where('admin_id', $user->id)->first();
        $shift = null;
        $shiftType = null;
        
        // Build date range for the selected month
        $monthStart = Carbon::create($year, $month, 1)->startOfMonth();
        $monthEnd = Carbon::create($year, $month, 1)->endOfMonth();
        
        if ($staff) {
            // Check shift with date range (same logic as Admin Dashboard)
            $shift = Shift::where('staff_id', $staff->id)
                ->where('status', true)
                ->where('from_date', '<=', $monthEnd)
                ->where(function($q) use ($monthStart) {
                    $q->whereNull('to_date')
                      ->orWhere('to_date', '>=', $monthStart);
                })
                ->with('shiftType')
                ->first();
            $shiftType = $shift?->shiftType;
        }

        // Get attendance for current user
        $attendances = Attendance::forCurrentUser()
            ->month($month, $year)
            ->orderBy('attendance_date', 'desc')
            ->get()
            ->keyBy(function($item) {
                return Carbon::parse($item->attendance_date)->format('Y-m-d');
            });

        // Get holidays for the month
        $holidays = DayOff::whereMonth('off_date', $month)
            ->whereYear('off_date', $year)
            ->get()
            ->keyBy(function($item) {
                return Carbon::parse($item->off_date)->format('Y-m-d');
            });

        // Get approved leaves for the month
        $leaves = collect();
        if ($staff) {
            $monthStart = Carbon::create($year, $month, 1)->startOfMonth();
            $monthEnd = Carbon::create($year, $month, 1)->endOfMonth();
            
            $leaveRecords = LeaveApplication::where('staff_id', $staff->id)
                ->where('status', 'approved')
                ->where(function($q) use ($monthStart, $monthEnd) {
                    $q->whereBetween('from_date', [$monthStart, $monthEnd])
                      ->orWhereBetween('to_date', [$monthStart, $monthEnd])
                      ->orWhere(function($q2) use ($monthStart, $monthEnd) {
                          $q2->where('from_date', '<=', $monthStart)
                             ->where('to_date', '>=', $monthEnd);
                      });
                })
                ->with('leaveType')
                ->get();
            
            // Expand leaves to individual dates
            foreach ($leaveRecords as $leave) {
                $start = Carbon::parse($leave->from_date);
                $end = Carbon::parse($leave->to_date);
                
                for ($date = $start->copy(); $date->lte($end); $date->addDay()) {
                    if ($date->month == $month && $date->year == $year) {
                        $leaves[$date->format('Y-m-d')] = $leave;
                    }
                }
            }
        }

        // Build calendar data
        $calendarData = [];
        $startOfMonth = Carbon::create($year, $month, 1);
        $endOfMonth = $startOfMonth->copy()->endOfMonth();
        $daysInMonth = $endOfMonth->day;
        
        // Get weekend days from shift type or default
        $weekendDays = ['saturday', 'sunday'];
        if ($shiftType && $shiftType->weekend_days) {
            $weekendDays = is_array($shiftType->weekend_days) 
                ? $shiftType->weekend_days 
                : json_decode($shiftType->weekend_days, true) ?? ['saturday', 'sunday'];
        }

        // Get shift date range for per-day validation
        $shiftFromDate = $shift ? Carbon::parse($shift->from_date) : null;
        $shiftToDate = ($shift && $shift->to_date) ? Carbon::parse($shift->to_date) : null;

        for ($day = 1; $day <= $daysInMonth; $day++) {
            $date = Carbon::create($year, $month, $day);
            $dateKey = $date->format('Y-m-d');
            $dayName = strtolower($date->format('l'));
            
            // Check if shift was active on THIS specific day
            $hasShiftOnThisDay = false;
            if ($shift && $shiftFromDate) {
                $hasShiftOnThisDay = $date->gte($shiftFromDate) && 
                    ($shiftToDate === null || $date->lte($shiftToDate));
            }
            
            $dayData = [
                'date' => $date,
                'day' => $day,
                'dayName' => $date->format('D'),
                'isToday' => $date->isToday(),
                'isWeekend' => in_array($dayName, $weekendDays),
                'isFuture' => $date->isFuture(),
                'status' => 'none',
                'hours' => 0,
                'attendance' => null,
                'holiday' => null,
                'leave' => null,
            ];

            // If no shift on this specific day, show no_shift for past days
            if (!$hasShiftOnThisDay && !$dayData['isFuture']) {
                $dayData['status'] = 'no_shift';
            }
            // Check holiday
            elseif (isset($holidays[$dateKey])) {
                $dayData['status'] = 'holiday';
                $dayData['holiday'] = $holidays[$dateKey];
            }
            // Check leave
            elseif (isset($leaves[$dateKey])) {
                $dayData['status'] = 'leave';
                $dayData['leave'] = $leaves[$dateKey];
            }
            // Check weekend (only if has shift)
            elseif ($dayData['isWeekend'] && $hasShiftOnThisDay) {
                $dayData['status'] = 'weekend';
            }
            // Check attendance
            elseif (isset($attendances[$dateKey])) {
                $att = $attendances[$dateKey];
                $dayData['attendance'] = $att;
                $dayData['hours'] = $att->worked_hours ?? 0;
                $dayData['status'] = $att->status ?? 'present';
            }
            // Past day with no attendance = absent (only for working days WITH shift on that day)
            elseif (!$dayData['isFuture'] && !$dayData['isWeekend'] && $hasShiftOnThisDay) {
                $dayData['status'] = 'absent';
            }

            $calendarData[$day] = $dayData;
        }

        // Calculate stats
        $stats = [
            'present' => $attendances->whereIn('status', ['present', 'late', 'early_leave'])->count(),
            'absent' => collect($calendarData)->where('status', 'absent')->count(),
            'late' => $attendances->where('status', 'late')->count(),
            'leave' => collect($calendarData)->where('status', 'leave')->count(),
            'holiday' => collect($calendarData)->where('status', 'holiday')->count(),
            'total_hours' => $attendances->sum('worked_hours'),
            'overtime_hours' => $attendances->sum('overtime_hours'),
            'working_days' => collect($calendarData)->whereNotIn('status', ['weekend', 'holiday'])->where('isFuture', false)->count(),
        ];

        // First day of month (for calendar grid offset)
        $firstDayOfWeek = $startOfMonth->dayOfWeekIso; // 1 = Monday, 7 = Sunday

        return view('attendance::attendance.my-attendance', compact(
            'pageTitle', 
            'attendances', 
            'stats', 
            'month', 
            'year',
            'calendarData',
            'firstDayOfWeek',
            'shiftType',
            'shift',
            'staff'
        ));
    }

    // =====================
    // ATTENDANCE LOGS
    // =====================
    public function logs(Request $request)
    {
        $pageTitle = 'Attendance Logs';
        
        $admin = auth()->guard('admin')->user();
        $staff = Staff::where('admin_id', $admin->id)->first();
        
        // Stats
        $totalLogs = AttendanceLog::count();
        $myLogs = $staff ? AttendanceLog::whereHas('attendance', function($q) use ($staff) {
            $q->where('staff_id', $staff->id);
        })->count() : 0;
        
        $todayLogs = AttendanceLog::whereDate('created_at', today())->count();
        $checkIns = AttendanceLog::where('action', 'check_in')->count();
        $checkOuts = AttendanceLog::where('action', 'check_out')->count();
        
        // Staff list for filter
        $staffList = Staff::where('status', true)->orderBy('first_name')->get();
        
        return view('attendance::attendance.logs', compact(
            'pageTitle',
            'totalLogs',
            'myLogs',
            'todayLogs',
            'checkIns',
            'checkOuts',
            'staffList'
        ));
    }

    public function logsData(Request $request)
    {
        $admin = auth()->guard('admin')->user();
        $currentStaff = Staff::where('admin_id', $admin->id)->first();
        
        $query = AttendanceLog::with(['attendance.staff', 'performedByAdmin'])
            ->orderBy('created_at', 'desc');
        
        // Filter: My Logs
        if ($request->filter === 'my' && $currentStaff) {
            $query->whereHas('attendance', function($q) use ($currentStaff) {
                $q->where('staff_id', $currentStaff->id);
            });
        }
        
        // Filter: Check-ins only
        if ($request->filter === 'checkin') {
            $query->where('action', 'check_in');
        }
        
        // Filter: Check-outs only
        if ($request->filter === 'checkout') {
            $query->where('action', 'check_out');
        }
        
        // Filter: Today
        if ($request->filter === 'today') {
            $query->whereDate('created_at', today());
        }
        
        // Filter: Staff
        if ($request->staff_id) {
            $query->whereHas('attendance', function($q) use ($request) {
                $q->where('staff_id', $request->staff_id);
            });
        }
        
        // Filter: Date range
        if ($request->from_date) {
            $query->whereDate('created_at', '>=', $request->from_date);
        }
        if ($request->to_date) {
            $query->whereDate('created_at', '<=', $request->to_date);
        }
        
        // Search
        if ($request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('location', 'like', "%{$search}%")
                  ->orWhere('ip_address', 'like', "%{$search}%")
                  ->orWhere('notes', 'like', "%{$search}%")
                  ->orWhereHas('attendance.staff', function($sq) use ($search) {
                      $sq->where('first_name', 'like', "%{$search}%")
                         ->orWhere('last_name', 'like', "%{$search}%");
                  });
            });
        }
        
        // Pagination
        $perPage = $request->per_page ?? 15;
        $logs = $query->paginate($perPage);
        
        $data = $logs->map(function($log) {
            $staff = $log->attendance->staff ?? null;
            $staffName = $staff ? trim($staff->first_name . ' ' . $staff->last_name) : 'N/A';
            
            return [
                'id' => $log->id,
                'staff_name' => $staffName,
                'staff_avatar' => $staff ? strtoupper(substr($staff->first_name, 0, 1) . substr($staff->last_name, 0, 1)) : '??',
                'date' => $log->created_at->format('d M Y'),
                'time' => Carbon::parse($log->action_time)->format('h:i:s A'),
                'action' => $log->action,
                'action_label' => $log->action_label,
                'action_color' => $log->action_color,
                'location' => $log->location,
                'latitude' => $log->latitude,
                'longitude' => $log->longitude,
                'ip_address' => $log->ip_address,
                'device_info' => $log->device_info,
                'performed_by' => $log->performer_name,
                'notes' => $log->notes,
            ];
        });
        
        return response()->json([
            'data' => $data,
            'current_page' => $logs->currentPage(),
            'last_page' => $logs->lastPage(),
            'per_page' => $logs->perPage(),
            'total' => $logs->total(),
            'from' => $logs->firstItem(),
            'to' => $logs->lastItem(),
        ]);
    }
}
