<?php

namespace Modules\Attendance\Livewire;

use Livewire\Component;
use Modules\Attendance\Models\AdditionalHour;

class MyExtraHours extends Component
{
    public $extraHours = [];
    public $stats = [];

    public function mount()
    {
        $this->loadExtraHours();
    }

    public function loadExtraHours()
    {
        $staffId = auth()->guard('admin')->user()->staff->id ?? null;
        
        if (!$staffId) {
            $this->extraHours = [];
            $this->stats = ['total_hours' => 0, 'assigned' => 0, 'completed' => 0];
            return;
        }

        // Get this month's extra hours
        $startOfMonth = now()->startOfMonth();
        $endOfMonth = now()->endOfMonth();

        $this->extraHours = AdditionalHour::with('workType')
            ->where('staff_id', $staffId)
            ->where('work_date', '>=', $startOfMonth)
            ->where('work_date', '<=', $endOfMonth)
            ->orderBy('work_date', 'desc')
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'work_date' => $item->work_date->format('d M Y'),
                    'work_date_raw' => $item->work_date->format('Y-m-d'),
                    'is_future' => $item->work_date->isFuture(),
                    'is_today' => $item->work_date->isToday(),
                    'day_name' => $item->work_date->format('l'),
                    'type_name' => $item->workType->name ?? 'Standard',
                    'type_color' => $item->workType->color ?? '#3b82f6',
                    'hours' => $item->hours,
                    'time_in' => $item->time_in ? \Carbon\Carbon::parse($item->time_in)->format('h:i A') : null,
                    'time_out' => $item->time_out ? \Carbon\Carbon::parse($item->time_out)->format('h:i A') : null,
                    'reason' => $item->reason,
                    'status' => $item->status,
                    'acknowledged_at' => $item->acknowledged_at ? $item->acknowledged_at->format('d M, h:i A') : null,
                ];
            })
            ->toArray();

        // Calculate stats
        $allHours = AdditionalHour::where('staff_id', $staffId)
            ->where('work_date', '>=', $startOfMonth)
            ->where('work_date', '<=', $endOfMonth);
            
        $this->stats = [
            'total_hours' => (clone $allHours)->sum('hours'),
            'assigned' => (clone $allHours)->where('status', 'assigned')->count(),
            'completed' => (clone $allHours)->where('status', 'completed')->count(),
        ];
    }

    public function acknowledge($id)
    {
        $staffId = auth()->guard('admin')->user()->staff->id ?? null;
        
        $record = AdditionalHour::where('id', $id)
            ->where('staff_id', $staffId)
            ->where('status', 'assigned')
            ->first();

        if ($record) {
            $record->update([
                'status' => 'acknowledged',
                'acknowledged_at' => now(),
            ]);
            
            $this->loadExtraHours();
            $this->dispatch('extra-hour-acknowledged');
        }
    }

    public function render()
    {
        return view('attendance::livewire.my-extra-hours');
    }
}
