<?php

namespace Modules\Attendance\Models;

use Illuminate\Database\Eloquent\Model;

class ShiftType extends Model
{
    protected $table = 'att_shift_types';

    protected $fillable = [
        'name',
        'color',
        'start_time',
        'end_time',
        'lunch_start_time',
        'lunch_end_time',
        'days_off',
        'description',
        'location_restricted',
        'location_name',
        'location_latitude',
        'location_longitude',
        'allowed_radius',
        'status',
    ];

    protected $casts = [
        'status' => 'boolean',
        'location_restricted' => 'boolean',
        'days_off' => 'array',
        'location_latitude' => 'decimal:8',
        'location_longitude' => 'decimal:8',
        'allowed_radius' => 'integer',
    ];

    // Formatted work timing
    public function getWorkTimingAttribute(): string
    {
        return date('h:i A', strtotime($this->start_time)) . ' - ' . date('h:i A', strtotime($this->end_time));
    }

    // Formatted lunch timing
    public function getLunchTimingAttribute(): ?string
    {
        if (!$this->lunch_start_time || !$this->lunch_end_time) {
            return null;
        }
        return date('h:i A', strtotime($this->lunch_start_time)) . ' - ' . date('h:i A', strtotime($this->lunch_end_time));
    }

    // Calculate total work hours (excluding lunch)
    public function getTotalWorkHoursAttribute(): float
    {
        $start = strtotime($this->start_time);
        $end = strtotime($this->end_time);
        $totalMinutes = ($end - $start) / 60;

        // Subtract lunch break if exists
        if ($this->lunch_start_time && $this->lunch_end_time) {
            $lunchStart = strtotime($this->lunch_start_time);
            $lunchEnd = strtotime($this->lunch_end_time);
            $lunchMinutes = ($lunchEnd - $lunchStart) / 60;
            $totalMinutes -= $lunchMinutes;
        }

        return round($totalMinutes / 60, 2);
    }

    // Check if given coordinates are within allowed radius
    public function isWithinAllowedRadius($latitude, $longitude): bool
    {
        if (!$this->location_restricted || !$this->location_latitude || !$this->location_longitude) {
            return true; // No restriction
        }

        $distance = $this->calculateDistance(
            $this->location_latitude,
            $this->location_longitude,
            $latitude,
            $longitude
        );

        return $distance <= $this->allowed_radius;
    }

    // Calculate distance between two coordinates in meters (Haversine formula)
    public function calculateDistance($lat1, $lon1, $lat2, $lon2): float
    {
        $earthRadius = 6371000; // meters

        $lat1Rad = deg2rad($lat1);
        $lat2Rad = deg2rad($lat2);
        $deltaLat = deg2rad($lat2 - $lat1);
        $deltaLon = deg2rad($lon2 - $lon1);

        $a = sin($deltaLat / 2) * sin($deltaLat / 2) +
             cos($lat1Rad) * cos($lat2Rad) *
             sin($deltaLon / 2) * sin($deltaLon / 2);

        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));

        return $earthRadius * $c;
    }

    // Get distance from given coordinates
    public function getDistanceFrom($latitude, $longitude): ?float
    {
        if (!$this->location_latitude || !$this->location_longitude) {
            return null;
        }

        return round($this->calculateDistance(
            $this->location_latitude,
            $this->location_longitude,
            $latitude,
            $longitude
        ), 2);
    }
}
