<?php

namespace Modules\Attendance\Database\Seeders;

use Illuminate\Database\Seeder;
use Modules\Attendance\Models\LeaveType;
use Modules\Attendance\Models\RequestType;
use Modules\Attendance\Models\TimesheetType;
use Modules\Attendance\Models\OffType;
use Modules\Attendance\Models\ApprovalProcess;
use Modules\Attendance\Models\AttendanceSetting;
use Spatie\Permission\Models\Permission;

class AttendanceSeeder extends Seeder
{
    public function run(): void
    {
        // Register Permissions
        $this->registerPermissions();
        
        // Request Types (Leave, Late for work, Go home early, Go out, Go on business)
        $requestTypes = [
            ['name' => 'Leave', 'color' => '#3b82f6', 'requires_leave_type' => true],
            ['name' => 'Late for work', 'color' => '#f59e0b', 'requires_leave_type' => false],
            ['name' => 'Go home early', 'color' => '#8b5cf6', 'requires_leave_type' => false],
            ['name' => 'Go out', 'color' => '#10b981', 'requires_leave_type' => false],
            ['name' => 'Go on business', 'color' => '#ec4899', 'requires_leave_type' => false],
        ];

        foreach ($requestTypes as $type) {
            RequestType::firstOrCreate(['name' => $type['name']], $type);
        }

        // Leave Types
        $leaveTypes = [
            ['name' => 'Annual Leave', 'code' => 'AL', 'color' => '#3b82f6', 'default_days' => 12, 'is_paid' => true],
            ['name' => 'Sick Leave', 'code' => 'SL', 'color' => '#ef4444', 'default_days' => 10, 'is_paid' => true],
            ['name' => 'Casual Leave', 'code' => 'CL', 'color' => '#10b981', 'default_days' => 6, 'is_paid' => true],
            ['name' => 'Maternity Leave', 'code' => 'ML', 'color' => '#ec4899', 'default_days' => 90, 'is_paid' => true],
            ['name' => 'Unpaid Leave', 'code' => 'UL', 'color' => '#6b7280', 'default_days' => 0, 'is_paid' => false],
        ];

        foreach ($leaveTypes as $type) {
            LeaveType::firstOrCreate(['name' => $type['name']], $type);
        }

        // Timesheet Types
        $timesheetTypes = [
            ['name' => 'Regular', 'color' => '#3b82f6', 'multiplier' => 1.00],
            ['name' => 'Overtime', 'color' => '#f59e0b', 'multiplier' => 1.50],
        ];

        foreach ($timesheetTypes as $type) {
            TimesheetType::firstOrCreate(['name' => $type['name']], $type);
        }

        // Off Types (Holiday, Event break, Unexpected break)
        $offTypes = [
            ['name' => 'Holiday', 'color' => '#ef4444'],
            ['name' => 'Event break', 'color' => '#8b5cf6'],
            ['name' => 'Unexpected break', 'color' => '#f59e0b'],
        ];

        foreach ($offTypes as $type) {
            OffType::firstOrCreate(['name' => $type['name']], $type);
        }

        // Approval Processes
        $approvalProcesses = [
            ['name' => 'Leave approval', 'related_to' => 'Leave'],
        ];

        foreach ($approvalProcesses as $process) {
            ApprovalProcess::firstOrCreate(['related_to' => $process['related_to']], $process);
        }

        // Default Attendance Settings
        $defaultSettings = [
            // General
            ['key' => 'allow_self_checkin', 'value' => '1', 'group' => 'general'],
            ['key' => 'require_location', 'value' => '0', 'group' => 'general'],
            
            // Timesheet
            ['key' => 'work_hours_per_day', 'value' => '8', 'group' => 'timesheet'],
            ['key' => 'overtime_multiplier', 'value' => '1.5', 'group' => 'timesheet'],
            
            // Admin Attendance Tracking
            ['key' => 'track_is_admin', 'value' => '0', 'group' => 'general'],  // Don't track is_admin=1 by default
            ['key' => 'track_admin_only', 'value' => '0', 'group' => 'general'], // Don't auto-create staff by default
            ['key' => 'exempt_roles', 'value' => '[]', 'group' => 'general'],    // No exempt roles by default
            
            // Maps
            ['key' => 'google_map_api_key', 'value' => '', 'group' => 'maps'],
        ];

        foreach ($defaultSettings as $setting) {
            AttendanceSetting::firstOrCreate(
                ['key' => $setting['key']], 
                $setting
            );
        }
    }

    /**
     * Register all attendance permissions
     */
    protected function registerPermissions(): void
    {
        $permissions = config('attendance.permissions', []);
        
        foreach ($permissions as $name => $description) {
            Permission::firstOrCreate(
                ['name' => $name, 'guard_name' => 'admin'],
                ['name' => $name, 'guard_name' => 'admin']
            );
        }
    }
}
