<?php

namespace Modules\Attendance\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use App\Traits\DataTable;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Attendance\Models\LeaveApplication;
use Modules\Attendance\Models\LeaveType;
use Modules\Attendance\Models\RequestType;
use Modules\Attendance\Models\StaffLeaveSetting;
use App\Models\Admin\Staff;
use Carbon\Carbon;

class LeaveApplicationController extends AdminController
{
    use DataTable;

    protected $model = LeaveApplication::class;
    protected $viewPrefix = 'attendance::leave';
    protected $routePrefix = 'admin.attendance.leave';
    protected $with = ['staff', 'requestType', 'leaveType', 'approver'];

    protected $searchable = ['subject', 'reason'];
    protected $sortable = ['id', 'from_date', 'to_date', 'created_at', 'status'];
    protected $filterable = ['status', 'request_type_id', 'leave_type_id', 'staff_id'];

    protected function mapRow($item)
    {
        return [
            'id' => $item->id,
            'subject' => $item->subject,
            'staff_name' => $item->staff ? trim($item->staff->first_name . ' ' . $item->staff->last_name) : '-',
            'start_date' => $item->from_date->format('d-m-Y'),
            'end_date' => $item->to_date->format('d-m-Y'),
            'request_type' => $item->requestType->name ?? '-',
            'leave_type' => $item->leaveType->name ?? '-',
            'number_of_days' => $item->number_of_days,
            'status' => $item->status,
            'created_at' => $item->created_at->format('d-m-Y'),
            '_edit_url' => route('admin.attendance.leave.edit', $item->id),
            '_delete_url' => route('admin.attendance.leave.destroy', $item->id),
        ];
    }

    /**
     * Admin view - All leave applications with approval
     */
    public function index(Request $request)
    {
        $pageTitle = 'Leave Applications';
        $currentStatus = $request->get('status', 'all');

        $stats = [
            'new' => LeaveApplication::where('status', 'new')->count(),
            'approved' => LeaveApplication::where('status', 'approved')->count(),
            'rejected' => LeaveApplication::where('status', 'rejected')->count(),
            'total' => LeaveApplication::count(),
        ];

        $query = LeaveApplication::with(['staff', 'leaveType', 'approver'])
            ->orderBy('created_at', 'desc');

        if ($currentStatus !== 'all') {
            $query->where('status', $currentStatus);
        }

        $applications = $query->paginate(20);

        return view('attendance::leave.index', compact('pageTitle', 'stats', 'currentStatus', 'applications'));
    }

    /**
     * Staff view - Only their own leaves
     */
    public function myLeaves(Request $request)
    {
        $pageTitle = 'My Leaves';
        $currentStatus = $request->get('status', 'all');
        
        $staffId = auth()->guard('admin')->user()->staff->id ?? null;
        
        if (!$staffId) {
            return redirect()->route('admin.attendance.index')
                ->with('error', 'Staff profile not found.');
        }

        // Check if staff has active shift
        if (!\Modules\Attendance\Services\AttendanceHelper::hasActiveShift($staffId)) {
            return view('attendance::leave.my-leaves', [
                'pageTitle' => $pageTitle,
                'noShift' => true,
                'stats' => ['new' => 0, 'approved' => 0, 'rejected' => 0, 'total' => 0],
                'currentStatus' => 'all',
                'leaves' => collect(),
                'leaveBalances' => collect(),
            ]);
        }

        // Get leave balance summary for this month
        $leaveBalances = $this->getMonthlyBalanceSummary($staffId);

        // Stats for this staff only
        $stats = [
            'new' => LeaveApplication::where('staff_id', $staffId)->where('status', 'new')->count(),
            'approved' => LeaveApplication::where('staff_id', $staffId)->where('status', 'approved')->count(),
            'rejected' => LeaveApplication::where('staff_id', $staffId)->where('status', 'rejected')->count(),
            'total' => LeaveApplication::where('staff_id', $staffId)->count(),
        ];

        $query = LeaveApplication::where('staff_id', $staffId)
            ->with(['leaveType', 'approver'])
            ->orderBy('created_at', 'desc');

        if ($currentStatus !== 'all') {
            $query->where('status', $currentStatus);
        }

        $leaves = $query->paginate(20);

        return view('attendance::leave.my-leaves', compact('pageTitle', 'stats', 'currentStatus', 'leaves', 'leaveBalances'));
    }

    /**
     * Get monthly balance summary for a staff
     */
    protected function getMonthlyBalanceSummary($staffId)
    {
        $leaveTypes = LeaveType::where('status', true)->get();
        $startOfMonth = now()->startOfMonth();
        $endOfMonth = now()->endOfMonth();
        
        $balances = [];
        
        foreach ($leaveTypes as $leaveType) {
            // Get limit (staff-specific or default from leave type)
            $staffSetting = StaffLeaveSetting::where('staff_id', $staffId)
                ->where('leave_type_id', $leaveType->id)
                ->first();
            
            $monthlyLimit = $staffSetting->days_per_month ?? $leaveType->days_per_month ?? 0;
            
            // Count approved leaves this month for this type
            $usedThisMonth = LeaveApplication::where('staff_id', $staffId)
                ->where('leave_type_id', $leaveType->id)
                ->where('status', 'approved')
                ->where('from_date', '>=', $startOfMonth)
                ->where('from_date', '<=', $endOfMonth)
                ->sum('number_of_days');
            
            $balances[] = (object)[
                'leave_type_id' => $leaveType->id,
                'leaveType' => $leaveType,
                'monthly_limit' => $monthlyLimit,
                'used_this_month' => $usedThisMonth,
                'remaining' => max(0, $monthlyLimit - $usedThisMonth),
            ];
        }
        
        return collect($balances);
    }

    /**
     * Apply for leave form
     */
    public function create()
    {
        $pageTitle = 'Apply for Leave';
        
        // Check if current staff has active shift
        $staffId = auth()->guard('admin')->user()->staff->id ?? null;
        if ($staffId && !\Modules\Attendance\Services\AttendanceHelper::hasActiveShift($staffId)) {
            return redirect()->route('admin.attendance.my_leaves')
                ->with('error', 'You do not have an active shift assigned. Please contact your administrator.');
        }
        
        $requestTypes = RequestType::where('status', true)->orderBy('name')->get();
        $leaveTypes = LeaveType::where('status', true)->orderBy('name')->get();
        
        // Only get staff WITH active shifts for followers/handovers
        $staffList = \Modules\Attendance\Services\AttendanceHelper::getStaffWithShifts();

        return view('attendance::leave.create', compact('pageTitle', 'requestTypes', 'leaveTypes', 'staffList'));
    }

    /**
     * Store leave application
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'request_type_id' => 'nullable',
            'leave_type_id' => 'required',
            'number_of_days' => 'nullable|numeric|min:0.5',
            'from_date' => 'required|date',
            'to_date' => 'required|date|after_or_equal:from_date',
            'reason' => 'nullable|string',
            'followers' => 'nullable|array',
            'handovers' => 'nullable|array',
            'attachment' => 'nullable|file|mimes:pdf,doc,docx,jpg,jpeg,png|max:5120',
        ]);

        // Get leave type
        $leaveType = LeaveType::find($validated['leave_type_id']);
        if (!$leaveType) {
            return redirect()->back()->withInput()->with('error', 'Invalid leave type selected.');
        }

        // Get current staff
        $admin = auth()->guard('admin')->user();
        $staff = $admin->staff ?? null;
        if (!$staff) {
            return redirect()->back()->withInput()->with('error', 'Staff profile not found.');
        }

        $staffName = trim($staff->first_name . ' ' . $staff->last_name);

        // Calculate days
        $fromDate = Carbon::parse($validated['from_date']);
        $toDate = Carbon::parse($validated['to_date']);
        $numberOfDays = $validated['number_of_days'] ?? ($fromDate->diffInDays($toDate) + 1);

        // Auto-generate subject
        $subject = $leaveType->name . ' - ' . $fromDate->format('d M Y');

        // Get or create request type
        $requestTypeId = $validated['request_type_id'] ?? null;
        if (!$requestTypeId) {
            $requestType = RequestType::where('status', true)->first();
            if (!$requestType) {
                $requestType = RequestType::create([
                    'name' => 'Leave',
                    'color' => '#3b82f6',
                    'requires_leave_type' => true,
                    'status' => true,
                ]);
            }
            $requestTypeId = $requestType->id;
        }

        // Handle attachment
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('leave-attachments', 'public');
        }

        // Create leave application (no blocking - just create it)
        $leave = LeaveApplication::create([
            'staff_id' => $staff->id,
            'subject' => $subject,
            'request_type_id' => $requestTypeId,
            'leave_type_id' => $leaveType->id,
            'number_of_days' => $numberOfDays,
            'from_date' => $validated['from_date'],
            'to_date' => $validated['to_date'],
            'reason' => $validated['reason'] ?? null,
            'attachment' => $attachmentPath,
            'status' => 'new',
        ]);

        // Sync followers and handovers
        $followerIds = array_filter($validated['followers'] ?? []);
        $handoverIds = array_filter($validated['handovers'] ?? []);
        
        if (!empty($followerIds)) {
            $leave->followers()->sync($followerIds);
        }
        if (!empty($handoverIds)) {
            $leave->handovers()->sync($handoverIds);
        }

        // Send notifications
        $dateRange = $fromDate->format('d M') . ($fromDate->ne($toDate) ? ' - ' . $toDate->format('d M Y') : ' ' . $fromDate->format('Y'));

        foreach ($followerIds as $followerStaffId) {
            $followerStaff = Staff::find($followerStaffId);
            if ($followerStaff && $followerStaff->admin_id) {
                $this->sendNotification(
                    $followerStaff->admin_id, 'admin', $admin->id, 'admin',
                    "📋 Leave Application: {$staffName}",
                    "{$staffName} has applied for {$leaveType->name} ({$dateRange}). You are added as a follower.",
                    'info',
                    route('admin.attendance.leave.edit', $leave->id)
                );
            }
        }

        foreach ($handoverIds as $handoverStaffId) {
            $handoverStaff = Staff::find($handoverStaffId);
            if ($handoverStaff && $handoverStaff->admin_id) {
                $this->sendNotification(
                    $handoverStaff->admin_id, 'admin', $admin->id, 'admin',
                    "🔄 Work Handover: {$staffName}",
                    "{$staffName} is on {$leaveType->name} ({$dateRange}). Please handle their work.",
                    'warning',
                    route('admin.attendance.my')
                );
            }
        }

        return redirect()->route('admin.attendance.my_leaves')
            ->with('success', 'Leave application submitted successfully.');
    }

    /**
     * Edit leave
     */
    public function edit($id)
    {
        $leave = LeaveApplication::with(['followers', 'handovers', 'staff', 'leaveType'])->findOrFail($id);
        $pageTitle = 'View Leave Application';
        $requestTypes = RequestType::where('status', true)->orderBy('name')->get();
        $leaveTypes = LeaveType::where('status', true)->orderBy('name')->get();
        $staffList = Staff::where('status', true)->orderBy('first_name')->get();

        return view('attendance::leave.edit', compact('leave', 'pageTitle', 'requestTypes', 'leaveTypes', 'staffList'));
    }

    /**
     * Update leave
     */
    public function update(Request $request, $id)
    {
        $leave = LeaveApplication::findOrFail($id);

        $validated = $request->validate([
            'leave_type_id' => 'required',
            'from_date' => 'required|date',
            'to_date' => 'required|date|after_or_equal:from_date',
            'reason' => 'nullable|string',
            'status' => 'nullable|in:new,approved,rejected',
        ]);

        $fromDate = Carbon::parse($validated['from_date']);
        $toDate = Carbon::parse($validated['to_date']);
        $validated['number_of_days'] = $fromDate->diffInDays($toDate) + 1;

        $leave->update($validated);

        return redirect()->route('admin.attendance.leave.index')
            ->with('success', 'Leave application updated.');
    }

    /**
     * Delete leave
     */
    public function destroy($id)
    {
        $leave = LeaveApplication::findOrFail($id);
        $leave->delete();
        return response()->json(['success' => true, 'message' => 'Deleted successfully']);
    }

    /**
     * Approve leave application
     */
    public function approve(Request $request, $id)
    {
        $leave = LeaveApplication::with('staff', 'leaveType')->findOrFail($id);
        
        if ($leave->status !== 'new') {
            return redirect()->back()->with('error', 'This leave has already been processed.');
        }

        // Update leave status
        $leave->update([
            'status' => 'approved',
            'approved_by' => auth()->guard('admin')->user()->staff->id ?? auth()->guard('admin')->user()->id,
            'approved_at' => now(),
        ]);

        // Send notification to staff
        if ($leave->staff && $leave->staff->admin_id) {
            $this->sendNotification(
                $leave->staff->admin_id, 'admin',
                auth()->guard('admin')->id(), 'admin',
                "✅ Leave Approved",
                "Your {$leave->leaveType->name} application ({$leave->from_date->format('d M')} - {$leave->to_date->format('d M')}) has been approved.",
                'success',
                route('admin.attendance.my_leaves')
            );
        }

        return redirect()->back()->with('success', 'Leave approved successfully.');
    }

    /**
     * Reject leave application
     */
    public function reject(Request $request, $id)
    {
        $leave = LeaveApplication::with('staff', 'leaveType')->findOrFail($id);

        $leave->update([
            'status' => 'rejected',
            'reject_reason' => $request->reason,
            'approved_by' => auth()->guard('admin')->user()->staff->id ?? auth()->guard('admin')->user()->id,
            'approved_at' => now(),
        ]);

        // Send notification to staff
        if ($leave->staff && $leave->staff->admin_id) {
            $rejectReason = $request->reason ? " Reason: {$request->reason}" : '';
            $this->sendNotification(
                $leave->staff->admin_id, 'admin',
                auth()->guard('admin')->id(), 'admin',
                "❌ Leave Rejected",
                "Your {$leave->leaveType->name} application ({$leave->from_date->format('d M')} - {$leave->to_date->format('d M')}) has been rejected.{$rejectReason}",
                'error',
                route('admin.attendance.my_leaves')
            );
        }

        return redirect()->back()->with('success', 'Leave rejected.');
    }

    /**
     * Get leave balance for AJAX (monthly)
     */
    public function getLeaveBalance(Request $request)
    {
        $staffId = auth()->guard('admin')->user()->staff->id ?? null;
        $leaveTypeId = $request->leave_type_id;
        
        if (!$staffId || !$leaveTypeId) {
            return response()->json(['remaining' => 0, 'limit' => 0, 'used' => 0]);
        }

        $leaveType = LeaveType::find($leaveTypeId);
        if (!$leaveType) {
            return response()->json(['remaining' => 0, 'limit' => 0, 'used' => 0]);
        }

        // Get staff-specific limit or default from leave type
        $staffSetting = StaffLeaveSetting::where('staff_id', $staffId)
            ->where('leave_type_id', $leaveTypeId)
            ->first();
        
        $monthlyLimit = $staffSetting->days_per_month ?? $leaveType->days_per_month ?? 0;

        // Count approved leaves this month
        $startOfMonth = now()->startOfMonth();
        $endOfMonth = now()->endOfMonth();
        
        $usedThisMonth = LeaveApplication::where('staff_id', $staffId)
            ->where('leave_type_id', $leaveTypeId)
            ->where('status', 'approved')
            ->where('from_date', '>=', $startOfMonth)
            ->where('from_date', '<=', $endOfMonth)
            ->sum('number_of_days');

        $remaining = max(0, $monthlyLimit - $usedThisMonth);

        return response()->json([
            'remaining' => $remaining,
            'total' => $monthlyLimit,  // For backward compatibility
            'limit' => $monthlyLimit,
            'used' => $usedThisMonth,
            'warning' => $remaining <= 0 && $monthlyLimit > 0,
        ]);
    }

    /**
     * Get handover work for current staff
     */
    public function getHandoverWork()
    {
        $staffId = auth()->guard('admin')->user()->staff->id ?? null;
        
        if (!$staffId) {
            return response()->json(['handovers' => []]);
        }

        $handoverWork = LeaveApplication::whereHas('handovers', function($q) use ($staffId) {
                $q->where('staff_id', $staffId);
            })
            ->where('status', 'approved')
            ->where('to_date', '>=', now())
            ->with(['staff', 'leaveType'])
            ->orderBy('from_date')
            ->get()
            ->map(function($leave) {
                return [
                    'id' => $leave->id,
                    'staff_name' => $leave->staff ? trim($leave->staff->first_name . ' ' . $leave->staff->last_name) : '-',
                    'leave_type' => $leave->leaveType->name ?? 'Leave',
                    'from_date' => $leave->from_date->format('d M Y'),
                    'to_date' => $leave->to_date->format('d M Y'),
                    'is_active' => $leave->from_date->lte(now()) && $leave->to_date->gte(now()),
                    'reason' => $leave->reason,
                ];
            });

        return response()->json(['handovers' => $handoverWork]);
    }

    /**
     * Send notification
     */
    protected function sendNotification(
        int $userId, string $userType, ?int $fromUserId, ?string $fromUserType,
        string $title, string $message, string $type = 'info', ?string $url = null
    ): void {
        try {
            DB::table('notifications')->insert([
                'user_id' => $userId,
                'user_type' => $userType,
                'from_user_id' => $fromUserId,
                'from_user_type' => $fromUserType,
                'title' => $title,
                'message' => $message,
                'type' => $type,
                'url' => $url,
                'is_read' => 0,
                'created_at' => now(),
            ]);
        } catch (\Exception $e) {
            \Log::error('Notification failed: ' . $e->getMessage());
        }
    }
}
