<?php

namespace Modules\Attendance\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use App\Traits\DataTable;
use Illuminate\Http\Request;
use Modules\Attendance\Models\DayOff;
use Modules\Attendance\Models\OffType;
use Modules\Attendance\Models\ApprovalProcess;
use Modules\Attendance\Models\AnnualLeaveSetting;
use Modules\Attendance\Models\AttendanceSetting;
use Modules\Attendance\Models\LeaveType;
use Modules\Attendance\Models\TimesheetType;
use App\Models\Admin\Staff;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;

class SettingsController extends AdminController
{
    use DataTable;

    // =====================
    // MAIN SETTINGS PAGE (Tabs)
    // =====================
    public function index()
    {
        $pageTitle = 'Attendance Settings';
        
        // Staff data
        $staffList = Staff::where('status', true)->orderBy('first_name')->get();
        
        // Leave Types
        $leaveTypes = LeaveType::where('status', true)->orderBy('name')->get();
        
        // Annual Leave Settings
        $annualSettings = AnnualLeaveSetting::with(['staff', 'leaveType'])->get()->groupBy('staff_id');
        
        // Holiday data
        $dayOffs = DayOff::with(['offType', 'addedBy'])->orderBy('off_date', 'desc')->get();
        $offTypes = OffType::where('status', true)->orderBy('name')->get();
        
        // Departments from DB
        $departments = collect();
        if (DB::getSchemaBuilder()->hasTable('departments')) {
            $departments = DB::table('departments')->where('is_active', true)->orderBy('name')->get();
        }
        
        // Roles
        $roles = Role::where('guard_name', 'admin')->orderBy('name')->get();
        
        // Approval Process data
        $approvalProcesses = ApprovalProcess::all();
        
        // Timesheet Types
        $timesheetTypes = TimesheetType::where('status', true)->orderBy('name')->get();
        
        // General Settings
        $settingsData = AttendanceSetting::all()->pluck('value', 'key')->toArray();
        $settings = $settingsData;

        return view('attendance::settings.index', compact(
            'pageTitle',
            'staffList', 
            'leaveTypes', 
            'annualSettings',
            'dayOffs', 
            'offTypes', 
            'departments', 
            'roles',
            'approvalProcesses',
            'timesheetTypes',
            'settings'
        ));
    }

    // =====================
    // ANNUAL LEAVE
    // =====================
    public function annualLeaveUpdate(Request $request)
    {
        $data = $request->input('settings', []);

        // Get first leave type or create default
        $leaveType = LeaveType::first();
        if (!$leaveType) {
            $leaveType = LeaveType::create([
                'name' => 'Annual leave',
                'color' => '#3b82f6',
                'default_days' => 12,
                'is_paid' => true,
            ]);
        }

        foreach ($data as $staffId => $days) {
            AnnualLeaveSetting::updateOrCreate(
                ['staff_id' => $staffId, 'leave_type_id' => $leaveType->id],
                ['annual_days' => floatval($days)]
            );
        }

        return redirect()->back()->with('success', 'Annual leave settings updated');
    }

    // =====================
    // DAY OFF / HOLIDAYS
    // =====================
    public function dayOffStore(Request $request)
    {
        $validated = $request->validate([
            'off_type_id' => 'required',
            'reason' => 'required|string|max:255',
            'off_date' => 'nullable|date',
            'department_id' => 'nullable',
            'role_name' => 'nullable|string',
            'repeat_yearly' => 'nullable',
        ]);

        // Handle hardcoded off_type_id
        $offTypeId = $validated['off_type_id'];
        $offType = OffType::find($offTypeId);
        if (!$offType) {
            // Create off type if it doesn't exist
            $offTypeNames = ['1' => 'Holiday', '2' => 'Event break', '3' => 'Unexpected break'];
            $offType = OffType::create([
                'name' => $offTypeNames[$offTypeId] ?? 'Holiday',
                'color' => '#3b82f6',
            ]);
            $offTypeId = $offType->id;
        }

        $validated['off_type_id'] = $offTypeId;
        $validated['off_date'] = $validated['off_date'] ?? now()->format('Y-m-d');
        $validated['repeat_yearly'] = $request->boolean('repeat_yearly');
        $validated['added_by'] = auth()->guard('admin')->user()->staff->id ?? null;

        DayOff::create($validated);

        return redirect()->route('admin.attendance.settings.index')
            ->with('success', 'Day off added successfully');
    }

    public function dayOffJson($id)
    {
        $dayOff = DayOff::findOrFail($id);
        return response()->json($dayOff);
    }

    public function dayOffUpdate(Request $request, $id)
    {
        $dayOff = DayOff::findOrFail($id);

        $validated = $request->validate([
            'off_type_id' => 'required',
            'reason' => 'required|string|max:255',
            'off_date' => 'nullable|date',
            'department_id' => 'nullable',
            'role_name' => 'nullable|string',
            'repeat_yearly' => 'nullable',
        ]);

        $validated['repeat_yearly'] = $request->boolean('repeat_yearly');
        $dayOff->update($validated);

        return redirect()->route('admin.attendance.settings.index')
            ->with('success', 'Day off updated successfully');
    }

    public function dayOffDestroy($id)
    {
        DayOff::findOrFail($id)->delete();
        return response()->json(['success' => true, 'message' => 'Day off deleted successfully']);
    }

    // =====================
    // APPROVAL PROCESS
    // =====================
    public function approvalUpdateAll(Request $request)
    {
        $approvers = $request->input('approvers', []);
        
        foreach ($approvers as $processId => $staffIds) {
            $process = ApprovalProcess::find($processId);
            if ($process) {
                $staffIds = json_decode($staffIds, true) ?? [];
                $process->update(['approvers' => $staffIds]);
            }
        }

        return redirect()->back()->with('success', 'Approvers updated successfully');
    }

    public function approvalEdit($id)
    {
        $process = ApprovalProcess::findOrFail($id);
        $pageTitle = 'Edit Approval Process';
        $staffList = Staff::where('status', true)->orderBy('first_name')->get();

        return view('attendance::settings.approval.edit', compact('process', 'pageTitle', 'staffList'));
    }

    public function approvalUpdate(Request $request, $id)
    {
        $process = ApprovalProcess::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'approvers' => 'nullable|array',
        ]);

        $process->update($validated);

        return redirect()->route('admin.attendance.settings.index')
            ->with('success', 'Approval process updated');
    }

    // =====================
    // TIMESHEET SETTINGS
    // =====================
    public function timesheetUpdate(Request $request)
    {
        $data = $request->input('settings', []);

        foreach ($data as $key => $value) {
            AttendanceSetting::set($key, $value, 'timesheet');
        }

        return redirect()->back()->with('success', 'Timesheet settings updated');
    }

    // =====================
    // GENERAL SETTINGS
    // =====================
    public function generalUpdate(Request $request)
    {
        $data = $request->input('settings', []);

        // Keys that are checkboxes - set to '0' if not present
        $checkboxKeys = ['track_super_admin', 'track_admin_only', 'allow_self_checkin', 'require_location'];
        
        foreach ($checkboxKeys as $key) {
            if (!isset($data[$key])) {
                $data[$key] = '0';
            }
        }

        foreach ($data as $key => $value) {
            // Handle array values (like exempt_roles)
            if (is_array($value)) {
                $value = json_encode(array_filter($value)); // Remove empty values
            }
            AttendanceSetting::set($key, $value ?? '');
        }

        // Clear attendance settings cache
        $cacheKeys = [
            'att_setting_track_super_admin',
            'att_setting_track_admin_only',
            'att_setting_exempt_roles',
            'att_setting_allow_self_checkin',
            'att_setting_require_location',
            'att_setting_google_map_api_key',
            'att_setting_work_hours_per_day',
            'att_setting_overtime_multiplier',
        ];
        
        foreach ($cacheKeys as $cacheKey) {
            \Illuminate\Support\Facades\Cache::forget($cacheKey);
        }

        return redirect()->back()->with('success', 'Settings updated successfully');
    }

    // =====================
    // LEAVE TYPES
    // =====================
    public function leaveTypeStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'code' => 'required|string|max:5',
            'color' => 'nullable|string|max:20',
            'days_per_month' => 'nullable|numeric|min:0',
            'days_per_year' => 'nullable|numeric|min:0',
            'is_paid' => 'nullable',
        ]);

        LeaveType::create([
            'name' => $validated['name'],
            'code' => strtoupper($validated['code']),
            'color' => $validated['color'] ?? '#3b82f6',
            'days_per_month' => $validated['days_per_month'] ?? 0,
            'days_per_year' => $validated['days_per_year'] ?? 0,
            'is_paid' => $request->boolean('is_paid'),
            'status' => true,
        ]);

        return redirect()->back()->with('success', 'Leave type created successfully');
    }

    public function leaveTypeUpdate(Request $request, $id)
    {
        $leaveType = LeaveType::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'code' => 'required|string|max:5',
            'color' => 'nullable|string|max:20',
            'days_per_month' => 'nullable|numeric|min:0',
            'days_per_year' => 'nullable|numeric|min:0',
            'is_paid' => 'nullable',
        ]);

        $leaveType->update([
            'name' => $validated['name'],
            'code' => strtoupper($validated['code']),
            'color' => $validated['color'] ?? $leaveType->color,
            'days_per_month' => $validated['days_per_month'] ?? 0,
            'days_per_year' => $validated['days_per_year'] ?? 0,
            'is_paid' => $request->boolean('is_paid'),
        ]);

        return redirect()->back()->with('success', 'Leave type updated successfully');
    }

    /**
     * Update leave type limit inline (AJAX)
     */
    public function leaveTypeUpdateLimit(Request $request, $id)
    {
        $leaveType = LeaveType::findOrFail($id);
        
        $field = $request->input('field');
        $value = (int) $request->input('value', 0);
        
        if (in_array($field, ['days_per_month', 'days_per_year'])) {
            $leaveType->update([$field => $value]);
            return response()->json(['success' => true, 'message' => 'Updated']);
        }
        
        return response()->json(['success' => false, 'message' => 'Invalid field'], 400);
    }

    public function leaveTypeDestroy($id)
    {
        LeaveType::findOrFail($id)->delete();
        return response()->json(['success' => true, 'message' => 'Leave type deleted']);
    }

    // =====================
    // HOLIDAYS
    // =====================
    public function holidayStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'date' => 'required|date',
            'type' => 'nullable|string|max:50',
        ]);

        // Get or create holiday off type
        $offType = OffType::firstOrCreate(
            ['name' => 'Holiday'],
            ['color' => '#f59e0b', 'status' => true]
        );

        DayOff::create([
            'reason' => $validated['name'],
            'off_date' => $validated['date'],
            'off_type_id' => $offType->id,
            'repeat_yearly' => false,
            'added_by' => auth()->guard('admin')->user()->staff->id ?? null,
        ]);

        return redirect()->back()->with('success', 'Holiday added successfully');
    }

    public function holidayUpdate(Request $request, $id)
    {
        $holiday = DayOff::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'date' => 'required|date',
            'type' => 'nullable|string|max:50',
        ]);

        $holiday->update([
            'reason' => $validated['name'],
            'off_date' => $validated['date'],
        ]);

        return redirect()->back()->with('success', 'Holiday updated successfully');
    }

    public function holidayDestroy($id)
    {
        DayOff::findOrFail($id)->delete();
        return response()->json(['success' => true, 'message' => 'Holiday deleted']);
    }

    // =====================
    // DAY OFF (Legacy separate pages)
    // =====================
    public function dayOffIndex()
    {
        $pageTitle = 'Day Off / Holidays';
        $dayOffs = DayOff::with(['offType', 'addedBy'])->orderBy('off_date', 'desc')->get();
        $offTypes = OffType::where('status', true)->orderBy('name')->get();
        
        return view('attendance::settings.day-off.index', compact('pageTitle', 'dayOffs', 'offTypes'));
    }

    public function dayOffCreate()
    {
        $pageTitle = 'Add Day Off';
        $offTypes = OffType::where('status', true)->orderBy('name')->get();
        $departments = collect();
        if (DB::getSchemaBuilder()->hasTable('departments')) {
            $departments = DB::table('departments')->where('is_active', true)->orderBy('name')->get();
        }
        $roles = Role::where('guard_name', 'admin')->orderBy('name')->get();
        
        return view('attendance::settings.day-off.create', compact('pageTitle', 'offTypes', 'departments', 'roles'));
    }

    public function dayOffEdit($id)
    {
        $dayOff = DayOff::findOrFail($id);
        $pageTitle = 'Edit Day Off';
        $offTypes = OffType::where('status', true)->orderBy('name')->get();
        $departments = collect();
        if (DB::getSchemaBuilder()->hasTable('departments')) {
            $departments = DB::table('departments')->where('is_active', true)->orderBy('name')->get();
        }
        $roles = Role::where('guard_name', 'admin')->orderBy('name')->get();
        
        return view('attendance::settings.day-off.edit', compact('pageTitle', 'dayOff', 'offTypes', 'departments', 'roles'));
    }

    public function dayOffData(Request $request)
    {
        $query = DayOff::with(['offType', 'addedBy'])->orderBy('off_date', 'desc');
        
        // Simple datatable response
        $items = $query->get()->map(function($item) {
            return [
                'id' => $item->id,
                'date' => $item->off_date ? $item->off_date->format('d M Y') : '-',
                'reason' => $item->reason,
                'type' => $item->offType?->name ?? 'Holiday',
                'department' => $item->department?->name ?? 'All',
                'role' => $item->role_name ?? 'All',
                '_edit_url' => route('admin.attendance.settings.day_off.edit', $item->id),
                '_delete_url' => route('admin.attendance.settings.day_off.destroy', $item->id),
            ];
        });

        return response()->json([
            'data' => $items,
            'recordsTotal' => $items->count(),
            'recordsFiltered' => $items->count(),
        ]);
    }

    // =====================
    // WORK TYPES (Timesheet Types)
    // =====================
    public function workTypeStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'color' => 'nullable|string|max:20',
            'multiplier' => 'nullable|numeric|min:0.5|max:5',
        ]);

        TimesheetType::create([
            'name' => $validated['name'],
            'color' => $validated['color'] ?? '#3b82f6',
            'multiplier' => $validated['multiplier'] ?? 1,
            'status' => true,
        ]);

        return redirect()->route('admin.attendance.settings.index')
            ->with('success', 'Work type created successfully');
    }

    public function workTypeUpdate(Request $request, $id)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'color' => 'nullable|string|max:20',
            'multiplier' => 'nullable|numeric|min:0.5|max:5',
        ]);

        $workType = TimesheetType::findOrFail($id);
        $workType->update([
            'name' => $validated['name'],
            'color' => $validated['color'] ?? '#3b82f6',
            'multiplier' => $validated['multiplier'] ?? 1,
        ]);

        return redirect()->route('admin.attendance.settings.index')
            ->with('success', 'Work type updated successfully');
    }

    public function workTypeDestroy($id)
    {
        $workType = TimesheetType::findOrFail($id);
        $workType->delete();

        return redirect()->route('admin.attendance.settings.index')
            ->with('success', 'Work type deleted successfully');
    }
}
