<?php

namespace Modules\Attendance\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use App\Traits\DataTable;
use Illuminate\Http\Request;
use Modules\Attendance\Models\Shift;
use Modules\Attendance\Models\ShiftType;
use Modules\Attendance\Services\AttendanceService;
use App\Models\Admin\Staff;
use Spatie\Permission\Models\Role;

class ShiftController extends AdminController
{
    use DataTable;

    protected $model = Shift::class;
    protected $viewPrefix = 'attendance::shifts';
    protected $routePrefix = 'admin.attendance.shifts';
    protected $with = ['shiftType', 'staff'];

    protected $searchable = [];
    protected $sortable = ['id', 'from_date', 'to_date', 'created_at'];
    protected $filterable = ['status', 'shift_type_id', 'staff_id'];

    protected function mapRow($item)
    {
        return [
            'id' => $item->id,
            'from_date' => $item->from_date->format('d-m-Y'),
            'to_date' => $item->to_date ? $item->to_date->format('d-m-Y') : '-',
            'department' => $item->department ?: '-',
            'role_name' => $item->role_name ?: '-',
            'staff_name' => $item->staff ? trim($item->staff->first_name . ' ' . $item->staff->last_name) : '-',
            'shift_type_name' => $item->shiftType->name ?? '-',
            'shift_type_color' => $item->shiftType->color ?? '#3b82f6',
            'created_at' => $item->created_at->format('d-m-Y'),
            'status' => $item->status,
            '_edit_url' => route('admin.attendance.shifts.edit', $item->id),
            '_delete_url' => route('admin.attendance.shifts.destroy', $item->id),
        ];
    }

    public function index()
    {
        $pageTitle = 'Shift Assignments';
        $shiftTypes = ShiftType::where('status', true)->orderBy('name')->get();
        return view('attendance::shifts.index', compact('pageTitle', 'shiftTypes'));
    }

    public function create()
    {
        $pageTitle = 'Assign Shift';

        // Only get staff WITHOUT active shifts
        $staffWithoutShifts = \Modules\Attendance\Services\AttendanceHelper::getStaffWithoutShifts();

        $departments = $staffWithoutShifts
            ->whereNotNull('department')
            ->where('department', '!=', '')
            ->pluck('department')
            ->unique()
            ->sort()
            ->values()
            ->toArray();

        $roles = Role::where('guard_name', 'admin')->orderBy('name')->pluck('name')->toArray();
        $staffList = $staffWithoutShifts;
        $shiftTypes = ShiftType::where('status', true)->orderBy('name')->get();

        return view('attendance::shifts.create', compact(
            'pageTitle', 'departments', 'roles', 'staffList', 'shiftTypes'
        ));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'shift_type_id' => 'required|exists:att_shift_types,id',
            'from_date' => 'required|date',
            'to_date' => 'nullable|date|after_or_equal:from_date',
            'assign_type' => 'required|in:filter,specific',
            'departments' => 'nullable|array',
            'roles' => 'nullable|array',
            'staff_ids' => 'nullable|array',
        ]);

        $assignType = $validated['assign_type'];
        $shiftTypeId = $validated['shift_type_id'];
        $fromDate = $validated['from_date'];
        $toDate = $validated['to_date'] ?? null;
        $selectedDepts = $validated['departments'] ?? [];
        $selectedRoles = $validated['roles'] ?? [];
        
        // Determine which staff to assign
        $staffIds = [];
        
        // Get all staff without active shifts
        $availableStaffIds = \Modules\Attendance\Services\AttendanceHelper::getStaffWithoutShifts()->pluck('id')->toArray();
        
        if ($assignType === 'specific') {
            // Use specifically selected staff, but only those without shifts
            $requestedIds = $validated['staff_ids'] ?? [];
            $staffIds = array_intersect($requestedIds, $availableStaffIds);
        } else {
            // Filter-based assignment - only from staff without shifts
            $query = Staff::where('status', true)
                ->whereIn('id', $availableStaffIds);
            
            // Filter by departments if selected
            if (!empty($selectedDepts)) {
                $query->whereIn('department', $selectedDepts);
            }
            
            // Filter by roles if selected
            if (!empty($selectedRoles)) {
                $query->whereHas('admin.roles', function($q) use ($selectedRoles) {
                    $q->whereIn('name', $selectedRoles);
                });
            }
            
            $staffIds = $query->pluck('id')->toArray();
        }
        
        if (empty($staffIds)) {
            return redirect()->back()->with('error', 'No staff selected for assignment. Please check your filters.')->withInput();
        }

        $created = 0;
        $updated = 0;

        foreach ($staffIds as $staffId) {
            // Check if staff already has active shift of this type for overlapping period
            $existing = Shift::where('staff_id', $staffId)
                ->where('shift_type_id', $shiftTypeId)
                ->where('status', true)
                ->where('from_date', '<=', $toDate ?? '2099-12-31')
                ->where(function($q) use ($fromDate) {
                    $q->whereNull('to_date')
                      ->orWhere('to_date', '>=', $fromDate);
                })
                ->first();

            if ($existing) {
                // Update existing - extend the date range
                $existing->update([
                    'from_date' => min($existing->from_date->format('Y-m-d'), $fromDate),
                    'to_date' => $toDate,
                ]);
                $updated++;
            } else {
                Shift::create([
                    'shift_type_id' => $shiftTypeId,
                    'staff_id' => $staffId,
                    'from_date' => $fromDate,
                    'to_date' => $toDate,
                    'repeat_type' => 'weekly',
                    'repeat_days' => ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    'status' => true,
                ]);
                $created++;
                
                // Assign attendance-staff role to this staff member
                AttendanceService::assignStaffRole($staffId);
            }
        }

        $total = count($staffIds);
        $msg = '';
        if ($created > 0) {
            $msg .= "{$created} new shift assignment(s) created";
        }
        if ($updated > 0) {
            $msg .= ($msg ? ', ' : '') . "{$updated} existing assignment(s) updated";
        }

        return redirect()->route('admin.attendance.shifts.index')
            ->with('success', $msg ?: 'Shift assignments processed for ' . $total . ' staff');
    }

    public function edit($id)
    {
        $shift = Shift::with(['shiftType', 'staff'])->findOrFail($id);
        $pageTitle = 'Edit Shift';
        
        $departments = Staff::whereNotNull('department')
            ->where('department', '!=', '')
            ->distinct()
            ->pluck('department')
            ->sort();
            
        $roles = Role::where('guard_name', 'admin')->orderBy('name')->get();
        $staffList = Staff::where('status', true)->orderBy('first_name')->get();
        $shiftTypes = ShiftType::where('status', true)->orderBy('name')->get();
        $weekdays = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];

        return view('attendance::shifts.edit', compact('shift', 'pageTitle', 'departments', 'roles', 'staffList', 'shiftTypes', 'weekdays'));
    }

    public function update(Request $request, $id)
    {
        $shift = Shift::findOrFail($id);
        $oldStaffId = $shift->staff_id;
        $oldStatus = $shift->status;

        $validated = $request->validate([
            'department' => 'nullable|string',
            'role_name' => 'nullable|string',
            'staff_id' => 'nullable|exists:staffs,id',
            'shift_type_id' => 'required|exists:att_shift_types,id',
            'from_date' => 'required|date',
            'to_date' => 'nullable|date|after_or_equal:from_date',
            'repeat_type' => 'required|in:weekly,specific',
            'repeat_days' => 'nullable|array',
        ]);

        $validated['status'] = $request->boolean('status');
        $validated['department'] = $validated['department'] ?: null;
        $validated['role_name'] = $validated['role_name'] ?: null;
        $validated['staff_id'] = $validated['staff_id'] ?: null;

        if ($validated['repeat_type'] === 'weekly' && empty($validated['repeat_days'])) {
            $validated['repeat_days'] = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri'];
        }

        $shift->update($validated);

        // Handle role assignment changes
        $newStaffId = $validated['staff_id'];
        $newStatus = $validated['status'];

        // If staff changed, handle roles for both old and new staff
        if ($oldStaffId != $newStaffId) {
            // Remove role from old staff if they have no other shifts
            if ($oldStaffId) {
                AttendanceService::removeStaffRole($oldStaffId);
            }
            // Assign role to new staff
            if ($newStaffId && $newStatus) {
                AttendanceService::assignStaffRole($newStaffId);
            }
        } 
        // If same staff but status changed
        elseif ($oldStatus != $newStatus && $newStaffId) {
            if ($newStatus) {
                AttendanceService::assignStaffRole($newStaffId);
            } else {
                AttendanceService::removeStaffRole($newStaffId);
            }
        }

        return redirect()->route('admin.attendance.shifts.index')
            ->with('success', 'Shift updated successfully');
    }

    public function destroy($id)
    {
        $shift = Shift::findOrFail($id);
        $staffId = $shift->staff_id;
        
        $shift->delete();
        
        // Remove role from staff if they have no more shifts
        if ($staffId) {
            AttendanceService::removeStaffRole($staffId);
        }
        
        return response()->json(['success' => true, 'message' => 'Shift deleted successfully']);
    }

    public function filterStaff(Request $request)
    {
        $query = Staff::where('status', true);

        // Handle multiple departments
        if ($request->filled('departments')) {
            $depts = array_filter(explode(',', $request->departments));
            if (!empty($depts)) {
                $query->whereIn('department', $depts);
            }
        } elseif ($request->filled('department')) {
            $query->where('department', $request->department);
        }

        // Handle multiple roles
        if ($request->filled('roles')) {
            $roles = array_filter(explode(',', $request->roles));
            if (!empty($roles)) {
                $query->whereHas('admin.roles', function ($q) use ($roles) {
                    $q->whereIn('name', $roles);
                });
            }
        } elseif ($request->filled('role')) {
            $query->whereHas('admin.roles', function ($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        $staff = $query->orderBy('first_name')->get()->map(fn($s) => [
            'id' => $s->id,
            'name' => trim($s->first_name . ' ' . $s->last_name),
            'department' => $s->department,
        ]);

        return response()->json($staff);
    }
}
