<?php

namespace Modules\Attendance\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use App\Traits\DataTable;
use Illuminate\Http\Request;
use Modules\Attendance\Models\ShiftType;

class ShiftTypeController extends AdminController
{
    use DataTable;

    // =========================================
    // DATATABLE CONFIGURATION
    // =========================================

    protected $model = ShiftType::class;
    protected $viewPrefix = 'attendance::shift_types';
    protected $routePrefix = 'admin.attendance.shift_types';

    protected $searchable = ['name', 'description'];
    protected $sortable = ['id', 'name', 'start_time', 'end_time', 'status', 'created_at'];
    protected $filterable = ['status'];

    /**
     * Map each row for DataTable
     */
    protected function mapRow($item)
    {
        return [
            'id' => $item->id,
            'name' => $item->name,
            'color' => $item->color,
            'work_timing' => $item->work_timing,
            'lunch_timing' => $item->lunch_timing ?: '-',
            'total_hours' => $item->total_work_hours . ' hrs',
            'description' => $item->description ? \Str::limit($item->description, 50) : '-',
            'status' => $item->status,
            '_edit_url' => route('admin.attendance.shift_types.edit', $item->id),
            '_delete_url' => route('admin.attendance.shift_types.destroy', $item->id),
        ];
    }

    // =========================================
    // INDEX
    // =========================================

    public function index()
    {
        $pageTitle = 'Shift Types';
        return view('attendance::shift_types.index', compact('pageTitle'));
    }

    // =========================================
    // CREATE / STORE
    // =========================================

    public function create()
    {
        $pageTitle = 'Add Shift Type';
        return view('attendance::shift_types.create', compact('pageTitle'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:att_shift_types,name',
            'color' => 'required|string|max:20',
            'start_time' => 'required',
            'end_time' => 'required',
            'lunch_start_time' => 'nullable',
            'lunch_end_time' => 'nullable|required_with:lunch_start_time',
            'days_off' => 'nullable|array',
            'description' => 'nullable|string|max:500',
            'location_restricted' => 'nullable',
            'location_name' => 'nullable|string|max:255',
            'location_latitude' => 'nullable|numeric|between:-90,90',
            'location_longitude' => 'nullable|numeric|between:-180,180',
            'allowed_radius' => 'nullable|integer|min:10|max:5000',
        ]);

        $validated['status'] = $request->boolean('status', true);
        $validated['days_off'] = $request->input('days_off', []);
        $validated['location_restricted'] = $request->boolean('location_restricted');
        $validated['allowed_radius'] = $request->input('allowed_radius', 100);

        ShiftType::create($validated);

        return redirect()->route('admin.attendance.shift_types.index')
            ->with('success', 'Shift type created successfully');
    }

    // =========================================
    // EDIT / UPDATE
    // =========================================

    public function edit($id)
    {
        $shiftType = ShiftType::findOrFail($id);
        $pageTitle = 'Edit Shift Type: ' . $shiftType->name;
        return view('attendance::shift_types.edit', compact('shiftType', 'pageTitle'));
    }

    public function update(Request $request, $id)
    {
        $shiftType = ShiftType::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:att_shift_types,name,' . $id,
            'color' => 'required|string|max:20',
            'start_time' => 'required',
            'end_time' => 'required',
            'lunch_start_time' => 'nullable',
            'lunch_end_time' => 'nullable|required_with:lunch_start_time',
            'days_off' => 'nullable|array',
            'description' => 'nullable|string|max:500',
            'location_restricted' => 'nullable',
            'location_name' => 'nullable|string|max:255',
            'location_latitude' => 'nullable|numeric|between:-90,90',
            'location_longitude' => 'nullable|numeric|between:-180,180',
            'allowed_radius' => 'nullable|integer|min:10|max:5000',
        ]);

        $validated['status'] = $request->boolean('status');
        $validated['days_off'] = $request->input('days_off', []);
        $validated['location_restricted'] = $request->boolean('location_restricted');
        $validated['allowed_radius'] = $request->input('allowed_radius', 100);
        
        // Clear location data if restriction disabled
        if (!$validated['location_restricted']) {
            $validated['location_name'] = null;
            $validated['location_latitude'] = null;
            $validated['location_longitude'] = null;
        }

        $shiftType->update($validated);

        return redirect()->route('admin.attendance.shift_types.index')
            ->with('success', 'Shift type updated successfully');
    }

    // =========================================
    // DELETE
    // =========================================

    public function destroy($id)
    {
        ShiftType::findOrFail($id)->delete();
        return response()->json(['success' => true, 'message' => 'Shift type deleted successfully']);
    }
}
