<?php

namespace Modules\Attendance\Livewire;

use Livewire\Component;
use App\Models\Admin\Staff;
use Modules\Attendance\Models\Shift;
use Modules\Attendance\Models\ShiftType;
use Spatie\Permission\Models\Role;
use Carbon\Carbon;

class WorkShiftTable extends Component
{
    // Filters
    public $month;
    public $year;
    public $filterDepartment = '';
    public $filterRole = '';
    public $filterStaff = [];
    
    // Search
    public $staffSearch = '';
    
    // Data
    public $departments = [];
    public $roles = [];
    public $shiftTypes = [];
    
    // Pagination
    public $perPage = 20;
    public $page = 1;
    
    protected $queryString = ['month', 'year', 'filterDepartment', 'filterRole'];
    
    public function mount()
    {
        $this->month = $this->month ?: now()->month;
        $this->year = $this->year ?: now()->year;
        
        $this->departments = Staff::whereNotNull('department')
            ->where('department', '!=', '')
            ->distinct()
            ->pluck('department')
            ->sort()
            ->values()
            ->toArray();
            
        $this->roles = Role::where('guard_name', 'admin')
            ->orderBy('name')
            ->pluck('name')
            ->toArray();
            
        $this->shiftTypes = ShiftType::where('status', true)
            ->get()
            ->keyBy('id')
            ->toArray();
    }
    
    public function updatedFilterDepartment()
    {
        $this->filterStaff = [];
        $this->page = 1;
    }
    
    public function updatedFilterRole()
    {
        $this->filterStaff = [];
        $this->page = 1;
    }
    
    public function updatedMonth()
    {
        $this->page = 1;
    }
    
    public function previousMonth()
    {
        $date = Carbon::createFromDate($this->year, $this->month, 1)->subMonth();
        $this->month = $date->month;
        $this->year = $date->year;
    }
    
    public function nextMonth()
    {
        $date = Carbon::createFromDate($this->year, $this->month, 1)->addMonth();
        $this->month = $date->month;
        $this->year = $date->year;
    }
    
    public function loadMore()
    {
        $this->perPage += 20;
    }
    
    public function getStaffListProperty()
    {
        $query = Staff::where('status', true);
        
        if ($this->filterDepartment) {
            $query->where('department', $this->filterDepartment);
        }
        
        if ($this->filterRole) {
            $query->whereHas('admin.roles', function ($q) {
                $q->where('name', $this->filterRole);
            });
        }
        
        if (!empty($this->filterStaff)) {
            $query->whereIn('id', $this->filterStaff);
        }
        
        return $query->orderBy('first_name')->take($this->perPage)->get();
    }
    
    public function getFilteredStaffOptionsProperty()
    {
        $query = Staff::where('status', true);
        
        if ($this->filterDepartment) {
            $query->where('department', $this->filterDepartment);
        }
        
        if ($this->filterRole) {
            $query->whereHas('admin.roles', function ($q) {
                $q->where('name', $this->filterRole);
            });
        }
        
        if ($this->staffSearch) {
            $query->where(function ($q) {
                $q->where('first_name', 'like', '%' . $this->staffSearch . '%')
                  ->orWhere('last_name', 'like', '%' . $this->staffSearch . '%');
            });
        }
        
        return $query->orderBy('first_name')->limit(50)->get();
    }
    
    public function getTotalStaffCountProperty()
    {
        $query = Staff::where('status', true);
        
        if ($this->filterDepartment) {
            $query->where('department', $this->filterDepartment);
        }
        
        if ($this->filterRole) {
            $query->whereHas('admin.roles', function ($q) {
                $q->where('name', $this->filterRole);
            });
        }
        
        if (!empty($this->filterStaff)) {
            $query->whereIn('id', $this->filterStaff);
        }
        
        return $query->count();
    }
    
    public function getDaysInMonthProperty()
    {
        $start = Carbon::createFromDate($this->year, $this->month, 1);
        $end = $start->copy()->endOfMonth();
        
        $days = [];
        for ($date = $start->copy(); $date <= $end; $date->addDay()) {
            $days[] = [
                'date' => $date->format('Y-m-d'),
                'day' => $date->format('d'),
                'dayName' => $date->format('D'),
                'isWeekend' => $date->isWeekend(),
                'isToday' => $date->isToday(),
            ];
        }
        
        return $days;
    }
    
    public function getShiftsDataProperty()
    {
        $staffList = $this->staffList;
        $startDate = Carbon::createFromDate($this->year, $this->month, 1)->format('Y-m-d');
        $endDate = Carbon::createFromDate($this->year, $this->month, 1)->endOfMonth()->format('Y-m-d');
        
        // Get all active shifts in this date range
        $shifts = Shift::with('shiftType')
            ->where('status', true)
            ->where('from_date', '<=', $endDate)
            ->where(function ($q) use ($startDate) {
                $q->whereNull('to_date')
                  ->orWhere('to_date', '>=', $startDate);
            })
            ->get();
        
        // Build shift map: staffId -> date -> shiftType
        $shiftMap = [];
        
        foreach ($staffList as $staff) {
            $staffId = $staff->id;
            $staffDept = $staff->department;
            $staffRoles = $staff->admin ? $staff->admin->getRoleNames()->toArray() : [];
            
            foreach ($shifts as $shift) {
                // Check if this shift applies to this staff
                $applies = false;
                
                // Direct staff assignment
                if ($shift->staff_id == $staffId) {
                    $applies = true;
                }
                // Department assignment (no specific staff)
                elseif (!$shift->staff_id && $shift->department && $shift->department == $staffDept) {
                    // Check role if specified
                    if ($shift->role_name) {
                        $applies = in_array($shift->role_name, $staffRoles);
                    } else {
                        $applies = true;
                    }
                }
                // Role-only assignment
                elseif (!$shift->staff_id && !$shift->department && $shift->role_name) {
                    $applies = in_array($shift->role_name, $staffRoles);
                }
                // Global (all staff) - no department, role, or staff specified
                elseif (!$shift->staff_id && !$shift->department && !$shift->role_name) {
                    $applies = true;
                }
                
                if (!$applies) continue;
                
                // Now map this shift to each day
                foreach ($this->daysInMonth as $day) {
                    $date = Carbon::parse($day['date']);
                    $dayName = $date->format('l'); // Full day name: Monday, Tuesday
                    $dayShort = $date->format('D'); // Short: Mon, Tue
                    
                    // Check date range
                    if ($date->lt($shift->from_date)) continue;
                    if ($shift->to_date && $date->gt($shift->to_date)) continue;
                    
                    $showShift = false;
                    
                    // Check repeat type
                    if ($shift->repeat_type === 'weekly' && !empty($shift->repeat_days)) {
                        // repeat_days might be ['Mon', 'Tue'] or ['Monday', 'Tuesday']
                        $showShift = in_array($dayShort, $shift->repeat_days) || in_array($dayName, $shift->repeat_days);
                    } elseif ($shift->repeat_type === 'specific' && !empty($shift->specific_dates)) {
                        $showShift = in_array($day['date'], $shift->specific_dates);
                    } else {
                        // No repeat type or daily - show every day
                        $showShift = true;
                    }
                    
                    // Check days off from shift type
                    if ($showShift && $shift->shiftType && !empty($shift->shiftType->days_off)) {
                        $daysOff = $shift->shiftType->days_off;
                        if (is_string($daysOff)) {
                            $daysOff = json_decode($daysOff, true) ?: [];
                        }
                        // Case-insensitive comparison
                        $daysOffLower = is_array($daysOff) ? array_map('strtolower', $daysOff) : [];
                        if (in_array(strtolower($dayName), $daysOffLower)) {
                            // It's a day off - mark differently
                            if (!isset($shiftMap[$staffId])) {
                                $shiftMap[$staffId] = [];
                            }
                            $shiftMap[$staffId][$day['date']] = [
                                'type' => 'off',
                                'label' => 'OFF',
                                'color' => '#9ca3af'
                            ];
                            continue;
                        }
                    }
                    
                    if ($showShift && $shift->shiftType) {
                        if (!isset($shiftMap[$staffId])) {
                            $shiftMap[$staffId] = [];
                        }
                        $shiftMap[$staffId][$day['date']] = [
                            'type' => 'shift',
                            'shift_type_id' => $shift->shift_type_id,
                            'name' => $shift->shiftType->name,
                            'code' => $shift->shiftType->code ?? substr($shift->shiftType->name, 0, 2),
                            'start' => $shift->shiftType->start_time ? Carbon::parse($shift->shiftType->start_time)->format('H:i') : null,
                            'end' => $shift->shiftType->end_time ? Carbon::parse($shift->shiftType->end_time)->format('H:i') : null,
                            'color' => $shift->shiftType->color ?? '#3b82f6'
                        ];
                    }
                }
            }
        }
        
        return $shiftMap;
    }
    
    public function toggleStaffFilter($staffId)
    {
        if (in_array($staffId, $this->filterStaff)) {
            $this->filterStaff = array_values(array_diff($this->filterStaff, [$staffId]));
        } else {
            $this->filterStaff[] = $staffId;
        }
        $this->page = 1;
    }
    
    public function clearStaffFilter()
    {
        $this->filterStaff = [];
    }
    
    public function render()
    {
        return view('attendance::livewire.work-shift-table', [
            'staffList' => $this->staffList,
            'daysInMonth' => $this->daysInMonth,
            'shiftsData' => $this->shiftsData,
            'totalStaffCount' => $this->totalStaffCount,
            'filteredStaffOptions' => $this->filteredStaffOptions,
            'monthName' => Carbon::createFromDate($this->year, $this->month, 1)->format('F Y'),
        ]);
    }
}
