<?php

namespace Modules\Attendance\Providers;

use Illuminate\Support\ServiceProvider;
use Livewire\Livewire;

class AttendanceServiceProvider extends ServiceProvider
{
    protected string $moduleName = 'Attendance';
    protected string $moduleNameLower = 'attendance';

    public function boot(): void
    {
        $this->registerConfig();
        $this->registerViews();
        $this->registerLivewireComponents();
        $this->loadMigrationsFrom(module_path($this->moduleName, 'Database/Migrations'));
        
        // Create attendance-staff role if not exists
        $this->createAttendanceRole();
    }

    public function register(): void
    {
        $this->app->register(RouteServiceProvider::class);
    }

    /**
     * Create the attendance-staff role on module boot
     */
    protected function createAttendanceRole(): void
    {
        // Only run after migrations (check if roles table exists)
        if (!$this->app->runningInConsole() || $this->app->runningUnitTests()) {
            try {
                if (\Schema::hasTable('roles')) {
                    \Spatie\Permission\Models\Role::firstOrCreate(
                        ['name' => 'attendance-staff', 'guard_name' => 'admin']
                    );
                }
            } catch (\Exception $e) {
                // Silently fail if roles table doesn't exist yet
            }
        }
    }

    protected function registerConfig(): void
    {
        $this->publishes([
            module_path($this->moduleName, 'Config/config.php') => config_path($this->moduleNameLower . '.php'),
        ], 'config');
        
        $this->mergeConfigFrom(
            module_path($this->moduleName, 'Config/config.php'), 
            $this->moduleNameLower
        );
    }

    protected function registerViews(): void
    {
        $viewPath = resource_path('views/modules/' . $this->moduleNameLower);
        $sourcePath = module_path($this->moduleName, 'Resources/views');

        $this->publishes([
            $sourcePath => $viewPath
        ], ['views', $this->moduleNameLower . '-module-views']);

        $this->loadViewsFrom(array_merge($this->getPublishableViewPaths(), [$sourcePath]), $this->moduleNameLower);
    }

    protected function registerLivewireComponents(): void
    {
        // Register Livewire components with module namespace
        Livewire::component('attendance::work-shift-table', \Modules\Attendance\Livewire\WorkShiftTable::class);
        Livewire::component('attendance::check-in-out', \Modules\Attendance\Livewire\CheckInOut::class);
        Livewire::component('attendance::attendance-dashboard', \Modules\Attendance\Livewire\AttendanceDashboard::class);
        Livewire::component('attendance::my-extra-hours', \Modules\Attendance\Livewire\MyExtraHours::class);
    }

    private function getPublishableViewPaths(): array
    {
        $paths = [];
        foreach (config('view.paths') as $path) {
            if (is_dir($path . '/modules/' . $this->moduleNameLower)) {
                $paths[] = $path . '/modules/' . $this->moduleNameLower;
            }
        }
        return $paths;
    }

    public function provides(): array
    {
        return [];
    }
}
