<?php

namespace Modules\Attendance\Services;

use App\Models\Admin;
use App\Models\Admin\Staff;
use Modules\Attendance\Models\AttendanceSetting;
use Illuminate\Support\Facades\Cache;

class AttendanceHelper
{
    /**
     * Check if current logged-in admin should track attendance
     * 
     * @return bool
     */
    public static function shouldTrackCurrentUser(): bool
    {
        $admin = auth('admin')->user();
        if (!$admin) {
            return false;
        }

        return self::shouldTrackAdmin($admin);
    }

    /**
     * Check if a specific admin should track attendance
     * 
     * @param Admin $admin
     * @return bool
     */
    public static function shouldTrackAdmin(Admin $admin): bool
    {
        // 1. Check if is_admin = 1 (system owner like EchoPx)
        if ($admin->is_admin == 1) {
            $trackIsAdmin = self::getSetting('track_is_admin', '0');
            if ($trackIsAdmin !== '1') {
                return false;
            }
        }

        // 2. Check if admin's roles are exempt
        $exemptRoles = self::getExemptRoles();
        if (!empty($exemptRoles)) {
            foreach ($admin->roles as $role) {
                if (in_array($role->name, $exemptRoles)) {
                    return false;
                }
            }
        }

        // 3. Check if admin has staff record
        $staff = $admin->staff;
        if (!$staff) {
            $trackAdminOnly = self::getSetting('track_admin_only', '0');
            if ($trackAdminOnly !== '1') {
                return false; // Don't track admins without staff record
            }
        }

        // 4. Check if staff is active and not exited
        if ($staff) {
            if (!$staff->status) {
                return false; // Inactive staff
            }
            if ($staff->exit_date && $staff->exit_date < today()) {
                return false; // Already exited
            }
        }

        return true;
    }

    /**
     * Get staff record for current admin (create if enabled in settings)
     * 
     * @return Staff|null
     */
    public static function getOrCreateStaffForCurrentAdmin(): ?Staff
    {
        $admin = auth('admin')->user();
        if (!$admin) {
            return null;
        }

        return self::getOrCreateStaffForAdmin($admin);
    }

    /**
     * Get staff record for admin, auto-create if setting enabled
     * 
     * @param Admin $admin
     * @return Staff|null
     */
    public static function getOrCreateStaffForAdmin(Admin $admin): ?Staff
    {
        // Return existing staff if found
        if ($admin->staff) {
            return $admin->staff;
        }

        // Check if auto-create is enabled
        $trackAdminOnly = self::getSetting('track_admin_only', '0');
        if ($trackAdminOnly !== '1') {
            return null;
        }

        // Auto-create staff profile for this admin
        $nameParts = explode(' ', $admin->name, 2);
        
        $staff = Staff::create([
            'admin_id' => $admin->id,
            'first_name' => $nameParts[0] ?? $admin->name,
            'last_name' => $nameParts[1] ?? '',
            'email' => $admin->email,
            'employee_code' => 'AUTO-' . str_pad($admin->id, 4, '0', STR_PAD_LEFT),
            'status' => $admin->is_active,
            'join_date' => $admin->created_at->toDateString(),
        ]);

        // Refresh admin model to load the new staff relationship
        $admin->refresh();

        return $staff;
    }

    /**
     * Get current admin's staff_id (for attendance tracking)
     * 
     * @return int|null
     */
    public static function getCurrentStaffId(): ?int
    {
        $admin = auth('admin')->user();
        if (!$admin) {
            return null;
        }

        // Check if should track
        if (!self::shouldTrackAdmin($admin)) {
            return null;
        }

        // Get or create staff
        $staff = self::getOrCreateStaffForAdmin($admin);
        
        return $staff ? $staff->id : null;
    }

    /**
     * Get all staff IDs that should be tracked for attendance
     * 
     * @return \Illuminate\Support\Collection
     */
    public static function getTrackableStaff()
    {
        $exemptRoles = self::getExemptRoles();
        $trackIsAdmin = self::getSetting('track_is_admin', '0') === '1';
        $trackAdminOnly = self::getSetting('track_admin_only', '0') === '1';

        $query = Staff::query()
            ->where('status', true)
            ->where(function ($q) {
                $q->whereNull('exit_date')
                  ->orWhere('exit_date', '>=', today());
            });

        // If not tracking admins without staff, only get those with staff records
        // This is already handled since we're querying Staff table

        // Filter out is_admin = 1 if not tracking them
        if (!$trackIsAdmin) {
            $query->whereHas('admin', function ($q) {
                $q->where('is_admin', '!=', 1);
            });
        }

        // Filter out exempt roles
        if (!empty($exemptRoles)) {
            $query->where(function ($q) use ($exemptRoles) {
                $q->whereDoesntHave('admin.roles', function ($rq) use ($exemptRoles) {
                    $rq->whereIn('name', $exemptRoles);
                })->orWhereNull('admin_id');
            });
        }

        return $query->get();
    }

    /**
     * Get exempt roles from settings
     * 
     * @return array
     */
    public static function getExemptRoles(): array
    {
        $setting = self::getSetting('exempt_roles', '[]');
        return json_decode($setting, true) ?: [];
    }

    /**
     * Get a single setting value
     * 
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function getSetting(string $key, $default = null)
    {
        return Cache::remember("att_setting_{$key}", 3600, function () use ($key, $default) {
            $setting = AttendanceSetting::where('key', $key)->first();
            return $setting ? $setting->value : $default;
        });
    }

    /**
     * Clear settings cache
     */
    public static function clearSettingsCache(): void
    {
        $keys = [
            'track_is_admin',
            'track_admin_only', 
            'exempt_roles',
            'allow_self_checkin',
            'require_location',
            'work_hours_per_day',
            'overtime_multiplier',
        ];

        foreach ($keys as $key) {
            Cache::forget("att_setting_{$key}");
        }
    }

    /**
     * Check if self check-in is allowed
     * 
     * @return bool
     */
    public static function isSelfCheckInAllowed(): bool
    {
        return self::getSetting('allow_self_checkin', '1') === '1';
    }

    /**
     * Check if location is required for check-in
     * 
     * @return bool
     */
    public static function isLocationRequired(): bool
    {
        return self::getSetting('require_location', '0') === '1';
    }

    /**
     * Get standard work hours per day
     * 
     * @return float
     */
    public static function getStandardWorkHours(): float
    {
        return (float) self::getSetting('work_hours_per_day', '8');
    }

    /**
     * Get overtime multiplier
     * 
     * @return float
     */
    public static function getOvertimeMultiplier(): float
    {
        return (float) self::getSetting('overtime_multiplier', '1.5');
    }

    /**
     * Get staff WITH active shifts assigned
     * Use for: Apply Leave, Assign Extra Work, Leave Management
     * 
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getStaffWithShifts()
    {
        // Get staff IDs that have active shifts (status = true/1)
        $staffIdsWithShifts = \Modules\Attendance\Models\Shift::where('status', true)
            ->pluck('staff_id')
            ->unique()
            ->toArray();

        return Staff::query()
            ->where('status', true)
            ->where(function ($q) {
                $q->whereNull('exit_date')
                  ->orWhere('exit_date', '>=', today());
            })
            ->whereIn('id', $staffIdsWithShifts)
            ->orderBy('first_name')
            ->get();
    }

    /**
     * Get staff WITHOUT active shifts assigned
     * Use for: Shift Assignment page
     * 
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getStaffWithoutShifts()
    {
        // Get staff IDs that have active shifts (status = true/1)
        $staffIdsWithShifts = \Modules\Attendance\Models\Shift::where('status', true)
            ->pluck('staff_id')
            ->unique()
            ->toArray();

        return Staff::query()
            ->where('status', true)
            ->where(function ($q) {
                $q->whereNull('exit_date')
                  ->orWhere('exit_date', '>=', today());
            })
            ->whereNotIn('id', $staffIdsWithShifts)
            ->orderBy('first_name')
            ->get();
    }

    /**
     * Check if a specific staff has an active shift assigned
     * 
     * @param int $staffId
     * @return bool
     */
    public static function hasActiveShift(int $staffId): bool
    {
        return \Modules\Attendance\Models\Shift::where('staff_id', $staffId)
            ->where('status', true)
            ->exists();
    }

    /**
     * Get current staff's active shift
     * 
     * @param int $staffId
     * @return \Modules\Attendance\Models\Shift|null
     */
    public static function getActiveShift(int $staffId)
    {
        return \Modules\Attendance\Models\Shift::where('staff_id', $staffId)
            ->where('status', true)
            ->with('shiftType')
            ->first();
    }
}
