<?php

namespace Modules\Attendance\Services;

class AttendanceService
{
    public static function config(): array
    {
        return [
            'name' => 'Attendance',
            'alias' => 'attendance',
            'description' => 'Staff Attendance, Shift, Leave Management',
            'version' => '1.0.0',
            'is_core' => false,
            'sort_order' => 30,
        ];
    }

    public static function menus(): array
    {
        return [
            // Staff Menu - My Attendance (visible to all staff)
            [
                'menu_name' => 'My Attendance',
                'slug' => 'my-attendance',
                'icon' => 'user-clock',
                'route' => null,
                'category' => 'core',
                'sort_order' => 5,
                'actions' => ['read'],
                'children' => [
                    [
                        'menu_name' => 'My Attendance',
                        'slug' => 'my-attendance-view',
                        'icon' => 'calendar-check',
                        'route' => 'admin.attendance.my',
                        'actions' => ['read'],
                    ],
                    [
                        'menu_name' => 'My Leaves',
                        'slug' => 'my-leaves',
                        'icon' => 'calendar-times',
                        'route' => 'admin.attendance.my_leaves',
                        'actions' => ['read'],
                    ],
                    [
                        'menu_name' => 'My Extra Hours',
                        'slug' => 'my-extra-hours',
                        'icon' => 'clock',
                        'route' => 'admin.attendance.my_extra',
                        'actions' => ['read'],
                    ],
                ],
            ],
            
            // Admin Menu - Attendance Management with children
            [
                'menu_name' => 'Attendance',
                'slug' => 'attendance',
                'icon' => 'clipboard-check',
                'route' => null,
                'category' => 'core',
                'sort_order' => 30,
                'actions' => ['read', 'create', 'edit', 'delete', 'export'],
                'children' => [
                    [
                        'menu_name' => 'Dashboard',
                        'slug' => 'attendance-dashboard',
                        'icon' => 'chart-bar',
                        'route' => 'admin.attendance.index',
                        'actions' => ['read'],
                    ],
                    [
                        'menu_name' => 'Attendance Logs',
                        'slug' => 'attendance-logs',
                        'icon' => 'clipboard-list',
                        'route' => 'admin.attendance.logs',
                        'actions' => ['read'],
                    ],
                    [
                        'menu_name' => 'Shift Types',
                        'slug' => 'attendance-shift-types',
                        'icon' => 'clock',
                        'route' => 'admin.attendance.shift_types.index',
                        'actions' => ['read', 'create', 'edit', 'delete'],
                    ],
                    [
                        'menu_name' => 'Shift Assignments',
                        'slug' => 'attendance-shifts',
                        'icon' => 'calendar',
                        'route' => 'admin.attendance.shifts.index',
                        'actions' => ['read', 'create', 'edit', 'delete'],
                    ],
                    [
                        'menu_name' => 'Work Shift Calendar',
                        'slug' => 'attendance-work-shift',
                        'icon' => 'calendar-alt',
                        'route' => 'admin.attendance.work_shift.index',
                        'actions' => ['read'],
                    ],
                    [
                        'menu_name' => 'Leave Applications',
                        'slug' => 'attendance-leave',
                        'icon' => 'plane-departure',
                        'route' => 'admin.attendance.leave.index',
                        'actions' => ['read', 'create', 'edit', 'delete', 'approve'],
                    ],
                    [
                        'menu_name' => 'Additional Work Hours',
                        'slug' => 'attendance-additional-hours',
                        'icon' => 'plus-circle',
                        'route' => 'admin.attendance.additional_hours.index',
                        'actions' => ['read', 'create', 'edit', 'delete', 'approve'],
                    ],
                ],
            ],
            
            // Settings menu - Goes under main Settings (category: settings)
            [
                'menu_name' => 'Attendance',
                'slug' => 'attendance-settings',
                'icon' => 'clipboard-check',
                'route' => 'admin.attendance.settings.index',
                'category' => 'settings',
                'sort_order' => 20,
                'actions' => ['read', 'edit'],
            ],
        ];
    }

    /**
     * Called when module is uninstalled
     * Removes the attendance-staff role
     */
    public static function onUninstall(): void
    {
        try {
            if (class_exists(\Spatie\Permission\Models\Role::class)) {
                // Remove role from all users first
                $role = \Spatie\Permission\Models\Role::where('name', 'attendance-staff')
                    ->where('guard_name', 'admin')
                    ->first();
                
                if ($role) {
                    // Remove role from all admins
                    \DB::table('model_has_roles')
                        ->where('role_id', $role->id)
                        ->delete();
                    
                    // Delete the role
                    $role->delete();
                }
                
                // Clear permission cache
                app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
            }
        } catch (\Exception $e) {
            \Log::error('Attendance module uninstall error: ' . $e->getMessage());
        }
    }

    /**
     * Assign attendance-staff role to a staff member
     */
    public static function assignStaffRole(int $staffId): void
    {
        try {
            $staff = \App\Models\Admin\Staff::find($staffId);
            if ($staff && $staff->admin) {
                $staff->admin->assignRole('attendance-staff');
                app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
            }
        } catch (\Exception $e) {
            \Log::error('Failed to assign attendance-staff role: ' . $e->getMessage());
        }
    }

    /**
     * Remove attendance-staff role from a staff member
     * Only removes if they have no more shift assignments
     */
    public static function removeStaffRole(int $staffId): void
    {
        try {
            // Check if staff has any other active shifts
            $hasOtherShifts = \Modules\Attendance\Models\Shift::where('staff_id', $staffId)
                ->where('status', true)
                ->exists();
            
            if (!$hasOtherShifts) {
                $staff = \App\Models\Admin\Staff::find($staffId);
                if ($staff && $staff->admin) {
                    $staff->admin->removeRole('attendance-staff');
                    app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to remove attendance-staff role: ' . $e->getMessage());
        }
    }

    public static function isInstalled(): bool
    {
        return class_exists(\Modules\Attendance\Models\Attendance::class);
    }

    public static function navbarItems(): array
    {
        return [
            [
                'type' => 'widget',
                'id' => 'attendance-checkin',
                'position' => 'right',
                'sort_order' => 5,
                'icon' => 'clock',
                'label' => 'Attendance',
                'view' => 'attendance::partials.navbar-checkin',
            ],
        ];
    }
}
