<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     * 
     * Consolidated migration for ecommerce module:
     * - website_settings (complete with all columns)
     * - website_orders
     * - website_order_items
     * - website_order_status_history
     * - product_reviews
     * - customers table modifications
     */
    public function up(): void
    {
        // =====================================================
        // 1. CREATE WEBSITE_SETTINGS TABLE
        // =====================================================
        Schema::create('website_settings', function (Blueprint $table) {
            $table->id();
            
            // Basic Settings
            $table->string('site_name')->nullable();
            $table->string('site_url')->nullable();
            $table->string('site_prefix')->nullable()->default('site');
            $table->string('shop_prefix')->nullable()->default('shop');
            $table->string('site_logo')->nullable();
            $table->string('site_favicon')->nullable();
            $table->enum('site_mode', ['website_only', 'ecommerce_only', 'both'])->default('both');
            $table->unsignedBigInteger('homepage_id')->nullable();
            
            // Contact Info
            $table->string('contact_email')->nullable();
            $table->string('contact_phone', 50)->nullable();
            
            // Meta/SEO
            $table->string('meta_title')->nullable();
            $table->text('meta_description')->nullable();
            
            // Shipping Settings
            $table->decimal('shipping_fee', 10, 2)->default(0);
            $table->decimal('free_shipping_min', 10, 2)->default(0);
            $table->string('delivery_days', 50)->nullable();
            
            // COD Settings
            $table->boolean('cod_enabled')->default(true);
            $table->decimal('cod_fee', 10, 2)->default(0);
            $table->decimal('cod_max_amount', 12, 2)->default(0);
            
            // Online Payment Settings
            $table->boolean('online_payment_enabled')->default(false);
            $table->string('online_payment_label', 100)->default('Pay Online (UPI/Card/NetBanking)');
            
            // Order Settings
            $table->string('order_prefix', 20)->default('ORD-');
            $table->decimal('min_order_amount', 10, 2)->default(0);
            $table->boolean('guest_checkout')->default(false);
            
            // Tax Settings
            $table->boolean('tax_included_in_price')->default(true);
            $table->boolean('show_tax_breakup')->default(true);
            
            // Store Address
            $table->string('store_address')->nullable();
            $table->string('store_city', 100)->nullable();
            $table->string('store_state', 100)->nullable();
            $table->string('store_pincode', 20)->nullable();
            
            // Location Restriction Settings
            $table->boolean('location_restriction_enabled')->default(false);
            $table->decimal('service_radius_km', 8, 2)->default(5.00);
            $table->decimal('store_latitude', 10, 7)->nullable();
            $table->decimal('store_longitude', 10, 7)->nullable();
            
            // Store Pickup Settings
            $table->string('store_google_maps_link', 500)->nullable();
            $table->boolean('store_pickup_enabled')->default(true);
            $table->string('pickup_ready_time', 100)->nullable()->default('2-4 hours');
            $table->string('pickup_instructions', 500)->nullable();
            
            // Store Tax Info
            $table->string('store_gstin', 20)->nullable();
            
            // Invoice Settings
            $table->string('invoice_prefix', 20)->default('INV-');
            $table->text('invoice_footer')->nullable();
            
            // Notification Settings
            $table->string('order_notification_email')->nullable();
            $table->boolean('send_customer_order_email')->default(true);
            $table->boolean('send_admin_order_alert')->default(true);
            
            // Status
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Insert default settings row
        DB::table('website_settings')->insert([
            'site_name' => config('app.name', 'My Website'),
            'site_url' => config('app.url', 'http://localhost'),
            'site_prefix' => 'site',
            'shop_prefix' => 'shop',
            'site_mode' => 'both',
            'shipping_fee' => 50,
            'free_shipping_min' => 500,
            'delivery_days' => '3-5 business days',
            'cod_enabled' => true,
            'cod_fee' => 0,
            'cod_max_amount' => 0,
            'online_payment_enabled' => false,
            'online_payment_label' => 'Pay Online (UPI/Card/NetBanking)',
            'order_prefix' => 'ORD-',
            'min_order_amount' => 0,
            'guest_checkout' => false,
            'tax_included_in_price' => true,
            'show_tax_breakup' => true,
            'location_restriction_enabled' => false,
            'service_radius_km' => 5.00,
            'store_pickup_enabled' => true,
            'pickup_ready_time' => '2-4 hours',
            'invoice_prefix' => 'INV-',
            'send_customer_order_email' => true,
            'send_admin_order_alert' => true,
            'is_active' => true,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // =====================================================
        // 2. MODIFY CUSTOMERS TABLE
        // =====================================================
        Schema::table('customers', function (Blueprint $table) {
            if (!Schema::hasColumn('customers', 'is_website_user')) {
                $table->boolean('is_website_user')->default(false)->after('is_supplier');
            }
            if (!Schema::hasColumn('customers', 'cart_data')) {
                $table->json('cart_data')->nullable()->after('is_website_user');
            }
        });

        // =====================================================
        // 3. CREATE WEBSITE_ORDERS TABLE
        // =====================================================
        Schema::create('website_orders', function (Blueprint $table) {
            $table->id();
            $table->string('order_no', 50)->unique();
            $table->unsignedBigInteger('customer_id')->nullable();
            $table->unsignedBigInteger('user_id')->nullable();
            
            // Customer Info (snapshot at order time)
            $table->string('customer_name');
            $table->string('customer_email')->nullable();
            $table->string('customer_phone', 20);
            
            // Shipping Address
            $table->text('shipping_address');
            $table->string('shipping_city', 100);
            $table->string('shipping_state', 100);
            $table->string('shipping_pincode', 20);
            $table->string('shipping_country', 100)->default('India');
            
            // Billing Address (optional - same as shipping if null)
            $table->text('billing_address')->nullable();
            $table->string('billing_city', 100)->nullable();
            $table->string('billing_state', 100)->nullable();
            $table->string('billing_pincode', 20)->nullable();
            $table->string('billing_country', 100)->nullable();
            
            // Amounts
            $table->decimal('subtotal', 12, 2)->default(0);
            $table->decimal('tax_amount', 12, 2)->default(0);
            $table->decimal('shipping_fee', 10, 2)->default(0);
            $table->decimal('cod_fee', 10, 2)->default(0);
            $table->decimal('discount_amount', 10, 2)->default(0);
            $table->string('discount_code', 50)->nullable();
            $table->decimal('total', 12, 2)->default(0);
            
            // Status
            $table->enum('status', [
                'pending',
                'confirmed',
                'processing',
                'shipped',
                'delivered',
                'cancelled',
                'returned'
            ])->default('pending');
            
            $table->enum('payment_status', [
                'pending',
                'paid',
                'failed',
                'refunded',
                'partial_refund'
            ])->default('pending');
            
            // Payment Info
            $table->string('payment_method', 50)->default('cash');
            $table->string('transaction_id', 100)->nullable();
            $table->string('payment_gateway', 50)->nullable();
            $table->timestamp('paid_at')->nullable();
            
            // Shipping/Delivery Info
            $table->string('tracking_number', 100)->nullable();
            $table->string('carrier', 100)->nullable();
            $table->timestamp('shipped_at')->nullable();
            $table->timestamp('delivered_at')->nullable();
            
            // Notes
            $table->text('customer_notes')->nullable();
            $table->text('admin_notes')->nullable();
            
            // Tracking Info
            $table->string('ip_address', 45)->nullable();
            $table->string('user_agent')->nullable();
            
            $table->timestamps();
            
            // Indexes
            $table->index('customer_id');
            $table->index('status');
            $table->index('payment_status');
            $table->index('created_at');
        });

        // =====================================================
        // 4. CREATE WEBSITE_ORDER_ITEMS TABLE
        // =====================================================
        Schema::create('website_order_items', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('order_id');
            $table->unsignedBigInteger('product_id');
            $table->unsignedBigInteger('variation_id')->nullable();
            
            // Product Snapshot
            $table->string('product_name');
            $table->string('variation_name')->nullable();
            $table->string('sku', 100);
            $table->string('hsn_code', 50)->nullable();
            
            // Unit Info
            $table->unsignedBigInteger('unit_id')->nullable();
            $table->string('unit_name', 50)->default('PCS');
            
            // Quantity & Pricing
            $table->decimal('qty', 12, 3);
            $table->decimal('unit_price', 12, 2);
            $table->decimal('mrp', 12, 2)->nullable();
            
            // Tax
            $table->decimal('tax_rate', 5, 2)->default(0);
            $table->decimal('tax_amount', 10, 2)->default(0);
            
            // Totals
            $table->decimal('subtotal', 12, 2);
            $table->decimal('total', 12, 2);
            
            $table->timestamps();
            
            // Foreign Keys & Indexes
            $table->foreign('order_id')->references('id')->on('website_orders')->onDelete('cascade');
            $table->index('product_id');
            $table->index('variation_id');
        });

        // =====================================================
        // 5. CREATE WEBSITE_ORDER_STATUS_HISTORY TABLE
        // =====================================================
        Schema::create('website_order_status_history', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('order_id');
            $table->string('status', 50);
            $table->string('comment')->nullable();
            $table->unsignedBigInteger('changed_by')->nullable();
            $table->timestamps();
            
            $table->foreign('order_id')->references('id')->on('website_orders')->onDelete('cascade');
        });

        // =====================================================
        // 6. CREATE PRODUCT_REVIEWS TABLE
        // =====================================================
        Schema::create('product_reviews', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('product_id');
            $table->unsignedBigInteger('customer_id')->nullable();
            $table->unsignedBigInteger('order_id')->nullable();
            $table->string('reviewer_name', 100);
            $table->string('reviewer_email', 255)->nullable();
            $table->tinyInteger('rating')->unsigned();
            $table->string('title', 255)->nullable();
            $table->text('review');
            $table->enum('status', ['pending', 'approved', 'rejected'])->default('pending');
            $table->boolean('is_verified_purchase')->default(false);
            $table->string('admin_reply')->nullable();
            $table->timestamp('replied_at')->nullable();
            $table->timestamps();

            $table->index(['product_id', 'status']);
            $table->index('customer_id');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Drop tables in reverse order (respecting foreign keys)
        Schema::dropIfExists('product_reviews');
        Schema::dropIfExists('website_order_status_history');
        Schema::dropIfExists('website_order_items');
        Schema::dropIfExists('website_orders');
        Schema::dropIfExists('website_settings');
        
        // Remove columns from customers table
        Schema::table('customers', function (Blueprint $table) {
            if (Schema::hasColumn('customers', 'cart_data')) {
                $table->dropColumn('cart_data');
            }
            if (Schema::hasColumn('customers', 'is_website_user')) {
                $table->dropColumn('is_website_user');
            }
        });
    }
};