<?php

namespace Modules\Ecommerce\Http\Controllers\Delivery;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Modules\Ecommerce\Models\DeliveryPerson;

class DeliveryAuthController extends Controller
{
    /**
     * Show login form
     */
    public function showLogin()
    {
        if (Auth::guard('delivery')->check()) {
            return redirect()->route('delivery.dashboard');
        }
        
        return view('ecommerce::delivery.auth.login');
    }

    /**
     * Handle login
     */
    public function login(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'password' => 'required|string',
        ]);

        $deliveryPerson = DeliveryPerson::where('phone', $request->phone)->first();

        if (!$deliveryPerson) {
            return back()->withErrors(['phone' => 'No account found with this phone number.'])->withInput();
        }

        if ($deliveryPerson->status === 'inactive') {
            return back()->withErrors(['phone' => 'Your account is inactive. Please contact admin.'])->withInput();
        }

        if (!Hash::check($request->password, $deliveryPerson->password)) {
            return back()->withErrors(['password' => 'Invalid password.'])->withInput();
        }

        Auth::guard('delivery')->login($deliveryPerson, $request->filled('remember'));

        return redirect()->intended(route('delivery.dashboard'));
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        Auth::guard('delivery')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('delivery.login');
    }

    /**
     * Show profile
     */
    public function profile()
    {
        $deliveryPerson = Auth::guard('delivery')->user();
        return view('ecommerce::delivery.profile', compact('deliveryPerson'));
    }

    /**
     * Update profile
     */
    public function updateProfile(Request $request)
    {
        $deliveryPerson = Auth::guard('delivery')->user();

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:delivery_persons,email,' . $deliveryPerson->id,
            'address' => 'nullable|string|max:500',
            'city' => 'nullable|string|max:100',
        ]);

        $deliveryPerson->update($request->only(['name', 'email', 'address', 'city']));

        return back()->with('success', 'Profile updated successfully.');
    }

    /**
     * Change password
     */
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required|string',
            'password' => 'required|string|min:6|confirmed',
        ]);

        $deliveryPerson = Auth::guard('delivery')->user();

        if (!Hash::check($request->current_password, $deliveryPerson->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect.']);
        }

        $deliveryPerson->update([
            'password' => Hash::make($request->password),
        ]);

        return back()->with('success', 'Password changed successfully.');
    }
}