<?php

namespace Modules\Ecommerce\Http\Controllers\Delivery;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Modules\Ecommerce\Models\WebsiteOrder;
use Modules\Ecommerce\Models\DeliveryStatusHistory;

class DeliveryDashboardController extends Controller
{
    /**
     * Get current delivery person
     */
    protected function deliveryPerson()
    {
        return Auth::guard('delivery')->user();
    }

    /**
     * Dashboard
     */
    public function dashboard()
    {
        $deliveryPerson = $this->deliveryPerson();

        $stats = [
            'pending' => $deliveryPerson->assignedOrders()
                ->whereIn('delivery_status', ['assigned', 'picked_up', 'in_transit'])->count(),
            'delivered_today' => $deliveryPerson->assignedOrders()
                ->where('delivery_status', 'delivered')
                ->whereDate('delivered_at', today())->count(),
            'delivered_total' => $deliveryPerson->completedOrders()->count(),
            'failed' => $deliveryPerson->assignedOrders()
                ->where('delivery_status', 'failed')->count(),
        ];

        $pendingOrders = $deliveryPerson->assignedOrders()
            ->with(['items', 'customer'])
            ->whereIn('delivery_status', ['assigned', 'picked_up', 'in_transit'])
            ->orderBy('delivery_assigned_at', 'desc')
            ->get();

        $recentDeliveries = $deliveryPerson->assignedOrders()
            ->with(['items', 'customer'])
            ->where('delivery_status', 'delivered')
            ->orderBy('delivered_at', 'desc')
            ->limit(10)
            ->get();

        return view('ecommerce::delivery.dashboard', compact('deliveryPerson', 'stats', 'pendingOrders', 'recentDeliveries'));
    }

    /**
     * My Orders - All assigned orders
     */
    public function myOrders(Request $request)
    {
        $deliveryPerson = $this->deliveryPerson();

        $query = $deliveryPerson->assignedOrders()->with(['items', 'customer']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('delivery_status', $request->status);
        }

        // Filter by date
        if ($request->filled('date')) {
            $query->whereDate('delivery_assigned_at', $request->date);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('order_no', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%")
                  ->orWhere('customer_phone', 'like', "%{$search}%");
            });
        }

        $orders = $query->orderBy('delivery_assigned_at', 'desc')->paginate(20)->withQueryString();

        return view('ecommerce::delivery.orders', compact('orders'));
    }

    /**
     * Order Details
     */
    public function orderDetails($id)
    {
        $deliveryPerson = $this->deliveryPerson();

        $order = WebsiteOrder::with(['items.product', 'customer', 'deliveryStatusHistory'])
            ->where('delivery_person_id', $deliveryPerson->id)
            ->findOrFail($id);

        return view('ecommerce::delivery.order-details', compact('order'));
    }

    /**
     * Update delivery status
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:picked_up,in_transit,delivered,failed',
            'notes' => 'nullable|string|max:500',
            'location' => 'nullable|string|max:255',
            'proof_image' => 'nullable|image|mimes:jpg,jpeg,png|max:5120',
        ]);

        $deliveryPerson = $this->deliveryPerson();

        $order = WebsiteOrder::where('delivery_person_id', $deliveryPerson->id)
            ->whereIn('delivery_status', ['assigned', 'picked_up', 'in_transit'])
            ->findOrFail($id);

        $proofImage = null;
        if ($request->hasFile('proof_image')) {
            $proofImage = $request->file('proof_image')->store('delivery-proofs', 'public');
        }

        // Update order delivery status
        $order->delivery_status = $request->status;
        
        if ($request->status === 'picked_up') {
            $order->delivery_picked_at = now();
        }
        
        if ($request->status === 'delivered') {
            $order->delivered_at = now();
            $order->status = 'delivered';
            if ($proofImage) {
                $order->delivery_proof = $proofImage;
            }
            // Mark payment as paid if COD
            if (in_array($order->payment_method, ['cod', 'cash']) && $order->payment_status === 'pending') {
                $order->payment_status = 'paid';
                $order->paid_at = now();
            }
        }

        $order->save();

        // Create status history
        DeliveryStatusHistory::create([
            'order_id' => $order->id,
            'delivery_person_id' => $deliveryPerson->id,
            'status' => $request->status,
            'notes' => $request->notes,
            'location' => $request->location,
            'proof_image' => $proofImage,
        ]);

        $statusLabels = [
            'picked_up' => 'Order picked up',
            'in_transit' => 'Order in transit',
            'delivered' => 'Order delivered successfully',
            'failed' => 'Delivery failed',
        ];

        return back()->with('success', $statusLabels[$request->status] ?? 'Status updated');
    }

    /**
     * Toggle availability
     */
    public function toggleAvailability()
    {
        $deliveryPerson = $this->deliveryPerson();
        
        $deliveryPerson->is_available = !$deliveryPerson->is_available;
        $deliveryPerson->save();

        $message = $deliveryPerson->is_available ? 'You are now available for deliveries' : 'You are now offline';

        return back()->with('success', $message);
    }

    /**
     * Earnings page
     */
    public function earnings(Request $request)
    {
        $deliveryPerson = $this->deliveryPerson();

        $month = $request->get('month', date('Y-m'));
        $startDate = \Carbon\Carbon::parse($month)->startOfMonth();
        $endDate = \Carbon\Carbon::parse($month)->endOfMonth();

        $deliveries = $deliveryPerson->assignedOrders()
            ->where('delivery_status', 'delivered')
            ->whereBetween('delivered_at', [$startDate, $endDate])
            ->get();

        $totalDeliveries = $deliveries->count();
        $totalEarnings = $deliveries->sum('delivery_amount');

        // Daily breakdown
        $dailyEarnings = $deliveries->groupBy(function($order) {
            return $order->delivered_at->format('Y-m-d');
        })->map(function($dayOrders) {
            return [
                'count' => $dayOrders->count(),
                'amount' => $dayOrders->sum('delivery_amount'),
            ];
        });

        return view('ecommerce::delivery.earnings', compact(
            'deliveryPerson', 'month', 'totalDeliveries', 'totalEarnings', 'dailyEarnings', 'deliveries'
        ));
    }
}