@extends('ecommerce::public.shop-layout')

@section('title', 'Checkout - ' . ($settings->site_name ?? 'Store'))

@section('content')
<div class="checkout-page">
    <div class="container">
        <h1 class="page-title">Checkout</h1>
        
        @if(session('error'))
        <div class="alert alert-error">{{ session('error') }}</div>
        @endif
        
        <form action="{{ route('ecommerce.checkout.place') }}" method="POST" id="checkoutForm">
            @csrf
            <div class="checkout-layout">
                {{-- Left: Forms --}}
                <div class="checkout-forms">
                    
                    {{-- Delivery Method Selection --}}
                    <div class="form-section">
                        <h2>Delivery Method</h2>
                        
                        <div class="delivery-options">
                            <label class="delivery-option active" data-method="delivery">
                                <input type="radio" name="delivery_method" value="delivery" checked>
                                <div class="option-content">
                                    <div class="option-icon delivery-icon">
                                        <svg width="24" height="24" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" d="M8.25 18.75a1.5 1.5 0 01-3 0m3 0a1.5 1.5 0 00-3 0m3 0h6m-9 0H3.375a1.125 1.125 0 01-1.125-1.125V14.25m17.25 4.5a1.5 1.5 0 01-3 0m3 0a1.5 1.5 0 00-3 0m3 0h1.125c.621 0 1.129-.504 1.09-1.124a17.902 17.902 0 00-3.213-9.193 2.056 2.056 0 00-1.58-.86H14.25M16.5 18.75h-2.25m0-11.177v-.958c0-.568-.422-1.048-.987-1.106a48.554 48.554 0 00-10.026 0 1.106 1.106 0 00-.987 1.106v7.635m12-6.677v6.677m0 4.5v-4.5m0 0h-12"/>
                                        </svg>
                                    </div>
                                    <div class="option-info">
                                        <span class="option-title">Home Delivery</span>
                                        <span class="option-desc">Get it delivered to your doorstep</span>
                                    </div>
                                    @if(($shippingFee ?? 0) > 0)
                                    <span class="option-fee">₹{{ number_format($shippingFee, 0) }}</span>
                                    @else
                                    <span class="option-fee free">FREE</span>
                                    @endif
                                </div>
                            </label>
                            
                            @if($wsSettings->store_pickup_enabled ?? true)
                            <label class="delivery-option" data-method="pickup">
                                <input type="radio" name="delivery_method" value="pickup">
                                <div class="option-content">
                                    <div class="option-icon pickup-icon">
                                        <svg width="24" height="24" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" d="M13.5 21v-7.5a.75.75 0 01.75-.75h3a.75.75 0 01.75.75V21m-4.5 0H2.36m11.14 0H18m0 0h3.64m-1.39 0V9.349m-16.5 11.65V9.35m0 0a3.001 3.001 0 003.75-.615A2.993 2.993 0 009.75 9.75c.896 0 1.7-.393 2.25-1.016a2.993 2.993 0 002.25 1.016c.896 0 1.7-.393 2.25-1.016a3.001 3.001 0 003.75.614m-16.5 0a3.004 3.004 0 01-.621-4.72L4.318 3.44A1.5 1.5 0 015.378 3h13.243a1.5 1.5 0 011.06.44l1.19 1.189a3 3 0 01-.621 4.72m-13.5 8.65h3.75a.75.75 0 00.75-.75V13.5a.75.75 0 00-.75-.75H6.75a.75.75 0 00-.75.75v3.75c0 .415.336.75.75.75z"/>
                                        </svg>
                                    </div>
                                    <div class="option-info">
                                        <span class="option-title">Pickup from Store</span>
                                        <span class="option-desc">Collect from our store location</span>
                                    </div>
                                    <span class="option-fee free">FREE</span>
                                </div>
                            </label>
                            @endif
                        </div>
                        
                        {{-- Store Pickup Info (hidden by default) --}}
                        <div id="pickupInfo" class="pickup-info" style="display: none;">
                            <div class="pickup-store-card">
                                <div class="store-icon">
                                    <svg width="24" height="24" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" d="M15 10.5a3 3 0 11-6 0 3 3 0 016 0z"/>
                                        <path stroke-linecap="round" stroke-linejoin="round" d="M19.5 10.5c0 7.142-7.5 11.25-7.5 11.25S4.5 17.642 4.5 10.5a7.5 7.5 0 1115 0z"/>
                                    </svg>
                                </div>
                                <div class="store-details">
                                    <strong>{{ $settings->site_name ?? 'Our Store' }}</strong>
                                    <p>{{ $wsSettings->store_address ?? 'Store Address' }}</p>
                                    <p>{{ $wsSettings->store_city ?? '' }}{{ $wsSettings->store_state ? ', ' . $wsSettings->store_state : '' }} {{ $wsSettings->store_pincode ?? '' }}</p>
                                    @if($wsSettings->contact_phone ?? false)
                                    <p class="store-phone">📞 {{ $wsSettings->contact_phone }}</p>
                                    @endif
                                </div>
                                @if($wsSettings->store_google_maps_link ?? false)
                                <a href="{{ $wsSettings->store_google_maps_link }}" target="_blank" class="directions-btn">
                                    <svg width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" d="M13.5 6H5.25A2.25 2.25 0 003 8.25v10.5A2.25 2.25 0 005.25 21h10.5A2.25 2.25 0 0018 18.75V10.5m-10.5 6L21 3m0 0h-5.25M21 3v5.25"/>
                                    </svg>
                                    Directions
                                </a>
                                @endif
                            </div>
                            <p class="pickup-note">📦 Your order will be ready for pickup within {{ $wsSettings->pickup_ready_time ?? '2-4 hours' }} after confirmation.</p>
                            @if($wsSettings->pickup_instructions ?? false)
                            <p class="pickup-instructions">ℹ️ {{ $wsSettings->pickup_instructions }}</p>
                            @endif
                        </div>
                    </div>
                    
                    {{-- Contact Information (always shown) --}}
                    <div class="form-section">
                        <h2>Contact Information</h2>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Full Name <span class="required">*</span></label>
                                <input type="text" name="customer_name" class="form-input" required
                                    value="{{ old('customer_name', $shippingAddress['name'] ?? $user->name ?? '') }}">
                                @error('customer_name')<span class="error">{{ $message }}</span>@enderror
                            </div>
                            <div class="form-group">
                                <label>Phone Number <span class="required">*</span></label>
                                <input type="tel" name="customer_phone" class="form-input" required
                                    value="{{ old('customer_phone', $shippingAddress['phone'] ?? '') }}">
                                @error('customer_phone')<span class="error">{{ $message }}</span>@enderror
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Email (for order updates)</label>
                            <input type="email" name="customer_email" class="form-input"
                                value="{{ old('customer_email', $customer->email ?? $user->email ?? '') }}">
                        </div>
                    </div>
                    
                    {{-- Shipping Address (only for delivery) --}}
                    <div class="form-section" id="shippingSection">
                        <h2>Shipping Address</h2>
                        
                        <div class="form-group">
                            <label>Address <span class="required">*</span></label>
                            <textarea name="shipping_address" id="shipping_address" class="form-input" rows="2" required
                                placeholder="House no, Building, Street, Locality">{{ old('shipping_address', $shippingAddress['address'] ?? '') }}</textarea>
                            @error('shipping_address')<span class="error">{{ $message }}</span>@enderror
                        </div>
                        
                        <div class="form-row form-row-3">
                            <div class="form-group">
                                <label>City <span class="required">*</span></label>
                                <input type="text" name="shipping_city" id="shipping_city" class="form-input" required
                                    value="{{ old('shipping_city', $shippingAddress['city'] ?? '') }}">
                            </div>
                            <div class="form-group">
                                <label>State <span class="required">*</span></label>
                                <input type="text" name="shipping_state" id="shipping_state" class="form-input" required
                                    value="{{ old('shipping_state', $shippingAddress['state'] ?? '') }}">
                            </div>
                            <div class="form-group">
                                <label>PIN Code <span class="required">*</span></label>
                                <input type="text" name="shipping_pincode" id="shipping_pincode" class="form-input" required
                                    value="{{ old('shipping_pincode', $shippingAddress['pincode'] ?? '') }}" maxlength="6" pattern="[0-9]{6}">
                                <span class="pincode-status" id="pincodeStatus"></span>
                            </div>
                        </div>
                        
                        {{-- Location validation error message --}}
                        <div id="locationError" class="location-error" style="display: none;">
                            <svg width="20" height="20" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                            </svg>
                            <div>
                                <strong>Delivery Not Available</strong>
                                <p id="locationErrorMsg">Sorry, we don't deliver to this location.</p>
                                <p class="pickup-suggestion">💡 You can choose <a href="#" onclick="selectPickup(); return false;"><strong>Pickup from Store</strong></a> instead!</p>
                            </div>
                        </div>
                        
                        {{-- Location validation success message --}}
                        <div id="locationSuccess" class="location-success" style="display: none;">
                            <svg width="20" height="20" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7"/>
                            </svg>
                            <div>
                                <strong>Delivery Available!</strong>
                                <p>Great news! We deliver to your location.</p>
                            </div>
                        </div>
                    </div>
                    
                    {{-- Payment Method --}}
                    <div class="form-section">
                        <h2>Payment Method</h2>
                        
                        <div class="payment-options">
                            @php 
                                $firstActive = true; 
                                $codDisabledByAmount = $wsSettings->cod_max_amount > 0 && $grandTotal > $wsSettings->cod_max_amount;
                            @endphp
                            
                            {{-- Cash on Delivery / Pay at Store --}}
                            @if($wsSettings->cod_enabled ?? true)
                                @if($codDisabledByAmount)
                                <div class="payment-option disabled">
                                    <div class="option-content">
                                        <div class="option-icon">💵</div>
                                        <div class="option-info">
                                            <span class="option-title cod-title">Cash on Delivery</span>
                                            <span class="option-desc">Not available for orders above ₹{{ number_format($wsSettings->cod_max_amount, 0) }}</span>
                                        </div>
                                    </div>
                                </div>
                                @else
                                <label class="payment-option {{ $firstActive ? 'active' : '' }}">
                                    <input type="radio" name="payment_method" value="cod" {{ $firstActive ? 'checked' : '' }}>
                                    <div class="option-content">
                                        <div class="option-icon">💵</div>
                                        <div class="option-info">
                                            <span class="option-title cod-title">Cash on Delivery</span>
                                            <span class="option-desc cod-desc">Pay when you receive your order</span>
                                        </div>
                                        @if(($wsSettings->cod_fee ?? 0) > 0)
                                        <span class="option-fee cod-fee-badge">+₹{{ number_format($wsSettings->cod_fee, 0) }}</span>
                                        @endif
                                    </div>
                                </label>
                                @php $firstActive = false; @endphp
                                @endif
                            @endif

                            {{-- Online Payment --}}
                            @if($wsSettings->online_payment_enabled ?? false)
                            <label class="payment-option {{ $firstActive ? 'active' : '' }}">
                                <input type="radio" name="payment_method" value="online" {{ $firstActive ? 'checked' : '' }}>
                                <div class="option-content">
                                    <div class="option-icon">💳</div>
                                    <div class="option-info">
                                        <span class="option-title">{{ $wsSettings->online_payment_label ?? 'Pay Online (UPI/Card/NetBanking)' }}</span>
                                        <span class="option-desc">Secure payment via payment gateway</span>
                                    </div>
                                </div>
                            </label>
                            @php $firstActive = false; @endphp
                            @endif

                            {{-- No payment methods available --}}
                            @if(!($wsSettings->cod_enabled ?? true) && !($wsSettings->online_payment_enabled ?? false))
                            <div class="payment-option disabled">
                                <div class="option-content">
                                    <div class="option-icon">⚠️</div>
                                    <div class="option-info">
                                        <span class="option-title">No Payment Methods Available</span>
                                        <span class="option-desc">Please contact store for assistance</span>
                                    </div>
                                </div>
                            </div>
                            @endif
                        </div>
                        
                        @if($minOrderAmount > 0 && $subtotal < $minOrderAmount)
                        <div class="min-order-warning">
                            <svg width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/></svg>
                            Minimum order amount is ₹{{ number_format($minOrderAmount, 0) }}. Add ₹{{ number_format($minOrderAmount - $subtotal, 0) }} more.
                        </div>
                        @endif
                    </div>
                    
                    {{-- Order Notes --}}
                    <div class="form-section">
                        <h2>Order Notes (Optional)</h2>
                        <textarea name="customer_notes" class="form-input" rows="2" 
                            placeholder="Any special instructions...">{{ old('customer_notes') }}</textarea>
                    </div>
                </div>
                
                {{-- Right: Order Summary --}}
                <div class="order-summary">
                    <h2>Order Summary</h2>
                    
                    <div class="summary-items">
                        @foreach($cartItems as $item)
                        <div class="summary-item">
                            <div class="item-img">
                                @if($item['product']->getPrimaryImageUrl())
                                <img src="{{ $item['product']->getPrimaryImageUrl() }}" alt="">
                                @endif
                            </div>
                            <div class="item-info">
                                <span class="item-name">{{ $item['product']->name }}</span>
                                @if($item['variation_name'])
                                <span class="item-variant">{{ $item['variation_name'] }}</span>
                                @endif
                                <span class="item-qty">Qty: {{ (int)$item['qty'] }}</span>
                            </div>
                            <div class="item-price">₹{{ number_format($item['total'], 0) }}</div>
                        </div>
                        @endforeach
                    </div>
                    
                    <div class="summary-totals">
                        <div class="sum-row">
                            <span>Subtotal</span>
                            <span>₹{{ number_format($subtotal, 0) }}</span>
                        </div>
                        <div class="sum-row" id="shippingRow">
                            <span id="shippingLabel">Shipping</span>
                            @if($shippingFee > 0)
                            <span id="shippingAmount">₹{{ number_format($shippingFee, 0) }}</span>
                            @else
                            <span id="shippingAmount" class="free">FREE</span>
                            @endif
                        </div>
                        @if($codFee > 0 && ($wsSettings->cod_enabled ?? true))
                        <div class="sum-row cod-row" id="codFeeRow">
                            <span>COD Fee</span>
                            <span>₹{{ number_format($codFee, 0) }}</span>
                        </div>
                        @endif
                        <div class="sum-row total">
                            <span>Total</span>
                            <span id="grandTotal">₹{{ number_format($grandTotal + (($wsSettings->cod_enabled ?? true) ? $codFee : 0), 0) }}</span>
                        </div>
                    </div>
                    
                    <div class="delivery-info" id="deliveryTimeInfo">
                        @if($wsSettings->delivery_days)
                        🚚 Estimated delivery: {{ $wsSettings->delivery_days }}
                        @endif
                    </div>
                    
                    <button type="submit" class="place-order-btn" id="placeOrderBtn">
                        Place Order
                    </button>
                    
                    <a href="{{ route('ecommerce.cart') }}" class="back-link">← Back to Cart</a>
                </div>
            </div>
        </form>
    </div>
</div>

<style>
.checkout-page { padding: 32px 0 60px; background: #f8fafc; min-height: 80vh; }
.page-title { font-size: 24px; font-weight: 700; color: #1e293b; margin-bottom: 24px; }
.alert-error { background: #fef2f2; border: 1px solid #fecaca; color: #991b1b; padding: 12px 16px; border-radius: 8px; margin-bottom: 20px; }
.checkout-layout { display: grid; grid-template-columns: 1fr 400px; gap: 32px; align-items: start; }

.form-section { background: #fff; border-radius: 12px; padding: 24px; margin-bottom: 20px; box-shadow: 0 1px 3px rgba(0,0,0,.05); }
.form-section h2 { font-size: 16px; font-weight: 700; color: #1e293b; margin-bottom: 20px; padding-bottom: 12px; border-bottom: 1px solid #f1f5f9; }

.form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; }
.form-row-3 { grid-template-columns: 1fr 1fr 1fr; }
.form-group { margin-bottom: 16px; }
.form-group label { display: block; font-size: 13px; font-weight: 600; color: #475569; margin-bottom: 6px; }
.form-group label .required { color: #ef4444; }
.form-input { width: 100%; padding: 12px 14px; border: 1px solid #e2e8f0; border-radius: 8px; font-size: 14px; transition: all .2s; }
.form-input:focus { outline: none; border-color: #3b82f6; box-shadow: 0 0 0 3px rgba(59,130,246,.1); }
.form-group .error { display: block; font-size: 12px; color: #ef4444; margin-top: 4px; }

/* Delivery Method Options */
.delivery-options { display: flex; gap: 12px; margin-bottom: 16px; }
.delivery-option { flex: 1; border: 2px solid #e2e8f0; border-radius: 12px; cursor: pointer; transition: all .2s; }
.delivery-option:hover { border-color: #cbd5e1; }
.delivery-option.active { border-color: #0891b2; background: #ecfeff; }
.delivery-option input { display: none; }
.delivery-option .option-content { display: flex; align-items: center; gap: 12px; padding: 16px; }
.delivery-option .option-icon { width: 48px; height: 48px; display: flex; align-items: center; justify-content: center; border-radius: 12px; flex-shrink: 0; }
.delivery-icon { background: linear-gradient(135deg, #0891b2, #0e7490); color: #fff; }
.pickup-icon { background: linear-gradient(135deg, #8b5cf6, #7c3aed); color: #fff; }
.delivery-option .option-info { flex: 1; min-width: 0; }
.delivery-option .option-title { display: block; font-size: 15px; font-weight: 600; color: #1e293b; }
.delivery-option .option-desc { display: block; font-size: 12px; color: #64748b; margin-top: 2px; }
.delivery-option .option-fee { font-size: 13px; font-weight: 600; color: #0891b2; background: #ecfeff; padding: 4px 10px; border-radius: 6px; white-space: nowrap; }
.delivery-option .option-fee.free { color: #10b981; background: #ecfdf5; }

/* Pickup Info Card */
.pickup-info { margin-top: 16px; }
.pickup-store-card { display: flex; align-items: flex-start; gap: 16px; background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 12px; padding: 20px; }
.pickup-store-card .store-icon { width: 48px; height: 48px; background: linear-gradient(135deg, #8b5cf6, #7c3aed); border-radius: 12px; display: flex; align-items: center; justify-content: center; color: #fff; flex-shrink: 0; }
.pickup-store-card .store-details { flex: 1; }
.pickup-store-card .store-details strong { display: block; font-size: 15px; color: #1e293b; margin-bottom: 4px; }
.pickup-store-card .store-details p { font-size: 13px; color: #64748b; margin: 2px 0; }
.pickup-store-card .store-phone { color: #0891b2; font-weight: 500; margin-top: 8px !important; }
.directions-btn { display: flex; align-items: center; gap: 6px; padding: 10px 16px; background: #0891b2; color: #fff; border-radius: 8px; font-size: 13px; font-weight: 600; text-decoration: none; white-space: nowrap; }
.directions-btn:hover { background: #0e7490; }
.pickup-note { margin-top: 12px; padding: 12px 16px; background: #fef3c7; border-radius: 8px; font-size: 13px; color: #92400e; }
.pickup-instructions { margin-top: 8px; padding: 12px 16px; background: #eff6ff; border-radius: 8px; font-size: 13px; color: #1e40af; }

/* PIN Code Status */
.pincode-status { display: block; font-size: 12px; margin-top: 4px; }
.pincode-status.checking { color: #3b82f6; }
.pincode-status.valid { color: #10b981; }
.pincode-status.invalid { color: #ef4444; }

/* Location Error/Success Messages */
.location-error, .location-success { display: flex; align-items: flex-start; gap: 12px; padding: 16px; border-radius: 10px; margin-top: 16px; }
.location-error { background: #fef2f2; border: 1px solid #fecaca; color: #991b1b; }
.location-error svg { color: #dc2626; flex-shrink: 0; margin-top: 2px; }
.location-error strong { display: block; font-size: 14px; margin-bottom: 4px; }
.location-error p { font-size: 13px; margin: 0; color: #b91c1c; }
.location-error .pickup-suggestion { margin-top: 8px; padding-top: 8px; border-top: 1px solid #fecaca; color: #92400e; }
.location-error .pickup-suggestion a { color: #7c3aed; text-decoration: underline; }

.location-success { background: #f0fdf4; border: 1px solid #bbf7d0; color: #166534; }
.location-success svg { color: #16a34a; flex-shrink: 0; margin-top: 2px; }
.location-success strong { display: block; font-size: 14px; margin-bottom: 4px; }
.location-success p { font-size: 13px; margin: 0; color: #15803d; }

.payment-options { display: flex; flex-direction: column; gap: 12px; }
.payment-option { border: 2px solid #e2e8f0; border-radius: 12px; cursor: pointer; transition: all .2s; }
.payment-option:hover { border-color: #cbd5e1; }
.payment-option.active { border-color: #3b82f6; background: #eff6ff; }
.payment-option.disabled { opacity: .5; cursor: not-allowed; background: #f8fafc; }
.payment-option input { display: none; }
.payment-option .option-content { display: flex; align-items: center; gap: 16px; padding: 16px; }
.payment-option .option-icon { width: 40px; height: 40px; display: flex; align-items: center; justify-content: center; background: linear-gradient(135deg, #667eea, #764ba2); border-radius: 10px; color: #fff; font-size: 20px; }
.payment-option .option-info { flex: 1; }
.payment-option .option-title { display: block; font-size: 15px; font-weight: 600; color: #1e293b; }
.payment-option .option-desc { display: block; font-size: 12px; color: #64748b; margin-top: 2px; }
.payment-option .option-fee { font-size: 13px; font-weight: 600; color: #f59e0b; background: #fef3c7; padding: 4px 10px; border-radius: 6px; }

.min-order-warning { display: flex; align-items: center; gap: 10px; margin-top: 16px; padding: 12px 16px; background: #fef3c7; border-radius: 8px; color: #92400e; font-size: 13px; font-weight: 500; }
.min-order-warning svg { flex-shrink: 0; }

.order-summary { background: #fff; border-radius: 12px; padding: 24px; position: sticky; top: 24px; box-shadow: 0 1px 3px rgba(0,0,0,.05); }
.order-summary h2 { font-size: 16px; font-weight: 700; color: #1e293b; margin-bottom: 20px; }

.summary-items { border-bottom: 1px solid #f1f5f9; padding-bottom: 16px; margin-bottom: 16px; max-height: 300px; overflow-y: auto; }
.summary-item { display: flex; gap: 12px; padding: 10px 0; }
.item-img { width: 50px; height: 50px; border-radius: 8px; overflow: hidden; background: #f8fafc; flex-shrink: 0; }
.item-img img { width: 100%; height: 100%; object-fit: contain; }
.item-info { flex: 1; min-width: 0; }
.item-name { display: block; font-size: 13px; font-weight: 500; color: #1e293b; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
.item-variant { display: block; font-size: 11px; color: #64748b; }
.item-qty { display: block; font-size: 12px; color: #94a3b8; }
.item-price { font-size: 14px; font-weight: 600; color: #1e293b; }

.summary-totals { padding: 16px 0; }
.sum-row { display: flex; justify-content: space-between; padding: 8px 0; font-size: 14px; color: #64748b; }
.sum-row .free { color: #10b981; font-weight: 500; }
.sum-row.total { font-size: 18px; font-weight: 700; color: #1e293b; border-top: 1px solid #e2e8f0; margin-top: 8px; padding-top: 16px; }

.delivery-info { font-size: 13px; color: #64748b; text-align: center; margin: 16px 0; }

.place-order-btn { width: 100%; padding: 16px; background: linear-gradient(135deg, #10b981, #059669); color: #fff; border: none; border-radius: 12px; font-size: 16px; font-weight: 700; cursor: pointer; margin-top: 8px; transition: all .2s; box-shadow: 0 4px 12px rgba(16,185,129,.3); }
.place-order-btn:hover { transform: translateY(-1px); box-shadow: 0 6px 16px rgba(16,185,129,.4); }
.place-order-btn:disabled { background: #94a3b8; cursor: not-allowed; transform: none; box-shadow: none; }

.back-link { display: block; text-align: center; padding: 12px; color: #64748b; font-size: 14px; margin-top: 8px; text-decoration: none; }
.back-link:hover { color: #3b82f6; }

@media (max-width: 900px) {
    .checkout-layout { grid-template-columns: 1fr; }
    .order-summary { position: static; order: -1; }
    .delivery-options { flex-direction: column; }
    .pickup-store-card { flex-direction: column; text-align: center; }
    .directions-btn { width: 100%; justify-content: center; }
}
@media (max-width: 640px) {
    .form-row, .form-row-3 { grid-template-columns: 1fr; }
}
</style>

<script>
// ==================== DELIVERY METHOD TOGGLE ====================
const shippingFee = {{ $shippingFee ?? 0 }};
const subtotal = {{ $subtotal ?? 0 }};
const codFee = {{ $codFee ?? 0 }};
let isPickup = false;

function selectPickup() {
    document.querySelector('.delivery-option[data-method="pickup"] input').checked = true;
    document.querySelectorAll('.delivery-option').forEach(o => o.classList.remove('active'));
    document.querySelector('.delivery-option[data-method="pickup"]').classList.add('active');
    toggleDeliveryMethod('pickup');
}

function toggleDeliveryMethod(method) {
    isPickup = (method === 'pickup');
    const shippingSection = document.getElementById('shippingSection');
    const pickupInfo = document.getElementById('pickupInfo');
    const shippingAmountEl = document.getElementById('shippingAmount');
    const shippingLabelEl = document.getElementById('shippingLabel');
    const deliveryTimeInfo = document.getElementById('deliveryTimeInfo');
    const codFeeRow = document.getElementById('codFeeRow');
    
    if (isPickup) {
        shippingSection.style.display = 'none';
        pickupInfo.style.display = 'block';
        
        // Remove required from shipping fields
        document.getElementById('shipping_address').removeAttribute('required');
        document.getElementById('shipping_city').removeAttribute('required');
        document.getElementById('shipping_state').removeAttribute('required');
        document.getElementById('shipping_pincode').removeAttribute('required');
        
        // Update shipping row
        shippingLabelEl.textContent = 'Pickup';
        shippingAmountEl.textContent = 'FREE';
        shippingAmountEl.className = 'free';
        
        // Hide COD fee for pickup
        if (codFeeRow) codFeeRow.style.display = 'none';
        
        // Update delivery time info
        deliveryTimeInfo.innerHTML = '🏪 Ready for pickup in {{ $wsSettings->pickup_ready_time ?? "2-4 hours" }}';
        
        // Update COD label to "Pay at Store"
        document.querySelectorAll('.cod-title').forEach(el => el.textContent = 'Pay at Store');
        document.querySelectorAll('.cod-desc').forEach(el => el.textContent = 'Pay when you collect your order');
        document.querySelectorAll('.cod-fee-badge').forEach(el => el.style.display = 'none');
        
        // Hide location messages and enable button
        hideLocationMessages();
        document.getElementById('placeOrderBtn').disabled = false;
        
    } else {
        shippingSection.style.display = 'block';
        pickupInfo.style.display = 'none';
        
        // Add required back
        document.getElementById('shipping_address').setAttribute('required', 'required');
        document.getElementById('shipping_city').setAttribute('required', 'required');
        document.getElementById('shipping_state').setAttribute('required', 'required');
        document.getElementById('shipping_pincode').setAttribute('required', 'required');
        
        // Restore shipping row
        shippingLabelEl.textContent = 'Shipping';
        if (shippingFee > 0) {
            shippingAmountEl.textContent = '₹' + shippingFee.toLocaleString('en-IN');
            shippingAmountEl.className = '';
        } else {
            shippingAmountEl.textContent = 'FREE';
            shippingAmountEl.className = 'free';
        }
        
        // Show COD fee row
        if (codFeeRow) codFeeRow.style.display = '';
        
        // Restore delivery time info
        deliveryTimeInfo.innerHTML = '🚚 Estimated delivery: {{ $wsSettings->delivery_days ?? "3-5 business days" }}';
        
        // Restore COD label
        document.querySelectorAll('.cod-title').forEach(el => el.textContent = 'Cash on Delivery');
        document.querySelectorAll('.cod-desc').forEach(el => el.textContent = 'Pay when you receive your order');
        document.querySelectorAll('.cod-fee-badge').forEach(el => el.style.display = '');
        
        // Re-check location if pincode exists
        const pincode = document.getElementById('shipping_pincode').value.trim();
        if (pincode.length === 6) {
            checkDeliveryAvailability();
        }
    }
    
    updateTotal();
}

function updateTotal() {
    let total = subtotal;
    
    if (!isPickup) {
        total += shippingFee;
        
        // Add COD fee if COD selected
        const codSelected = document.querySelector('input[name="payment_method"][value="cod"]:checked');
        if (codSelected && codFee > 0) {
            total += codFee;
        }
    }
    
    document.getElementById('grandTotal').textContent = '₹' + total.toLocaleString('en-IN');
}

// Delivery method toggle
document.querySelectorAll('.delivery-option').forEach(opt => {
    opt.addEventListener('click', () => {
        document.querySelectorAll('.delivery-option').forEach(o => o.classList.remove('active'));
        opt.classList.add('active');
        const radio = opt.querySelector('input[type="radio"]');
        if (radio) radio.checked = true;
        toggleDeliveryMethod(opt.dataset.method);
    });
});

// Payment method toggle
document.querySelectorAll('.payment-option:not(.disabled)').forEach(opt => {
    opt.addEventListener('click', () => {
        document.querySelectorAll('.payment-option').forEach(o => o.classList.remove('active'));
        opt.classList.add('active');
        const radio = opt.querySelector('input[type="radio"]');
        if (radio) radio.checked = true;
        updateTotal();
    });
});

// ==================== LOCATION RESTRICTION VALIDATION ====================
@if($wsSettings->location_restriction_enabled ?? false)

const STORE_LOCATION = {
    lat: {{ $wsSettings->store_latitude ?? 12.9996206 }},
    lng: {{ $wsSettings->store_longitude ?? 77.5436373 }}
};
const SERVICE_RADIUS_KM = {{ $wsSettings->service_radius_km ?? 5 }};

let deliveryAvailable = null;
let isCheckingLocation = false;

function calculateDistance(lat1, lon1, lat2, lon2) {
    const R = 6371;
    const dLat = (lat2 - lat1) * Math.PI / 180;
    const dLon = (lon2 - lon1) * Math.PI / 180;
    const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
              Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
              Math.sin(dLon/2) * Math.sin(dLon/2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    return R * c;
}

function showLocationError(distance) {
    document.getElementById('locationError').style.display = 'flex';
    document.getElementById('locationErrorMsg').textContent = `Sorry, this location is ${distance.toFixed(1)} km away. We deliver within ${SERVICE_RADIUS_KM} km of our store.`;
    document.getElementById('locationSuccess').style.display = 'none';
    document.getElementById('placeOrderBtn').disabled = true;
    deliveryAvailable = false;
}

function showLocationSuccess() {
    document.getElementById('locationSuccess').style.display = 'flex';
    document.getElementById('locationError').style.display = 'none';
    document.getElementById('placeOrderBtn').disabled = false;
    deliveryAvailable = true;
}

function hideLocationMessages() {
    document.getElementById('locationError').style.display = 'none';
    document.getElementById('locationSuccess').style.display = 'none';
    document.getElementById('placeOrderBtn').disabled = false;
    deliveryAvailable = null;
}

function updatePincodeStatus(status, message) {
    const statusEl = document.getElementById('pincodeStatus');
    statusEl.textContent = message;
    statusEl.className = 'pincode-status ' + status;
}

async function checkDeliveryAvailability() {
    if (isPickup) return;
    
    const pincode = document.getElementById('shipping_pincode').value.trim();
    const city = document.getElementById('shipping_city').value.trim();
    const state = document.getElementById('shipping_state').value.trim();
    
    if (pincode.length !== 6) {
        hideLocationMessages();
        deliveryAvailable = null;
        return;
    }
    
    isCheckingLocation = true;
    updatePincodeStatus('checking', '⏳ Checking delivery availability...');
    
    try {
        const searchQuery = encodeURIComponent(`${pincode}, ${city}, ${state}, India`);
        const response = await fetch(`https://nominatim.openstreetmap.org/search?q=${searchQuery}&format=json&limit=1&countrycodes=in`);
        const data = await response.json();
        
        if (data && data.length > 0) {
            const location = data[0];
            const userLat = parseFloat(location.lat);
            const userLng = parseFloat(location.lon);
            
            const distance = calculateDistance(userLat, userLng, STORE_LOCATION.lat, STORE_LOCATION.lng);
            
            console.log('PIN Code Location:', userLat, userLng);
            console.log('Distance from store:', distance.toFixed(2), 'km');
            
            if (distance > SERVICE_RADIUS_KM) {
                updatePincodeStatus('invalid', '❌ Outside delivery area');
                showLocationError(distance);
            } else {
                updatePincodeStatus('valid', '✓ Delivery available');
                showLocationSuccess();
            }
        } else {
            updatePincodeStatus('', '');
            hideLocationMessages();
        }
    } catch (error) {
        console.error('Geocoding error:', error);
        updatePincodeStatus('', '');
        hideLocationMessages();
    }
    
    isCheckingLocation = false;
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

const debouncedCheck = debounce(checkDeliveryAvailability, 800);

document.getElementById('shipping_pincode').addEventListener('input', function(e) {
    this.value = this.value.replace(/[^0-9]/g, '');
    
    if (this.value.length === 6) {
        debouncedCheck();
    } else {
        hideLocationMessages();
        updatePincodeStatus('', '');
        deliveryAvailable = null;
    }
});

document.getElementById('shipping_city').addEventListener('blur', function() {
    if (document.getElementById('shipping_pincode').value.length === 6 && !isPickup) {
        debouncedCheck();
    }
});

document.getElementById('shipping_state').addEventListener('blur', function() {
    if (document.getElementById('shipping_pincode').value.length === 6 && !isPickup) {
        debouncedCheck();
    }
});

document.getElementById('checkoutForm').addEventListener('submit', function(e) {
    if (isPickup) return true;
    
    if (isCheckingLocation) {
        e.preventDefault();
        alert('Please wait while we verify your delivery location...');
        return false;
    }
    
    if (deliveryAvailable === false) {
        e.preventDefault();
        alert('Sorry, we cannot deliver to your location. Please check your PIN code or choose Pickup from Store.');
        return false;
    }
    
    return true;
});

document.addEventListener('DOMContentLoaded', function() {
    const pincode = document.getElementById('shipping_pincode').value.trim();
    if (pincode.length === 6 && !isPickup) {
        setTimeout(checkDeliveryAvailability, 500);
    }
});

@else
function hideLocationMessages() {}
@endif
</script>
@endsection