<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>@yield('title', 'Shop')</title>
    <meta name="csrf-token" content="{{ csrf_token() }}">
    @if($settings && method_exists($settings, 'getFaviconUrl') && $settings->getFaviconUrl())
    <link rel="icon" href="{{ $settings->getFaviconUrl() }}">
    @endif
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">
    @livewireStyles
    <style>
        :root {
            --primary: #0891b2;
            --primary-dark: #0e7490;
            --secondary: #f97316;
            --dark: #0f172a;
            --text: #334155;
            --text-light: #64748b;
            --border: #e2e8f0;
            --bg: #f8fafc;
        }
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Poppins', sans-serif; background: var(--bg); color: var(--text); }
        a { text-decoration: none; color: inherit; }
        .container { max-width: 1280px; margin: 0 auto; padding: 0 20px; }
        
        /* Header */
        .header { background: #fff; box-shadow: 0 1px 3px rgba(0,0,0,.08); position: sticky; top: 0; z-index: 100; }
        .header-main { display: flex; align-items: center; height: 70px; gap: 40px; }
        
        .logo { font-size: 24px; font-weight: 700; color: var(--dark); display: flex; align-items: center; gap: 10px; }
        .logo img { height: 45px; }
        .logo-icon { width: 40px; height: 40px; background: var(--primary); border-radius: 10px; display: flex; align-items: center; justify-content: center; color: #fff; }
        
        /* Search */
        .search-wrap { flex: 1; max-width: 520px; position: relative; }
        .search-box { display: flex; border: 2px solid var(--border); border-radius: 10px; overflow: hidden; transition: border-color .2s; }
        .search-box:focus-within { border-color: var(--primary); }
        .search-box input { flex: 1; padding: 12px 16px; border: none; font-size: 14px; outline: none; }
        .search-box button { padding: 0 18px; background: var(--primary); border: none; color: #fff; cursor: pointer; }
        .search-box button:hover { background: var(--primary-dark); }
        
        /* Nav */
        .header-nav { display: flex; align-items: center; gap: 8px; margin-left: auto; }
        .nav-item { display: flex; flex-direction: column; align-items: center; padding: 8px 14px; border-radius: 8px; font-size: 12px; color: var(--text); transition: all .2s; position: relative; }
        .nav-item:hover { background: var(--bg); color: var(--primary); }
        .nav-item svg { width: 22px; height: 22px; margin-bottom: 2px; }
        .cart-badge { position: absolute; top: 2px; right: 8px; background: var(--secondary); color: #fff; font-size: 10px; font-weight: 600; padding: 2px 6px; border-radius: 10px; }
        
        /* Hero Banner */
        .hero { background: linear-gradient(135deg, var(--primary) 0%, #0e7490 50%, #164e63 100%); color: #fff; padding: 50px 0; margin-bottom: 30px; }
        .hero-content { display: flex; align-items: center; justify-content: space-between; }
        .hero-text h1 { font-size: 36px; font-weight: 700; margin-bottom: 12px; }
        .hero-text p { font-size: 16px; opacity: .9; margin-bottom: 24px; max-width: 400px; }
        .hero-btn { display: inline-block; padding: 14px 32px; background: #fff; color: var(--primary); font-weight: 600; border-radius: 8px; transition: transform .2s; }
        .hero-btn:hover { transform: translateY(-2px); }
        .hero-stats { display: flex; gap: 40px; }
        .stat { text-align: center; }
        .stat-num { font-size: 32px; font-weight: 700; }
        .stat-label { font-size: 13px; opacity: .8; }
        
        /* Features Bar */
        .features { background: #fff; padding: 20px 0; margin-bottom: 30px; border-bottom: 1px solid var(--border); }
        .features-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; }
        .feature { display: flex; align-items: center; gap: 12px; }
        .feature-icon { width: 45px; height: 45px; background: linear-gradient(135deg, #ecfeff 0%, #cffafe 100%); border-radius: 12px; display: flex; align-items: center; justify-content: center; color: var(--primary); flex-shrink: 0; }
        .feature-info { min-width: 0; }
        .feature-text { font-size: 14px; font-weight: 600; color: var(--dark); }
        .feature-sub { font-size: 12px; color: var(--text-light); }
        
        /* Main */
        main { min-height: 60vh; padding-bottom: 60px; }
        
        /* Footer */
        .footer { background: var(--dark); color: #fff; padding: 60px 0 30px; }
        .footer-grid { display: grid; grid-template-columns: 2fr repeat(3, 1fr); gap: 50px; margin-bottom: 40px; }
        .footer-brand { font-size: 22px; font-weight: 700; margin-bottom: 16px; }
        .footer-desc { font-size: 14px; color: #94a3b8; line-height: 1.7; }
        .footer-title { font-size: 15px; font-weight: 600; margin-bottom: 20px; }
        .footer-links { list-style: none; }
        .footer-links li { margin-bottom: 12px; }
        .footer-links a { color: #94a3b8; font-size: 14px; transition: color .2s; }
        .footer-links a:hover { color: #fff; }
        .footer-bottom { text-align: center; padding-top: 30px; border-top: 1px solid #334155; color: #64748b; font-size: 13px; }
        
        /* Toast */
        .toast { position: fixed; bottom: 24px; right: 24px; padding: 16px 24px; background: var(--dark); color: #fff; border-radius: 10px; font-size: 14px; display: none; z-index: 9999; box-shadow: 0 10px 40px rgba(0,0,0,.2); }
        .toast.show { display: flex; align-items: center; gap: 10px; animation: slideIn .3s ease; }
        .toast.success { background: #059669; }
        .toast.error { background: #dc2626; }
        .toast.info { background: var(--primary); }
        @keyframes slideIn { from { transform: translateX(100%); opacity: 0; } to { transform: translateX(0); opacity: 1; } }
        
        /* ==================== CATEGORY SIDEBAR ==================== */
        .sidebar-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 998;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        .sidebar-overlay.active {
            opacity: 1;
            visibility: visible;
        }

        .category-sidebar {
            position: fixed;
            top: 0;
            left: -320px;
            width: 320px;
            height: 100%;
            background: #fff;
            z-index: 999;
            transition: left 0.3s ease;
            display: flex;
            flex-direction: column;
            box-shadow: 4px 0 20px rgba(0, 0, 0, 0.15);
        }
        .category-sidebar.active {
            left: 0;
        }

        .sidebar-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 20px;
            border-bottom: 1px solid var(--border);
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            color: #fff;
        }
        .sidebar-header h3 {
            font-size: 18px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .sidebar-header h3 svg {
            width: 22px;
            height: 22px;
        }
        .sidebar-close {
            width: 36px;
            height: 36px;
            border: none;
            background: rgba(255,255,255,0.2);
            color: #fff;
            border-radius: 8px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background 0.2s;
        }
        .sidebar-close:hover {
            background: rgba(255,255,255,0.3);
        }
        .sidebar-close svg {
            width: 20px;
            height: 20px;
        }

        .sidebar-search {
            padding: 16px 20px;
            border-bottom: 1px solid var(--border);
            background: #f8fafc;
        }
        .sidebar-search-box {
            display: flex;
            align-items: center;
            background: #fff;
            border: 2px solid var(--border);
            border-radius: 10px;
            overflow: hidden;
            transition: border-color 0.2s;
        }
        .sidebar-search-box:focus-within {
            border-color: var(--primary);
        }
        .sidebar-search-box svg {
            width: 18px;
            height: 18px;
            margin-left: 12px;
            color: var(--text-light);
            flex-shrink: 0;
        }
        .sidebar-search-box input {
            flex: 1;
            padding: 12px;
            border: none;
            outline: none;
            font-size: 14px;
            background: transparent;
        }
        .sidebar-search-box input::placeholder {
            color: var(--text-light);
        }

        .sidebar-categories {
            flex: 1;
            overflow-y: auto;
            padding: 12px 0;
        }
        .sidebar-categories::-webkit-scrollbar {
            width: 6px;
        }
        .sidebar-categories::-webkit-scrollbar-track {
            background: #f1f1f1;
        }
        .sidebar-categories::-webkit-scrollbar-thumb {
            background: #c1c1c1;
            border-radius: 3px;
        }
        .sidebar-categories::-webkit-scrollbar-thumb:hover {
            background: #a1a1a1;
        }

        .category-item {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 14px 20px;
            cursor: pointer;
            transition: all 0.2s;
            border-left: 3px solid transparent;
        }
        .category-item:hover {
            background: #f0f9ff;
            border-left-color: var(--primary);
        }
        .category-item.active {
            background: linear-gradient(90deg, #e0f2fe 0%, #fff 100%);
            border-left-color: var(--primary);
            color: var(--primary);
        }
        .category-item.active .category-name {
            font-weight: 600;
        }
        .category-item-left {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        .category-icon {
            width: 38px;
            height: 38px;
            background: linear-gradient(135deg, #ecfeff 0%, #cffafe 100%);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--primary);
        }
        .category-icon svg {
            width: 18px;
            height: 18px;
        }
        .category-name {
            font-size: 14px;
        }
        .category-count {
            font-size: 12px;
            color: var(--text-light);
            background: #f1f5f9;
            padding: 4px 10px;
            border-radius: 20px;
        }
        .category-item.active .category-count {
            background: var(--primary);
            color: #fff;
        }

        .no-categories {
            padding: 40px 20px;
            text-align: center;
            color: var(--text-light);
        }
        .no-categories svg {
            width: 48px;
            height: 48px;
            margin-bottom: 12px;
            opacity: 0.5;
        }
        .no-categories p {
            font-size: 14px;
        }

        .sidebar-footer {
            padding: 16px 20px;
            border-top: 1px solid var(--border);
            background: #f8fafc;
        }
        .sidebar-footer-btn {
            width: 100%;
            padding: 12px;
            background: var(--primary);
            color: #fff;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.2s;
        }
        .sidebar-footer-btn:hover {
            background: var(--primary-dark);
        }
        
        /* Mobile First - Tablet */
        @media (max-width: 1024px) {
            .header-main { gap: 20px; }
            .search-wrap { max-width: 400px; }
            .hero-stats { gap: 30px; }
            .footer-grid { grid-template-columns: 1fr 1fr; gap: 30px; }
        }
        
        /* Mobile */
        @media (max-width: 768px) {
            .container { padding: 0 16px; }
            
            .header-main { 
                flex-wrap: wrap; 
                height: auto; 
                padding: 12px 0; 
                gap: 12px;
            }
            .logo { font-size: 20px; flex: 1; }
            .logo img { height: 36px; }
            .logo-icon { width: 34px; height: 34px; }
            
            .search-wrap { 
                order: 3; 
                flex: 0 0 100%; 
                max-width: 100%; 
            }
            .search-box input { padding: 10px 14px; font-size: 14px; }
            .search-box button { padding: 0 14px; }
            
            .header-nav { 
                margin-left: 0;
                gap: 4px;
            }
            .nav-item { 
                padding: 6px 10px; 
                font-size: 11px; 
            }
            .nav-item svg { width: 20px; height: 20px; }
            .cart-badge { top: 0; right: 4px; font-size: 9px; padding: 1px 5px; }
            
            .hero { padding: 30px 0; margin-bottom: 0; }
            .hero-content { 
                flex-direction: column; 
                text-align: center; 
                gap: 24px; 
            }
            .hero-text { max-width: 100%; }
            .hero-text h1 { font-size: 24px; margin-bottom: 8px; }
            .hero-text p { font-size: 14px; margin-bottom: 16px; max-width: 100%; }
            .hero-btn { padding: 12px 24px; font-size: 14px; }
            .hero-stats { 
                gap: 16px;
                justify-content: center;
            }
            .stat-num { font-size: 22px; }
            .stat-label { font-size: 11px; }
            
            .features { padding: 16px 0; margin-bottom: 20px; }
            .features-grid { 
                grid-template-columns: 1fr 1fr;
                gap: 16px;
            }
            .feature { 
                flex-direction: column; 
                text-align: center; 
                gap: 8px;
                padding: 12px 8px;
                background: #f8fafc;
                border-radius: 12px;
            }
            .feature-icon { width: 42px; height: 42px; }
            .feature-icon svg { width: 20px; height: 20px; }
            .feature-text { font-size: 13px; }
            .feature-sub { font-size: 11px; }
            
            .footer { padding: 40px 0 20px; }
            .footer-grid { 
                grid-template-columns: 1fr; 
                gap: 30px; 
                text-align: center;
            }
            .footer-brand { font-size: 20px; }
            .footer-desc { font-size: 13px; }
            .footer-title { margin-bottom: 14px; }
            .footer-links li { margin-bottom: 10px; }
            
            .toast { 
                left: 16px; 
                right: 16px; 
                bottom: 16px; 
            }
            
            /* Sidebar Mobile */
            .category-sidebar {
                width: 100%;
                left: -100%;
            }
        }
        
        /* Small Mobile */
        @media (max-width: 480px) {
            .logo { font-size: 18px; }
            .logo img { height: 32px; }
            .hero-text h1 { font-size: 20px; }
            .hero-stats { gap: 12px; }
            .stat-num { font-size: 20px; }
            .stat-label { font-size: 10px; }
            .nav-item span { display: none; }
            .nav-item { padding: 8px; }
        }
        @yield('styles')
    </style>
</head>
<body>

{{-- Category Sidebar Overlay --}}
<div class="sidebar-overlay" id="sidebarOverlay" onclick="closeCategorySidebar()"></div>

{{-- Category Sidebar --}}
<aside class="category-sidebar" id="categorySidebar">
    <div class="sidebar-header">
        <h3>
            <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M3.75 6A2.25 2.25 0 016 3.75h2.25A2.25 2.25 0 0110.5 6v2.25a2.25 2.25 0 01-2.25 2.25H6a2.25 2.25 0 01-2.25-2.25V6zM3.75 15.75A2.25 2.25 0 016 13.5h2.25a2.25 2.25 0 012.25 2.25V18a2.25 2.25 0 01-2.25 2.25H6A2.25 2.25 0 013.75 18v-2.25zM13.5 6a2.25 2.25 0 012.25-2.25H18A2.25 2.25 0 0120.25 6v2.25A2.25 2.25 0 0118 10.5h-2.25a2.25 2.25 0 01-2.25-2.25V6zM13.5 15.75a2.25 2.25 0 012.25-2.25H18a2.25 2.25 0 012.25 2.25V18A2.25 2.25 0 0118 20.25h-2.25A2.25 2.25 0 0113.5 18v-2.25z"/>
            </svg>
            Categories
        </h3>
        <button class="sidebar-close" onclick="closeCategorySidebar()">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/>
            </svg>
        </button>
    </div>
    
    <div class="sidebar-search">
        <div class="sidebar-search-box">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M21 21l-5.197-5.197m0 0A7.5 7.5 0 105.196 5.196a7.5 7.5 0 0010.607 10.607z"/>
            </svg>
            <input type="text" id="categorySearchInput" placeholder="Search categories..." oninput="filterCategories(this.value)">
        </div>
    </div>
    
    <div class="sidebar-categories" id="categoryList">
        {{-- Categories populated by JavaScript --}}
    </div>
    
    <div class="sidebar-footer">
        <button class="sidebar-footer-btn" onclick="selectCategory('', 'All Products')">
            View All Products
        </button>
    </div>
</aside>

{{-- Header --}}
<header class="header">
    <div class="container">
        <div class="header-main">
            <a href="{{ route('ecommerce.shop') }}" class="logo">
                @if($settings->getLogoUrl())
                    <img src="{{ $settings->getLogoUrl() }}" alt="{{ $settings->site_name }}">
                @else
                    <span class="logo-icon"><svg width="22" height="22" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg></span>
                    {{ $settings->site_name ?? 'Store' }}
                @endif
            </a>
            
            <div class="search-wrap">
                @livewire('ecommerce::product-search')
            </div>
            
            <nav class="header-nav">
                @auth
                    <a href="{{ route('ecommerce.account') }}" class="nav-item">
                        <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M15.75 6a3.75 3.75 0 11-7.5 0 3.75 3.75 0 017.5 0zM4.501 20.118a7.5 7.5 0 0114.998 0A17.933 17.933 0 0112 21.75c-2.676 0-5.216-.584-7.499-1.632z"/></svg>
                        <span>Account</span>
                    </a>
                    
                    <a href="{{ route('ecommerce.orders') }}" class="nav-item">
                        <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M15.75 10.5V6a3.75 3.75 0 10-7.5 0v4.5m11.356-1.993l1.263 12c.07.665-.45 1.243-1.119 1.243H4.25a1.125 1.125 0 01-1.12-1.243l1.264-12A1.125 1.125 0 015.513 7.5h12.974c.576 0 1.059.435 1.119 1.007zM8.625 10.5a.375.375 0 11-.75 0 .375.375 0 01.75 0zm7.5 0a.375.375 0 11-.75 0 .375.375 0 01.75 0z"/>
                        </svg>
                        <span>My Orders</span>
                    </a>
                @else
                    <a href="{{ route('ecommerce.login') }}" class="nav-item">
                        <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M15.75 6a3.75 3.75 0 11-7.5 0 3.75 3.75 0 017.5 0zM4.501 20.118a7.5 7.5 0 0114.998 0A17.933 17.933 0 0112 21.75c-2.676 0-5.216-.584-7.499-1.632z"/></svg>
                        <span>Login</span>
                    </a>
                @endauth

                <a href="{{ route('ecommerce.wishlist') }}" class="nav-item">
                    <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M21 8.25c0-2.485-2.099-4.5-4.688-4.5-1.935 0-3.597 1.126-4.312 2.733-.715-1.607-2.377-2.733-4.313-2.733C5.1 3.75 3 5.765 3 8.25c0 7.22 9 12 9 12s9-4.78 9-12z"/></svg>
                    <span>Wishlist</span>
                    @php $wishCount = Auth::check() ? count(session('wishlist_' . Auth::id(), [])) : 0; @endphp
                    <span class="cart-badge" id="headerWishCount" style="{{ $wishCount > 0 ? '' : 'display:none' }}">{{ $wishCount }}</span>
                </a>
                
                <a href="{{ route('ecommerce.cart') }}" class="nav-item cart">
                    <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M15.75 10.5V6a3.75 3.75 0 10-7.5 0v4.5m11.356-1.993l1.263 12c.07.665-.45 1.243-1.119 1.243H4.25a1.125 1.125 0 01-1.12-1.243l1.264-12A1.125 1.125 0 015.513 7.5h12.974c.576 0 1.059.435 1.119 1.007zM8.625 10.5a.375.375 0 11-.75 0 .375.375 0 01.75 0zm7.5 0a.375.375 0 11-.75 0 .375.375 0 01.75 0z"/></svg>
                    <span>Cart</span>
                    @php $cartCount = Auth::check() ? array_sum(array_column(session('cart_' . Auth::id(), []), 'qty')) : 0; @endphp
                    <span class="cart-badge" id="headerCartCount" style="{{ $cartCount > 0 ? '' : 'display:none' }}">{{ $cartCount }}</span>
                </a>
            </nav>
        </div>
    </div>
</header>

<main>@yield('content')</main>

{{-- Footer --}}
<footer class="footer">
    <div class="container">
        <div class="footer-grid">
            <div>
                <div class="footer-brand">{{ $settings->site_name ?? 'Store' }}</div>
                <p class="footer-desc">Your one-stop destination for quality products at amazing prices. Shop with confidence!</p>
            </div>
            <div>
                <div class="footer-title">Quick Links</div>
                <ul class="footer-links">
                    <li><a href="{{ route('ecommerce.shop') }}">All Products</a></li>
                    <li><a href="{{ route('ecommerce.cart') }}">Shopping Cart</a></li>
                    <li><a href="{{ route('ecommerce.wishlist') }}">My Wishlist</a></li>
                    <li><a href="{{ route('ecommerce.page.about') }}">About Us</a></li>
                    <li><a href="{{ route('ecommerce.page.contact') }}">Contact Us</a></li>
                </ul>
            </div>
            <div>
                <div class="footer-title">Policies</div>
                <ul class="footer-links">
                    <li><a href="{{ route('ecommerce.page.privacy') }}">Privacy Policy</a></li>
                    <li><a href="{{ route('ecommerce.page.terms') }}">Terms & Conditions</a></li>
                    <li><a href="{{ route('ecommerce.page.refund') }}">Refund Policy</a></li>
                    <li><a href="{{ route('ecommerce.page.shipping') }}">Shipping Policy</a></li>
                </ul>
            </div>
            <div>
                <div class="footer-title">Contact Us</div>
                <ul class="footer-links">
                    @if($settings->contact_email)<li><a href="mailto:{{ $settings->contact_email }}">{{ $settings->contact_email }}</a></li>@endif
                    @if($settings->contact_phone)<li>{{ $settings->contact_phone }}</li>@endif
                    @if($settings->store_city)<li>{{ $settings->store_city }}, {{ $settings->store_state }}</li>@endif
                </ul>
            </div>
        </div>
        <div class="footer-bottom">© {{ date('Y') }} {{ $settings->site_name ?? 'Store' }}. All Rights Reserved.</div>
    </div>
</footer>

<div class="toast" id="toast"><span id="toastMsg"></span></div>

@livewireScripts
<script>
    // Toast Notification
    function showToast(msg, type) {
        const t = document.getElementById('toast');
        if (!t) return;
        document.getElementById('toastMsg').textContent = msg;
        t.className = 'toast show ' + (type || '');
        setTimeout(() => t.className = 'toast', 3500);
    }
    
    // ==================== CATEGORY SIDEBAR FUNCTIONS ====================
    
    function openCategorySidebar() {
        document.getElementById('categorySidebar').classList.add('active');
        document.getElementById('sidebarOverlay').classList.add('active');
        document.body.style.overflow = 'hidden';
    }
    
    function closeCategorySidebar() {
        document.getElementById('categorySidebar').classList.remove('active');
        document.getElementById('sidebarOverlay').classList.remove('active');
        document.body.style.overflow = '';
    }
    
    function filterCategories(searchTerm) {
        const items = document.querySelectorAll('.category-item');
        const lowerSearch = searchTerm.toLowerCase();
        let hasResults = false;
        
        items.forEach(item => {
            const name = item.querySelector('.category-name')?.textContent.toLowerCase() || '';
            if (name.includes(lowerSearch)) {
                item.style.display = 'flex';
                hasResults = true;
            } else {
                item.style.display = 'none';
            }
        });
        
        let noResults = document.getElementById('noResultsMsg');
        if (!hasResults) {
            if (!noResults) {
                const msg = document.createElement('div');
                msg.id = 'noResultsMsg';
                msg.className = 'no-categories';
                msg.innerHTML = `
                    <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M21 21l-5.197-5.197m0 0A7.5 7.5 0 105.196 5.196a7.5 7.5 0 0010.607 10.607z"/>
                    </svg>
                    <p>No categories found</p>
                `;
                document.getElementById('categoryList').appendChild(msg);
            }
        } else if (noResults) {
            noResults.remove();
        }
    }
    
    // MAIN FUNCTION - Select Category using Livewire dispatch event
    function selectCategory(categoryId, categoryName) {
        // Update active state in sidebar
        document.querySelectorAll('.category-item').forEach(item => {
            item.classList.remove('active');
        });
        if (categoryId) {
            document.querySelector(`[data-category-id="${categoryId}"]`)?.classList.add('active');
        } else {
            document.querySelector('[data-category-id="all"]')?.classList.add('active');
        }
        
        // Use Livewire.dispatch to send event to ProductGrid component
        // This broadcasts to ALL components, and only ProductGrid listens for 'filterByCategory'
        if (typeof Livewire !== 'undefined') {
            Livewire.dispatch('filterByCategory', { categoryId: categoryId });
        }
        
        closeCategorySidebar();
        showToast(categoryId ? `Showing: ${categoryName}` : 'Showing all products', 'info');
    }
    
    // Close sidebar on Escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeCategorySidebar();
        }
    });
    
    // ==================== LIVEWIRE EVENTS ====================
    
    document.addEventListener('livewire:initialized', () => {
        Livewire.on('show-notification', (d) => showToast(d.message, d.type));
        
        Livewire.on('cart-count-updated', (d) => { 
            let c = document.getElementById('headerCartCount');
            if(c) {
                c.textContent = d.count;
                c.style.display = d.count > 0 ? '' : 'none';
            }
        });
        
        Livewire.on('wishlist-count-updated', (d) => { 
            let w = document.getElementById('headerWishCount');
            if(w) {
                w.textContent = d.count;
                w.style.display = d.count > 0 ? '' : 'none';
            }
        });
    });
</script>
@yield('scripts')



{{-- ==================== LOCATION SERVICE AREA CHECK ==================== --}}
{{-- Add this code BEFORE the closing </body> tag in shop-layout.blade.php --}}

{{-- Location Service Modal --}}
<div id="locationModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.6); z-index: 9999; align-items: center; justify-content: center;">
    <div style="background: #fff; border-radius: 16px; padding: 40px; max-width: 450px; width: 90%; text-align: center; box-shadow: 0 20px 60px rgba(0,0,0,0.3); animation: modalSlide 0.3s ease;">
        <div style="width: 80px; height: 80px; background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
            <svg style="width: 40px; height: 40px; color: #dc2626;" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M15 10.5a3 3 0 11-6 0 3 3 0 016 0z"/>
                <path stroke-linecap="round" stroke-linejoin="round" d="M19.5 10.5c0 7.142-7.5 11.25-7.5 11.25S4.5 17.642 4.5 10.5a7.5 7.5 0 1115 0z"/>
            </svg>
        </div>
        <h3 style="font-size: 22px; font-weight: 700; color: #0f172a; margin-bottom: 12px;">Service Not Available</h3>
        <p style="font-size: 15px; color: #64748b; line-height: 1.6; margin-bottom: 8px;">
            Sorry! We currently don't deliver to your location.
        </p>
        <p style="font-size: 14px; color: #94a3b8; margin-bottom: 24px;">
            Our delivery service is available within <strong style="color: #0891b2;">5 km</strong> of our store in Rajajinagar, Bengaluru.
        </p>
        <div style="background: #f8fafc; border-radius: 10px; padding: 16px; margin-bottom: 24px;">
            <p style="font-size: 13px; color: #64748b; margin: 0;">
                <strong style="color: #0f172a;">Store Location:</strong><br>
                11th Cross Rd, West of Chord Road 2nd Stage,<br>
                Bovipalya, Nagapura, Bengaluru - 560086
            </p>
        </div>
        <div style="display: flex; gap: 12px; justify-content: center;">
            <button onclick="closeLocationModal()" style="padding: 12px 24px; background: #e2e8f0; color: #475569; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: background 0.2s;">
                Continue Browsing
            </button>
            <a href="https://maps.google.com/?q=12.9996206,77.5436373" target="_blank" style="padding: 12px 24px; background: #0891b2; color: #fff; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; text-decoration: none; transition: background 0.2s;">
                View on Map
            </a>
        </div>
    </div>
</div>

{{-- Location Permission Modal --}}
<div id="locationPermissionModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.6); z-index: 9999; align-items: center; justify-content: center;">
    <div style="background: #fff; border-radius: 16px; padding: 40px; max-width: 450px; width: 90%; text-align: center; box-shadow: 0 20px 60px rgba(0,0,0,0.3);">
        <div style="width: 80px; height: 80px; background: linear-gradient(135deg, #ecfeff 0%, #cffafe 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
            <svg style="width: 40px; height: 40px; color: #0891b2;" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M15 10.5a3 3 0 11-6 0 3 3 0 016 0z"/>
                <path stroke-linecap="round" stroke-linejoin="round" d="M19.5 10.5c0 7.142-7.5 11.25-7.5 11.25S4.5 17.642 4.5 10.5a7.5 7.5 0 1115 0z"/>
            </svg>
        </div>
        <h3 style="font-size: 22px; font-weight: 700; color: #0f172a; margin-bottom: 12px;">Enable Location</h3>
        <p style="font-size: 15px; color: #64748b; line-height: 1.6; margin-bottom: 24px;">
            Please allow location access to check if delivery is available in your area.
        </p>
        <div style="display: flex; gap: 12px; justify-content: center;">
            <button onclick="skipLocationCheck()" style="padding: 12px 24px; background: #e2e8f0; color: #475569; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer;">
                Skip
            </button>
            <button onclick="requestLocation()" style="padding: 12px 24px; background: #0891b2; color: #fff; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer;">
                Allow Location
            </button>
        </div>
    </div>
</div>

<style>
@keyframes modalSlide {
    from { transform: translateY(-20px); opacity: 0; }
    to { transform: translateY(0); opacity: 1; }
}
</style>

{{-- ==================== LOCATION SERVICE AREA CHECK ==================== --}}
{{-- Add this code BEFORE the closing </body> tag in shop-layout.blade.php --}}

@if($settings->location_restriction_enabled ?? false)
{{-- Location Service Modal --}}
<div id="locationModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.6); z-index: 9999; align-items: center; justify-content: center;">
    <div style="background: #fff; border-radius: 16px; padding: 40px; max-width: 450px; width: 90%; text-align: center; box-shadow: 0 20px 60px rgba(0,0,0,0.3); animation: modalSlide 0.3s ease;">
        <div style="width: 80px; height: 80px; background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
            <svg style="width: 40px; height: 40px; color: #dc2626;" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M15 10.5a3 3 0 11-6 0 3 3 0 016 0z"/>
                <path stroke-linecap="round" stroke-linejoin="round" d="M19.5 10.5c0 7.142-7.5 11.25-7.5 11.25S4.5 17.642 4.5 10.5a7.5 7.5 0 1115 0z"/>
            </svg>
        </div>
        <h3 style="font-size: 22px; font-weight: 700; color: #0f172a; margin-bottom: 12px;">Service Not Available</h3>
        <p style="font-size: 15px; color: #64748b; line-height: 1.6; margin-bottom: 8px;">
            Sorry! We currently don't deliver to your location.
        </p>
        <p id="distanceInfo" style="font-size: 14px; color: #dc2626; margin-bottom: 16px; font-weight: 500;"></p>
        <p style="font-size: 14px; color: #94a3b8; margin-bottom: 24px;">
            Our delivery service is available within <strong style="color: #0891b2;">{{ $settings->service_radius_km ?? 5 }} km</strong> of our store.
        </p>
        <div style="background: #f8fafc; border-radius: 10px; padding: 16px; margin-bottom: 24px;">
            <p style="font-size: 13px; color: #64748b; margin: 0;">
                <strong style="color: #0f172a;">Store Location:</strong><br>
                {{ $settings->store_address ?? '' }}<br>
                {{ $settings->store_city ?? '' }}, {{ $settings->store_state ?? '' }} - {{ $settings->store_pincode ?? '' }}
            </p>
        </div>
        <div style="display: flex; gap: 12px; justify-content: center; flex-wrap: wrap;">
            <button onclick="closeLocationModal()" style="padding: 12px 24px; background: #e2e8f0; color: #475569; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: background 0.2s;">
                Continue Browsing
            </button>
            <a href="https://maps.google.com/?q={{ $settings->store_latitude ?? '12.9996206' }},{{ $settings->store_longitude ?? '77.5436373' }}" target="_blank" style="padding: 12px 24px; background: #0891b2; color: #fff; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; text-decoration: none; transition: background 0.2s;">
                View Store Location
            </a>
        </div>
    </div>
</div>

{{-- Location Permission Modal --}}
<div id="locationPermissionModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.6); z-index: 9999; align-items: center; justify-content: center;">
    <div style="background: #fff; border-radius: 16px; padding: 40px; max-width: 450px; width: 90%; text-align: center; box-shadow: 0 20px 60px rgba(0,0,0,0.3);">
        <div style="width: 80px; height: 80px; background: linear-gradient(135deg, #ecfeff 0%, #cffafe 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
            <svg style="width: 40px; height: 40px; color: #0891b2;" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M15 10.5a3 3 0 11-6 0 3 3 0 016 0z"/>
                <path stroke-linecap="round" stroke-linejoin="round" d="M19.5 10.5c0 7.142-7.5 11.25-7.5 11.25S4.5 17.642 4.5 10.5a7.5 7.5 0 1115 0z"/>
            </svg>
        </div>
        <h3 style="font-size: 22px; font-weight: 700; color: #0f172a; margin-bottom: 12px;">Check Delivery Availability</h3>
        <p style="font-size: 15px; color: #64748b; line-height: 1.6; margin-bottom: 24px;">
            Allow location access to check if delivery is available in your area.
        </p>
        <div style="display: flex; gap: 12px; justify-content: center; flex-wrap: wrap;">
            <button onclick="skipLocationCheck()" style="padding: 12px 24px; background: #e2e8f0; color: #475569; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer;">
                Skip
            </button>
            <button onclick="requestLocation()" style="padding: 12px 24px; background: #0891b2; color: #fff; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer;">
                Check My Location
            </button>
        </div>
    </div>
</div>

{{-- Service Available Modal (Success) --}}
<div id="locationSuccessModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.6); z-index: 9999; align-items: center; justify-content: center;">
    <div style="background: #fff; border-radius: 16px; padding: 40px; max-width: 450px; width: 90%; text-align: center; box-shadow: 0 20px 60px rgba(0,0,0,0.3); animation: modalSlide 0.3s ease;">
        <div style="width: 80px; height: 80px; background: linear-gradient(135deg, #dcfce7 0%, #bbf7d0 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
            <svg style="width: 40px; height: 40px; color: #16a34a;" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7"/>
            </svg>
        </div>
        <h3 style="font-size: 22px; font-weight: 700; color: #0f172a; margin-bottom: 12px;">Great News! 🎉</h3>
        <p style="font-size: 15px; color: #64748b; line-height: 1.6; margin-bottom: 8px;">
            Delivery is available at your location!
        </p>
        <p id="successDistanceInfo" style="font-size: 14px; color: #16a34a; margin-bottom: 24px; font-weight: 500;"></p>
        <button onclick="closeSuccessModal()" style="padding: 12px 32px; background: #16a34a; color: #fff; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer;">
            Start Shopping
        </button>
    </div>
</div>

<style>
@keyframes modalSlide {
    from { transform: translateY(-20px); opacity: 0; }
    to { transform: translateY(0); opacity: 1; }
}
</style>

<script>
// ==================== LOCATION SERVICE AREA CHECK ====================

// Shop coordinates from settings
const SHOP_LOCATION = {
    lat: {{ $settings->store_latitude ?? 12.9996206 }},
    lng: {{ $settings->store_longitude ?? 77.5436373 }}
};

// Service radius from settings
const SERVICE_RADIUS_KM = {{ $settings->service_radius_km ?? 5 }};

// Session storage key
const LOCATION_CHECK_KEY = 'location_checked_{{ $settings->id ?? 1 }}';

// Calculate distance using Haversine formula
function calculateDistance(lat1, lon1, lat2, lon2) {
    const R = 6371;
    const dLat = (lat2 - lat1) * Math.PI / 180;
    const dLon = (lon2 - lon1) * Math.PI / 180;
    const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
              Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
              Math.sin(dLon/2) * Math.sin(dLon/2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    return R * c;
}

function showLocationModal(distance) {
    document.getElementById('distanceInfo').textContent = 'You are ' + distance.toFixed(1) + ' km away from our store.';
    document.getElementById('locationModal').style.display = 'flex';
    document.body.style.overflow = 'hidden';
}

function closeLocationModal() {
    document.getElementById('locationModal').style.display = 'none';
    document.body.style.overflow = '';
    localStorage.setItem(LOCATION_CHECK_KEY, 'checked');
}

function showSuccessModal(distance) {
    document.getElementById('successDistanceInfo').textContent = 'You are only ' + distance.toFixed(1) + ' km from our store.';
    document.getElementById('locationSuccessModal').style.display = 'flex';
    document.body.style.overflow = 'hidden';
}

function closeSuccessModal() {
    document.getElementById('locationSuccessModal').style.display = 'none';
    document.body.style.overflow = '';
    localStorage.setItem(LOCATION_CHECK_KEY, 'within_area');
}

function showPermissionModal() {
    document.getElementById('locationPermissionModal').style.display = 'flex';
    document.body.style.overflow = 'hidden';
}

function closePermissionModal() {
    document.getElementById('locationPermissionModal').style.display = 'none';
    document.body.style.overflow = '';
}

function skipLocationCheck() {
    closePermissionModal();
    localStorage.setItem(LOCATION_CHECK_KEY, 'skipped');
}

function requestLocation() {
    closePermissionModal();
    
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                
                const distance = calculateDistance(
                    userLat, userLng,
                    SHOP_LOCATION.lat, SHOP_LOCATION.lng
                );
                
                console.log('User location:', userLat, userLng);
                console.log('Distance from shop:', distance.toFixed(2), 'km');
                console.log('Service radius:', SERVICE_RADIUS_KM, 'km');
                
                if (distance > SERVICE_RADIUS_KM) {
                    showLocationModal(distance);
                } else {
                    showSuccessModal(distance);
                }
            },
            function(error) {
                console.log('Geolocation error:', error.message);
                localStorage.setItem(LOCATION_CHECK_KEY, 'error');
            },
            {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 0
            }
        );
    }
}

function initLocationCheck() {
    const locationStatus = localStorage.getItem(LOCATION_CHECK_KEY);
    
    if (locationStatus) {
        console.log('Location already checked:', locationStatus);
        return;
    }
    
    if (!navigator.geolocation) {
        console.log('Geolocation not supported');
        return;
    }
    
    setTimeout(showPermissionModal, 2000);
}

document.addEventListener('DOMContentLoaded', function() {
    setTimeout(initLocationCheck, 1500);
});

// Manual test function (for debugging in console)
function testLocation(lat, lng) {
    const distance = calculateDistance(lat, lng, SHOP_LOCATION.lat, SHOP_LOCATION.lng);
    console.log('Test location:', lat, lng);
    console.log('Distance:', distance.toFixed(2), 'km');
    console.log('Service radius:', SERVICE_RADIUS_KM, 'km');
    
    if (distance > SERVICE_RADIUS_KM) {
        showLocationModal(distance);
    } else {
        showSuccessModal(distance);
    }
}

// testLocation(13.035413315939238, 77.59980581006931); 7km away

// Test examples (run in browser console):
// testLocation(52.3676, 4.9041);     // Amsterdam - far away
// testLocation(28.6139, 77.2090);    // Delhi - far away
// testLocation(12.9716, 77.5946);    // Bangalore MG Road - may be within range
</script>
@endif


{{-- function testLocation(lat, lng) {
    const distance = calculateDistance(lat, lng, SHOP_LOCATION.lat, SHOP_LOCATION.lng);
    console.log('Test location:', lat, lng);
    console.log('Distance:', distance.toFixed(2), 'km');
    
    if (distance > SERVICE_RADIUS_KM) {
        showLocationModal(distance);
    } else {
        showSuccessModal(distance);
    }
} --}}

</body>
</html>