<?php

namespace Modules\Inventory\Console\Commands;

use Illuminate\Console\Command;
use Modules\Inventory\Services\ExpiryAlertService;

class CheckExpiryAlerts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:check-expiry 
                            {--dry-run : Show results without sending emails}
                            {--days=30 : Days to check for expiring items}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check for expiring inventory lots and send email alerts';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Checking inventory expiry dates...');
        $this->newLine();

        $dryRun = $this->option('dry-run');
        
        if ($dryRun) {
            $this->warn('⚠️  DRY RUN MODE - No emails will be sent');
            $this->newLine();
        }

        $results = ExpiryAlertService::checkAndSendAlerts();

        // Display Expired Items
        if (count($results['expired']) > 0) {
            $this->error('🚫 EXPIRED ITEMS: ' . count($results['expired']));
            $this->table(
                ['Product', 'Lot No', 'Batch', 'Expired On', 'Stock'],
                collect($results['expired'])->map(fn($item) => [
                    $item['product_name'],
                    $item['lot_no'],
                    $item['batch_no'] ?? '-',
                    $item['expiry_date'],
                    $item['stock'],
                ])->toArray()
            );
            $this->newLine();
        }

        // Display Critical Items (≤7 days)
        if (count($results['critical']) > 0) {
            $this->error('🔴 CRITICAL (≤7 days): ' . count($results['critical']));
            $this->table(
                ['Product', 'Lot No', 'Expiry Date', 'Days Left', 'Stock'],
                collect($results['critical'])->map(fn($item) => [
                    $item['product_name'],
                    $item['lot_no'],
                    $item['expiry_date'],
                    $item['days'] . ' days',
                    $item['stock'],
                ])->toArray()
            );
            $this->newLine();
        }

        // Display Warning Items (≤30 days)
        if (count($results['warning']) > 0) {
            $this->warn('🟡 WARNING (≤30 days): ' . count($results['warning']));
            $this->table(
                ['Product', 'Lot No', 'Expiry Date', 'Days Left', 'Stock'],
                collect($results['warning'])->map(fn($item) => [
                    $item['product_name'],
                    $item['lot_no'],
                    $item['expiry_date'],
                    $item['days'] . ' days',
                    $item['stock'],
                ])->toArray()
            );
            $this->newLine();
        }

        // Display Info Items (≤90 days)
        if (count($results['info']) > 0) {
            $this->info('ℹ️  INFO (≤90 days): ' . count($results['info']));
            $this->newLine();
        }

        // Summary
        $this->newLine();
        $this->info('📊 SUMMARY:');
        $this->line("   • Expired: " . count($results['expired']));
        $this->line("   • Critical (≤7 days): " . count($results['critical']));
        $this->line("   • Warning (≤30 days): " . count($results['warning']));
        $this->line("   • Info (≤90 days): " . count($results['info']));
        
        if ($results['emails_sent'] > 0) {
            $this->newLine();
            $this->info('✅ Alert email sent successfully!');
        } elseif (!$dryRun && (count($results['expired']) > 0 || count($results['critical']) > 0 || count($results['warning']) > 0)) {
            $this->newLine();
            $this->warn('⚠️  No email sent. Check email configuration and alert recipients.');
        } else {
            $this->newLine();
            $this->info('✨ No critical alerts to report.');
        }

        return Command::SUCCESS;
    }
}
