<?php

namespace Modules\Inventory\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;

class ManageModuleRoles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:roles {action : install or uninstall}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Install or uninstall Inventory module roles';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $action = $this->argument('action');
        
        if (!in_array($action, ['install', 'uninstall'])) {
            $this->error('Invalid action. Use "install" or "uninstall".');
            return 1;
        }
        
        if (!Schema::hasTable('roles') || !Schema::hasTable('permissions')) {
            $this->error('Roles and permissions tables not found. Please run migrations first.');
            return 1;
        }
        
        if ($action === 'install') {
            return $this->installRoles();
        }
        
        return $this->uninstallRoles();
    }
    
    /**
     * Install module roles
     */
    protected function installRoles(): int
    {
        $this->info('Installing Inventory module roles...');
        
        try {
            $roleClass = config('permission.models.role', \Spatie\Permission\Models\Role::class);
            $permissionClass = config('permission.models.permission', \Spatie\Permission\Models\Permission::class);
            
            $moduleRoles = config('inventory.roles', []);
            
            if (empty($moduleRoles)) {
                $this->warn('No roles defined in config.');
                return 0;
            }
            
            foreach ($moduleRoles as $roleConfig) {
                // Create role
                $role = $roleClass::firstOrCreate(
                    ['name' => $roleConfig['name'], 'guard_name' => $roleConfig['guard'] ?? 'admin']
                );
                
                $this->line("  ✓ Created role: {$roleConfig['name']}");
                
                // Sync permissions
                if (!empty($roleConfig['permissions'])) {
                    $permissionNames = [];
                    
                    foreach ($roleConfig['permissions'] as $permPattern) {
                        if (strpos($permPattern, '*') !== false) {
                            // Wildcard pattern
                            $pattern = str_replace('.', '\.', $permPattern);
                            $pattern = str_replace('*', '.*', $pattern);
                            
                            $matchingPerms = $permissionClass::where('guard_name', $roleConfig['guard'] ?? 'admin')
                                ->get()
                                ->filter(function($p) use ($pattern) {
                                    return preg_match("/^{$pattern}$/", $p->name);
                                })
                                ->pluck('name')
                                ->toArray();
                            
                            $permissionNames = array_merge($permissionNames, $matchingPerms);
                        } else {
                            if ($permissionClass::where('name', $permPattern)->where('guard_name', $roleConfig['guard'] ?? 'admin')->exists()) {
                                $permissionNames[] = $permPattern;
                            }
                        }
                    }
                    
                    $permissionNames = array_unique($permissionNames);
                    
                    if (!empty($permissionNames)) {
                        $role->syncPermissions($permissionNames);
                        $this->line("    → Assigned " . count($permissionNames) . " permissions");
                    }
                }
            }
            
            // Clear permission cache
            app(\Spatie\Permission\PermissionRegistrar::class)->forgetCachedPermissions();
            
            $this->info('Roles installed successfully!');
            return 0;
            
        } catch (\Exception $e) {
            $this->error('Failed to install roles: ' . $e->getMessage());
            Log::error('Inventory roles install failed', ['error' => $e->getMessage()]);
            return 1;
        }
    }
    
    /**
     * Uninstall module roles
     */
    protected function uninstallRoles(): int
    {
        $this->info('Uninstalling Inventory module roles...');
        
        if (!$this->confirm('This will delete the warehouse-manager and warehouse-staff roles. Continue?')) {
            $this->info('Cancelled.');
            return 0;
        }
        
        try {
            $roleClass = config('permission.models.role', \Spatie\Permission\Models\Role::class);
            
            $moduleRoles = config('inventory.roles', []);
            
            foreach ($moduleRoles as $roleConfig) {
                $deleted = $roleClass::where('name', $roleConfig['name'])
                    ->where('guard_name', $roleConfig['guard'] ?? 'admin')
                    ->delete();
                
                if ($deleted) {
                    $this->line("  ✓ Deleted role: {$roleConfig['name']}");
                }
            }
            
            // Clear permission cache
            app(\Spatie\Permission\PermissionRegistrar::class)->forgetCachedPermissions();
            
            $this->info('Roles uninstalled successfully!');
            return 0;
            
        } catch (\Exception $e) {
            $this->error('Failed to uninstall roles: ' . $e->getMessage());
            Log::error('Inventory roles uninstall failed', ['error' => $e->getMessage()]);
            return 1;
        }
    }
}
