<?php

namespace Modules\Inventory\Console\Commands;

use Illuminate\Console\Command;
use Modules\Inventory\Services\ExpiryAlertService;

class SendExpiryAlerts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:expiry-alerts 
                            {--dry-run : Run without sending emails}
                            {--force : Force send even if already sent today}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check for expiring lots and send email alerts';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Checking for expiring inventory lots...');
        $this->newLine();

        $results = ExpiryAlertService::checkAndSendAlerts();

        // Display results
        $this->table(
            ['Category', 'Count'],
            [
                ['🚫 Expired', count($results['expired'])],
                ['🔴 Critical (≤7 days)', count($results['critical'])],
                ['🟡 Warning (≤30 days)', count($results['warning'])],
                ['🟢 Info (≤90 days)', count($results['info'])],
            ]
        );

        $this->newLine();

        // Show details for critical items
        if (!empty($results['expired'])) {
            $this->error('⚠️  EXPIRED ITEMS:');
            foreach ($results['expired'] as $item) {
                $this->line("   • {$item['product_name']} - Lot: {$item['lot_no']} - Expired: {$item['expiry_date']} - Stock: {$item['stock']}");
            }
            $this->newLine();
        }

        if (!empty($results['critical'])) {
            $this->warn('🔴 CRITICAL - Expiring within 7 days:');
            foreach ($results['critical'] as $item) {
                $this->line("   • {$item['product_name']} - Lot: {$item['lot_no']} - {$item['days']} days - Stock: {$item['stock']}");
            }
            $this->newLine();
        }

        // Email status
        if ($this->option('dry-run')) {
            $this->info('📧 Dry run mode - No emails sent');
        } elseif ($results['emails_sent'] > 0) {
            $this->info('📧 Alert email sent successfully!');
        } else {
            $this->line('📧 No email sent (no critical items or no recipients configured)');
        }

        $this->newLine();
        $this->info('✅ Expiry check completed!');

        return Command::SUCCESS;
    }
}
