<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

/**
 * SMART INVENTORY MIGRATION v2.1
 * 
 * NOTE: Does NOT require doctrine/dbal package - uses raw SQL
 */
return new class extends Migration
{
    public function up(): void
    {
        // 1. BRANDS
        if (!Schema::hasTable('brands')) {
            Schema::create('brands', function (Blueprint $table) {
                $table->id();
                $table->string('name', 191);
                $table->string('logo', 255)->nullable();
                $table->text('description')->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->index('name');
                $table->index('is_active');
            });
        }

        // 2. PRODUCT CATEGORIES
        if (!Schema::hasTable('product_categories')) {
            Schema::create('product_categories', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('parent_id')->nullable();
                $table->string('code', 50)->nullable();
                $table->string('name', 191);
                $table->text('description')->nullable();
                $table->integer('sort_order')->default(0);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->foreign('parent_id')->references('id')->on('product_categories')->onDelete('set null');
                $table->index('parent_id');
                $table->index('is_active');
            });
        }

        // 3. UNITS
        if (!Schema::hasTable('units')) {
            Schema::create('units', function (Blueprint $table) {
                $table->id();
                $table->string('name', 50);
                $table->string('short_name', 20);
                $table->unsignedBigInteger('base_unit_id')->nullable();
                $table->decimal('conversion_factor', 15, 4)->default(1);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->foreign('base_unit_id')->references('id')->on('units')->onDelete('set null');
                $table->index('short_name');
                $table->index('is_active');
            });
        }

        // 4. TAGS
        if (!Schema::hasTable('tags')) {
            Schema::create('tags', function (Blueprint $table) {
                $table->id();
                $table->string('name', 100);
                $table->string('slug', 100)->unique();
                $table->string('color', 7)->nullable();
                $table->timestamps();
            });
        }

        // 5. PRODUCT ATTRIBUTES
        if (!Schema::hasTable('product_attributes')) {
            Schema::create('product_attributes', function (Blueprint $table) {
                $table->id();
                $table->string('name', 100);
                $table->string('slug', 100)->unique();
                $table->string('type', 20)->default('select');
                $table->integer('sort_order')->default(0);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        // 6. ATTRIBUTE VALUES
        if (!Schema::hasTable('attribute_values')) {
            Schema::create('attribute_values', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('attribute_id');
                $table->string('value', 100);
                $table->string('slug', 100);
                $table->string('color_code', 7)->nullable();
                $table->integer('sort_order')->default(0);
                $table->timestamps();
                $table->foreign('attribute_id')->references('id')->on('product_attributes')->onDelete('cascade');
                $table->unique(['attribute_id', 'slug']);
            });
        }

        // 7. PRODUCTS TABLE
        if (Schema::hasTable('products')) {
            $this->addMissingProductColumns();
        } else {
            $this->createFullProductsTable();
        }

        // 8. PRODUCT VARIATIONS
        if (!Schema::hasTable('product_variations')) {
            Schema::create('product_variations', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('product_id');
                $table->string('sku', 100);
                $table->string('barcode', 100)->nullable();
                $table->string('variation_name', 191)->nullable();
                $table->decimal('purchase_price', 12, 2)->nullable();
                $table->decimal('sale_price', 12, 2)->nullable();
                $table->decimal('mrp', 12, 2)->nullable();
                $table->string('image_path', 255)->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->unique('sku');
                $table->index('product_id');
                $table->index('barcode');
            });
        }

        // 9. PRODUCT IMAGES
        if (!Schema::hasTable('product_images')) {
            Schema::create('product_images', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('variation_id')->nullable();
                $table->string('image_path', 255);
                $table->string('alt_text', 255)->nullable();
                $table->integer('sort_order')->default(0);
                $table->boolean('is_primary')->default(false);
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->index('product_id');
            });
        }

        // 10. PRODUCT UNITS
        if (!Schema::hasTable('product_units')) {
            Schema::create('product_units', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('unit_id');
                $table->string('unit_name', 100)->nullable();
                $table->decimal('conversion_factor', 10, 4)->default(1);
                $table->decimal('purchase_price', 12, 2)->nullable();
                $table->decimal('sale_price', 12, 2)->nullable();
                $table->string('barcode', 100)->nullable();
                $table->boolean('is_purchase_unit')->default(true);
                $table->boolean('is_sale_unit')->default(true);
                $table->boolean('is_default')->default(false);
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->foreign('unit_id')->references('id')->on('units')->onDelete('restrict');
                $table->unique(['product_id', 'unit_id']);
            });
        }

        // 11-13. PIVOT TABLES
        if (!Schema::hasTable('product_attribute_map')) {
            Schema::create('product_attribute_map', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('attribute_id');
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->foreign('attribute_id')->references('id')->on('product_attributes')->onDelete('cascade');
                $table->unique(['product_id', 'attribute_id']);
            });
        }

        if (!Schema::hasTable('variation_attribute_values')) {
            Schema::create('variation_attribute_values', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('variation_id');
                $table->unsignedBigInteger('attribute_value_id');
                $table->timestamps();
                $table->foreign('variation_id')->references('id')->on('product_variations')->onDelete('cascade');
                $table->foreign('attribute_value_id')->references('id')->on('attribute_values')->onDelete('cascade');
                $table->unique(['variation_id', 'attribute_value_id'], 'var_attr_val_unique');
            });
        }

        if (!Schema::hasTable('product_tags')) {
            Schema::create('product_tags', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('tag_id');
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->foreign('tag_id')->references('id')->on('tags')->onDelete('cascade');
                $table->unique(['product_id', 'tag_id']);
            });
        }

        // PRODUCT RELATIONS (Cross-sell, Upsell, Accessories)
        if (!Schema::hasTable('product_relations')) {
            Schema::create('product_relations', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('related_product_id');
                $table->enum('relation_type', ['cross_sell', 'upsell', 'accessory'])->default('cross_sell');
                $table->integer('sort_order')->default(0);
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->foreign('related_product_id')->references('id')->on('products')->onDelete('cascade');
                $table->unique(['product_id', 'related_product_id', 'relation_type'], 'product_rel_unique');
                $table->index('product_id');
                $table->index('relation_type');
            });
        }

        // 14. WAREHOUSES
        if (!Schema::hasTable('warehouses')) {
            Schema::create('warehouses', function (Blueprint $table) {
                $table->id();
                $table->string('name', 191);
                $table->string('code', 50);
                $table->string('address', 255)->nullable();
                $table->string('city', 100)->nullable();
                $table->string('state', 100)->nullable();
                $table->string('country', 100)->default('India');
                $table->unsignedBigInteger('contact_staff_id')->nullable();
                $table->string('contact_person', 191)->nullable();
                $table->string('phone', 50)->nullable();
                $table->enum('type', ['STORAGE', 'SHOP', 'RETURN_CENTER'])->default('STORAGE');
                $table->boolean('is_default')->default(false);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->unique('code');
                $table->index('is_default');
                $table->index('is_active');
                $table->index('contact_staff_id');
            });
        }

        // 15. RACKS
        if (!Schema::hasTable('racks')) {
            Schema::create('racks', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('warehouse_id');
                $table->string('code', 50);
                $table->string('name', 191)->nullable();
                $table->string('zone', 50)->nullable();
                $table->string('aisle', 50)->nullable();
                $table->string('level', 50)->nullable();
                $table->enum('storage_type', ['PALLET', 'SHELF', 'BIN', 'FLOOR', 'COLD', 'FREEZER'])->default('SHELF');
                $table->decimal('max_capacity', 15, 2)->nullable();
                $table->unsignedBigInteger('capacity_unit_id')->nullable();
                $table->decimal('max_weight', 15, 2)->nullable();
                $table->text('description')->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->foreign('warehouse_id')->references('id')->on('warehouses')->onDelete('cascade');
                $table->unique(['warehouse_id', 'code']);
            });
        }

        // 16. LOTS
        if (!Schema::hasTable('lots')) {
            Schema::create('lots', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('variation_id')->nullable();
                $table->string('lot_no', 100);
                $table->string('batch_no', 100)->nullable();
                $table->date('manufacturing_date')->nullable();
                $table->date('expiry_date')->nullable();
                $table->decimal('initial_qty', 12, 3)->default(0);
                $table->decimal('purchase_price', 12, 2)->nullable();
                $table->decimal('sale_price', 12, 2)->nullable();
                $table->enum('status', ['ACTIVE', 'EXPIRED', 'RECALLED', 'CONSUMED'])->default('ACTIVE');
                $table->text('notes')->nullable();
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->unique(['product_id', 'lot_no']);
                $table->index('expiry_date');
                $table->index('status');
            });
        }

        // 17. STOCK LEVELS
        if (!Schema::hasTable('stock_levels')) {
            Schema::create('stock_levels', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('variation_id')->nullable();
                $table->unsignedBigInteger('warehouse_id');
                $table->unsignedBigInteger('rack_id')->nullable();
                $table->unsignedBigInteger('lot_id')->nullable();
                $table->unsignedBigInteger('unit_id')->nullable();
                $table->decimal('qty', 12, 3)->default(0);
                $table->decimal('reserved_qty', 12, 3)->default(0);
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->foreign('warehouse_id')->references('id')->on('warehouses')->onDelete('restrict');
                $table->unique(['product_id', 'variation_id', 'warehouse_id', 'rack_id', 'lot_id'], 'stock_levels_unique');
                $table->index('product_id');
                $table->index('warehouse_id');
            });
        }

        // 18. STOCK MOVEMENTS
        if (!Schema::hasTable('stock_movements')) {
            Schema::create('stock_movements', function (Blueprint $table) {
                $table->id();
                $table->string('reference_no', 50);
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('variation_id')->nullable();
                $table->unsignedBigInteger('warehouse_id');
                $table->unsignedBigInteger('rack_id')->nullable();
                $table->unsignedBigInteger('lot_id')->nullable();
                $table->unsignedBigInteger('unit_id')->nullable();
                $table->decimal('qty', 12, 3);
                $table->decimal('base_qty', 12, 3);
                $table->decimal('stock_before', 12, 3)->default(0);
                $table->decimal('stock_after', 12, 3)->default(0);
                $table->decimal('purchase_price', 12, 2)->nullable();
                $table->enum('movement_type', ['IN', 'OUT', 'TRANSFER', 'ADJUSTMENT', 'RETURN']);
                $table->enum('reference_type', ['PURCHASE', 'SALE', 'TRANSFER', 'ADJUSTMENT', 'RETURN', 'OPENING']);
                $table->unsignedBigInteger('reference_id')->nullable();
                $table->string('reason', 255)->nullable();
                $table->text('notes')->nullable();
                $table->unsignedBigInteger('created_by')->nullable();
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->foreign('warehouse_id')->references('id')->on('warehouses')->onDelete('restrict');
                $table->index('reference_no');
                $table->index('product_id');
                $table->index('movement_type');
                $table->index('created_at');
            });
        }

        // 19. STOCK TRANSFERS
        if (!Schema::hasTable('stock_transfers')) {
            Schema::create('stock_transfers', function (Blueprint $table) {
                $table->id();
                $table->string('transfer_no', 50);
                $table->unsignedBigInteger('product_id');
                $table->unsignedBigInteger('variation_id')->nullable();
                $table->unsignedBigInteger('lot_id')->nullable();
                $table->unsignedBigInteger('unit_id')->nullable();
                $table->unsignedBigInteger('from_warehouse_id');
                $table->unsignedBigInteger('to_warehouse_id');
                $table->unsignedBigInteger('from_rack_id')->nullable();
                $table->unsignedBigInteger('to_rack_id')->nullable();
                $table->decimal('qty', 12, 3);
                $table->decimal('base_qty', 12, 3);
                $table->enum('status', ['PENDING', 'IN_TRANSIT', 'COMPLETED', 'CANCELLED'])->default('PENDING');
                $table->string('reason', 255)->nullable();
                $table->text('notes')->nullable();
                $table->unsignedBigInteger('created_by')->nullable();
                $table->timestamps();
                $table->foreign('product_id')->references('id')->on('products')->onDelete('cascade');
                $table->foreign('from_warehouse_id')->references('id')->on('warehouses')->onDelete('restrict');
                $table->foreign('to_warehouse_id')->references('id')->on('warehouses')->onDelete('restrict');
                $table->unique('transfer_no');
                $table->index('product_id');
                $table->index('status');
            });
        }

        $this->seedDefaultData();
    }

    public function down(): void
    {
        Schema::dropIfExists('stock_transfers');
        Schema::dropIfExists('stock_movements');
        Schema::dropIfExists('stock_levels');
        Schema::dropIfExists('lots');
        Schema::dropIfExists('racks');
        Schema::dropIfExists('warehouses');
        Schema::dropIfExists('product_tags');
        Schema::dropIfExists('variation_attribute_values');
        Schema::dropIfExists('product_attribute_map');
        Schema::dropIfExists('product_units');
        Schema::dropIfExists('product_images');
        Schema::dropIfExists('product_variations');
        Schema::dropIfExists('attribute_values');
        Schema::dropIfExists('product_attributes');
        Schema::dropIfExists('tags');
        Schema::dropIfExists('units');
        Schema::dropIfExists('product_categories');
        Schema::dropIfExists('brands');
    }

    private function addMissingProductColumns(): void
    {
        $existingColumns = $this->getTableColumns('products');

        Schema::table('products', function (Blueprint $table) use ($existingColumns) {
            if (!in_array('category_id', $existingColumns)) {
                $table->unsignedBigInteger('category_id')->nullable()->after('id');
            }
            if (!in_array('brand_id', $existingColumns)) {
                $table->unsignedBigInteger('brand_id')->nullable()->after('category_id');
            }
            if (!in_array('unit_id', $existingColumns)) {
                $table->unsignedBigInteger('unit_id')->nullable()->after('brand_id');
            }
            if (!in_array('barcode', $existingColumns)) {
                $table->string('barcode', 100)->nullable()->after('sku');
            }
            if (!in_array('hsn_code', $existingColumns)) {
                $table->string('hsn_code', 50)->nullable()->after('barcode');
            }
            if (!in_array('model_number', $existingColumns)) {
                $table->string('model_number', 100)->nullable()->after('hsn_code');
            }
            if (!in_array('short_description', $existingColumns)) {
                $table->text('short_description')->nullable()->after('description');
            }
            if (!in_array('default_profit_rate', $existingColumns)) {
                $table->decimal('default_profit_rate', 5, 2)->default(0)->after('mrp');
            }
            if (!in_array('tax_1_id', $existingColumns)) {
                $table->unsignedBigInteger('tax_1_id')->nullable()->after('default_profit_rate');
            }
            if (!in_array('tax_2_id', $existingColumns)) {
                $table->unsignedBigInteger('tax_2_id')->nullable()->after('tax_1_id');
            }
            if (!in_array('min_stock_level', $existingColumns)) {
                $table->decimal('min_stock_level', 12, 3)->default(0)->after('tax_2_id');
            }
            if (!in_array('max_stock_level', $existingColumns)) {
                $table->decimal('max_stock_level', 12, 3)->default(0)->after('min_stock_level');
            }
            if (!in_array('is_batch_managed', $existingColumns)) {
                $table->boolean('is_batch_managed')->default(false)->after('max_stock_level');
            }
            if (!in_array('can_be_sold', $existingColumns)) {
                $table->boolean('can_be_sold')->default(true)->after('is_batch_managed');
            }
            if (!in_array('can_be_purchased', $existingColumns)) {
                $table->boolean('can_be_purchased')->default(true)->after('can_be_sold');
            }
            if (!in_array('track_inventory', $existingColumns)) {
                $table->boolean('track_inventory')->default(true)->after('can_be_purchased');
            }
            if (!in_array('has_variants', $existingColumns)) {
                $table->boolean('has_variants')->default(false)->after('track_inventory');
            }
            if (!in_array('is_featured', $existingColumns)) {
                $table->boolean('is_featured')->default(false)->after('has_variants');
            }
            if (!in_array('status', $existingColumns)) {
                $table->enum('status', ['draft', 'active', 'paused', 'out_of_stock', 'discontinued', 'archived'])->default('active')->after('is_featured');
            }
        });

        // Add indexes using raw SQL
        $this->addIndexIfNotExists('products', 'products_sku_unique', 'sku', true);
        $this->addIndexIfNotExists('products', 'products_barcode_index', 'barcode');
        $this->addIndexIfNotExists('products', 'products_category_id_index', 'category_id');
        $this->addIndexIfNotExists('products', 'products_brand_id_index', 'brand_id');
        $this->addIndexIfNotExists('products', 'products_status_index', 'status');
        $this->addIndexIfNotExists('products', 'products_is_featured_index', 'is_featured');

        // Add foreign keys
        $this->addForeignKeyIfNotExists('products', 'category_id', 'product_categories', 'id', 'SET NULL');
        $this->addForeignKeyIfNotExists('products', 'brand_id', 'brands', 'id', 'SET NULL');
        $this->addForeignKeyIfNotExists('products', 'unit_id', 'units', 'id', 'SET NULL');
    }

    private function createFullProductsTable(): void
    {
        Schema::create('products', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('category_id')->nullable();
            $table->unsignedBigInteger('brand_id')->nullable();
            $table->unsignedBigInteger('unit_id')->nullable();
            $table->string('name', 191);
            $table->string('sku', 100)->nullable();
            $table->string('barcode', 100)->nullable();
            $table->string('hsn_code', 50)->nullable();
            $table->text('description')->nullable();
            $table->text('short_description')->nullable();
            $table->decimal('purchase_price', 12, 2)->default(0);
            $table->decimal('sale_price', 12, 2)->default(0);
            $table->decimal('mrp', 12, 2)->nullable();
            $table->decimal('default_profit_rate', 5, 2)->default(0);
            $table->unsignedBigInteger('tax_1_id')->nullable();
            $table->unsignedBigInteger('tax_2_id')->nullable();
            $table->decimal('min_stock_level', 12, 3)->default(0);
            $table->decimal('max_stock_level', 12, 3)->default(0);
            $table->boolean('is_batch_managed')->default(false);
            $table->boolean('can_be_sold')->default(true);
            $table->boolean('can_be_purchased')->default(true);
            $table->boolean('track_inventory')->default(true);
            $table->boolean('has_variants')->default(false);
            $table->boolean('is_featured')->default(false);
            $table->enum('status', ['draft', 'active', 'paused', 'out_of_stock', 'discontinued', 'archived'])->default('active');
            $table->timestamps();
            $table->unique('sku');
            $table->index('barcode');
            $table->index('category_id');
            $table->index('brand_id');
            $table->index('status');
            $table->index('is_featured');
        });

        Schema::table('products', function (Blueprint $table) {
            $table->foreign('category_id')->references('id')->on('product_categories')->onDelete('set null');
            $table->foreign('brand_id')->references('id')->on('brands')->onDelete('set null');
            $table->foreign('unit_id')->references('id')->on('units')->onDelete('set null');
        });
    }

    private function getTableColumns(string $table): array
    {
        $columns = [];
        $database = DB::connection()->getDatabaseName();
        $results = DB::select("SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?", [$database, $table]);
        foreach ($results as $row) {
            $columns[] = $row->COLUMN_NAME;
        }
        return $columns;
    }

    private function indexExists(string $table, string $indexName): bool
    {
        $database = DB::connection()->getDatabaseName();
        $result = DB::select("SELECT COUNT(*) as cnt FROM INFORMATION_SCHEMA.STATISTICS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ? AND INDEX_NAME = ?", [$database, $table, $indexName]);
        return $result[0]->cnt > 0;
    }

    private function addIndexIfNotExists(string $table, string $indexName, string $column, bool $unique = false): void
    {
        if (!$this->indexExists($table, $indexName)) {
            try {
                $indexType = $unique ? 'UNIQUE INDEX' : 'INDEX';
                DB::statement("ALTER TABLE `{$table}` ADD {$indexType} `{$indexName}` (`{$column}`)");
            } catch (\Exception $e) {}
        }
    }

    private function foreignKeyExists(string $table, string $column): bool
    {
        $database = DB::connection()->getDatabaseName();
        $result = DB::select("SELECT COUNT(*) as cnt FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ? AND COLUMN_NAME = ? AND REFERENCED_TABLE_NAME IS NOT NULL", [$database, $table, $column]);
        return $result[0]->cnt > 0;
    }

    private function addForeignKeyIfNotExists(string $table, string $column, string $refTable, string $refColumn, string $onDelete = 'RESTRICT'): void
    {
        if (!Schema::hasTable($refTable)) return;
        if (!$this->foreignKeyExists($table, $column)) {
            try {
                $fkName = "{$table}_{$column}_foreign";
                DB::statement("ALTER TABLE `{$table}` ADD CONSTRAINT `{$fkName}` FOREIGN KEY (`{$column}`) REFERENCES `{$refTable}` (`{$refColumn}`) ON DELETE {$onDelete}");
            } catch (\Exception $e) {}
        }
    }

    private function seedDefaultData(): void
    {
        if (Schema::hasTable('units') && DB::table('units')->count() === 0) {
            $now = now();
            DB::table('units')->insert([
                ['name' => 'Pieces', 'short_name' => 'PCS', 'base_unit_id' => null, 'conversion_factor' => 1, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now],
                ['name' => 'Grams', 'short_name' => 'G', 'base_unit_id' => null, 'conversion_factor' => 1, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now],
                ['name' => 'Kilograms', 'short_name' => 'KG', 'base_unit_id' => null, 'conversion_factor' => 1000, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now],
                ['name' => 'Milliliters', 'short_name' => 'ML', 'base_unit_id' => null, 'conversion_factor' => 1, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now],
                ['name' => 'Liters', 'short_name' => 'LTR', 'base_unit_id' => null, 'conversion_factor' => 1000, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now],
                ['name' => 'Meters', 'short_name' => 'M', 'base_unit_id' => null, 'conversion_factor' => 1, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now],
                ['name' => 'Dozen', 'short_name' => 'DZN', 'base_unit_id' => null, 'conversion_factor' => 12, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now],
                ['name' => 'Box', 'short_name' => 'BOX', 'base_unit_id' => null, 'conversion_factor' => 1, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now],
            ]);
        }

        if (Schema::hasTable('warehouses') && DB::table('warehouses')->count() === 0) {
            DB::table('warehouses')->insert(['name' => 'Main Warehouse', 'code' => 'WH-MAIN', 'type' => 'STORAGE', 'is_default' => true, 'is_active' => true, 'created_at' => now(), 'updated_at' => now()]);
        }

        if (Schema::hasTable('product_categories') && DB::table('product_categories')->count() === 0) {
            DB::table('product_categories')->insert(['code' => 'GEN', 'name' => 'General', 'is_active' => true, 'created_at' => now(), 'updated_at' => now()]);
        }

        if (Schema::hasTable('product_attributes') && DB::table('product_attributes')->count() === 0) {
            $now = now();
            $colorId = DB::table('product_attributes')->insertGetId(['name' => 'Color', 'slug' => 'color', 'type' => 'color', 'sort_order' => 1, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now]);
            $sizeId = DB::table('product_attributes')->insertGetId(['name' => 'Size', 'slug' => 'size', 'type' => 'select', 'sort_order' => 2, 'is_active' => true, 'created_at' => $now, 'updated_at' => $now]);

            DB::table('attribute_values')->insert([
                ['attribute_id' => $colorId, 'value' => 'Black', 'slug' => 'black', 'color_code' => '#000000', 'sort_order' => 1, 'created_at' => $now, 'updated_at' => $now],
                ['attribute_id' => $colorId, 'value' => 'White', 'slug' => 'white', 'color_code' => '#FFFFFF', 'sort_order' => 2, 'created_at' => $now, 'updated_at' => $now],
                ['attribute_id' => $colorId, 'value' => 'Red', 'slug' => 'red', 'color_code' => '#EF4444', 'sort_order' => 3, 'created_at' => $now, 'updated_at' => $now],
            ]);

            DB::table('attribute_values')->insert([
                ['attribute_id' => $sizeId, 'value' => 'S', 'slug' => 's', 'color_code' => null, 'sort_order' => 1, 'created_at' => $now, 'updated_at' => $now],
                ['attribute_id' => $sizeId, 'value' => 'M', 'slug' => 'm', 'color_code' => null, 'sort_order' => 2, 'created_at' => $now, 'updated_at' => $now],
                ['attribute_id' => $sizeId, 'value' => 'L', 'slug' => 'l', 'color_code' => null, 'sort_order' => 3, 'created_at' => $now, 'updated_at' => $now],
                ['attribute_id' => $sizeId, 'value' => 'XL', 'slug' => 'xl', 'color_code' => null, 'sort_order' => 4, 'created_at' => $now, 'updated_at' => $now],
            ]);
        }
    }
};
