<?php

namespace Modules\Inventory\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Modules\Inventory\Models\DeliveryReceipt;
use Modules\Inventory\Models\DeliveryReceiptItem;
use Modules\Inventory\Models\Warehouse;
use Modules\Inventory\Models\Rack;
use Modules\Inventory\Models\Product;
use Modules\Inventory\Models\Lot;
use Modules\Inventory\Models\StockLevel;
use Modules\Inventory\Models\StockMovement;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Traits\DataTable;

class DeliveryController extends BaseController
{
    use DataTable;

    /*
    |--------------------------------------------------------------------------
    | DataTable Configuration
    |--------------------------------------------------------------------------
    */
    
    protected $model = DeliveryReceipt::class;
    
    protected $with = ['invoice', 'customer', 'warehouse'];
    
    protected $searchable = ['receipt_number', 'invoice.invoice_number', 'customer.name'];
    
    protected $sortable = ['id', 'receipt_number', 'delivery_date', 'status', 'created_at'];
    
    protected $filterable = ['status', 'warehouse_id'];
    
    protected $defaultSort = 'id';
    
    protected $exportTitle = 'Delivery Receipts Export';

    /*
    |--------------------------------------------------------------------------
    | Permission Configuration
    |--------------------------------------------------------------------------
    */
    
    protected $viewPermission = 'inventory.delivery.read';
    protected $createPermission = 'inventory.delivery.create';
    protected $editPermission = 'inventory.delivery.edit';
    protected $deletePermission = 'inventory.delivery.delete';
    protected $approvePermission = 'inventory.delivery.approve';
    protected $exportPermission = 'inventory.delivery.export';

    /*
    |--------------------------------------------------------------------------
    | Index - List all delivery receipts
    |--------------------------------------------------------------------------
    */
    
    public function index()
    {
        $this->authorize('inventory.delivery.read');
        
        $stats = [
            'total' => DeliveryReceipt::count(),
            'pending' => DeliveryReceipt::pending()->count(),
            'approved' => DeliveryReceipt::approved()->count(),
            'delivered' => DeliveryReceipt::delivered()->count(),
            'rejected' => DeliveryReceipt::rejected()->count(),
        ];
        
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $permissions = $this->getDataTablePermissions();
        
        return view('inventory::delivery.index', compact('stats', 'warehouses', 'permissions'));
    }

    /*
    |--------------------------------------------------------------------------
    | DataTable Data Endpoint
    |--------------------------------------------------------------------------
    */
    
    public function data(Request $request)
    {
        $this->authorize('inventory.delivery.read');
        return $this->handleData($request);
    }

    /*
    |--------------------------------------------------------------------------
    | Custom Row Mapping for DataTable
    |--------------------------------------------------------------------------
    */
    
    protected function mapRow($item)
    {
        return [
            'id' => $item->id,
            'receipt_number' => $item->receipt_number,
            'invoice_number' => $item->invoice->invoice_number ?? '-',
            'invoice_id' => $item->invoice_id,
            'customer_name' => $item->customer->name ?? '-',
            'warehouse_name' => $item->warehouse->name ?? '-',
            'delivery_date' => $item->delivery_date ? $item->delivery_date->format('d M Y') : '-',
            'status' => $item->status,
            'status_label' => $item->status_label,
            'status_badge' => $item->status_badge,
            'total_items' => $item->items->count(),
            'total_qty' => number_format($item->items->sum('quantity'), 2),
            'created_by' => $item->created_by ?? '-',
            'approved_by' => $item->approved_by ?? '-',
            'created_at' => $item->created_at ? $item->created_at->format('d M Y H:i') : '-',
            'can_edit' => $item->canEdit(),
            'can_approve' => $item->canApprove(),
            'can_deliver' => $item->canMarkDelivered(),
            '_show_url' => route('inventory.delivery.show', $item->id),
            '_edit_url' => $item->canEdit() ? route('inventory.delivery.edit', $item->id) : null,
            '_delete_url' => $item->canEdit() ? route('inventory.delivery.destroy', $item->id) : null,
            '_print_url' => route('inventory.delivery.print', $item->id),
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Create - Show form
    |--------------------------------------------------------------------------
    */
    
    public function create(Request $request)
    {
        $this->authorize('inventory.delivery.create');
        
    $invoices = Invoice::with('customer')
        ->whereIn('status', ['sent', 'paid', 'partially_paid', 'unpaid'])
        ->whereNotIn('id', function($query) {
            $query->select('invoice_id')->from('delivery_receipts');
        })
        ->orderBy('id', 'desc')
        ->get();
        
        $invoice = null;
        $invoiceId = $request->get('invoice_id');
        if ($invoiceId) {
            $invoice = $invoices->firstWhere('id', $invoiceId);
        }
        
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $receiptNumber = DeliveryReceipt::generateReceiptNumber();
        
        return view('inventory::delivery.create', compact('invoices', 'invoice', 'warehouses', 'receiptNumber'));
    }

    /*
    |--------------------------------------------------------------------------
    | Store - Save new delivery receipt
    |--------------------------------------------------------------------------
    */
    
    public function store(Request $request)
    {
        $this->authorize('inventory.delivery.create');
        
        $validated = $request->validate([
            'invoice_id' => 'required|exists:invoices,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'rack_id' => 'nullable|exists:racks,id',
            'delivery_date' => 'required|date',
            'remarks' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|numeric|min:0.0001',
        ]);

        DB::beginTransaction();
        
        try {
            $invoice = Invoice::with('customer')->findOrFail($validated['invoice_id']);
            
            // Create delivery receipt
            $receipt = DeliveryReceipt::create([
                'receipt_number' => DeliveryReceipt::generateReceiptNumber(),
                'invoice_id' => $validated['invoice_id'],
                'customer_id' => $invoice->customer_id,
                'warehouse_id' => $validated['warehouse_id'],
                'rack_id' => $validated['rack_id'] ?? null,
                'delivery_date' => $validated['delivery_date'],
                'status' => DeliveryReceipt::STATUS_PENDING,
                'remarks' => $validated['remarks'] ?? null,
                'created_by' => Auth::guard('admin')->user()->name ?? 'System',
            ]);
            
            // Save items
            foreach ($validated['items'] as $item) {
                if (empty($item['product_id']) || floatval($item['quantity']) <= 0) {
                    continue;
                }
                
                $product = Product::find($item['product_id']);
                
                DeliveryReceiptItem::create([
                    'delivery_receipt_id' => $receipt->id,
                    'invoice_item_id' => $item['invoice_item_id'] ?? null,
                    'product_id' => $item['product_id'],
                    'variation_id' => $item['variation_id'] ?? null,
                    'unit_id' => $product->unit_id ?? null,
                    'description' => $item['description'] ?? $product->name,
                    'quantity' => $item['quantity'],
                    'delivered_qty' => $item['quantity'],
                ]);
            }
            
            DB::commit();
            
            return redirect()
                ->route('inventory.delivery.show', $receipt->id)
                ->with('success', 'Delivery receipt created successfully. Pending approval.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Delivery receipt creation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to create delivery receipt: ' . $e->getMessage());
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Show - View delivery receipt details
    |--------------------------------------------------------------------------
    */
    
    public function show($id)
    {
        $this->authorize('inventory.delivery.read');
        
        $receipt = DeliveryReceipt::with([
            'invoice.customer',
            'warehouse',
            'rack',
            'items.product',
            'items.variation',
            'items.unit'
        ])->findOrFail($id);
        
        $canApprove = Auth::guard('admin')->user()->can('inventory.delivery.approve');
        
        return view('inventory::delivery.show', compact('receipt', 'canApprove'));
    }

    /*
    |--------------------------------------------------------------------------
    | Edit - Show edit form
    |--------------------------------------------------------------------------
    */
    
    public function edit($id)
    {
        $this->authorize('inventory.delivery.edit');
        
        $receipt = DeliveryReceipt::with([
            'invoice.items.product',
            'items.product',
            'items.variation'
        ])->findOrFail($id);
        
        if (!$receipt->canEdit()) {
            return redirect()
                ->route('inventory.delivery.show', $id)
                ->with('error', 'This delivery receipt cannot be edited.');
        }
        
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $racks = Rack::where('warehouse_id', $receipt->warehouse_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();
        $invoice = $receipt->invoice;
        $invoiceItems = $invoice->items->where('item_type', 'product');
        
        return view('inventory::delivery.edit', compact('receipt', 'warehouses', 'racks', 'invoice', 'invoiceItems'));
    }

    /*
    |--------------------------------------------------------------------------
    | Update - Save changes
    |--------------------------------------------------------------------------
    */
    
    public function update(Request $request, $id)
    {
        $this->authorize('inventory.delivery.edit');
        
        $receipt = DeliveryReceipt::findOrFail($id);
        
        if (!$receipt->canEdit()) {
            return redirect()
                ->route('inventory.delivery.show', $id)
                ->with('error', 'This delivery receipt cannot be edited.');
        }
        
        $validated = $request->validate([
            'warehouse_id' => 'required|exists:warehouses,id',
            'rack_id' => 'nullable|exists:racks,id',
            'delivery_date' => 'required|date',
            'remarks' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|numeric|min:0.0001',
        ]);

        DB::beginTransaction();
        
        try {
            $receipt->update([
                'warehouse_id' => $validated['warehouse_id'],
                'rack_id' => $validated['rack_id'] ?? null,
                'delivery_date' => $validated['delivery_date'],
                'remarks' => $validated['remarks'] ?? null,
            ]);
            
            // Delete existing items and recreate
            $receipt->items()->delete();
            
            foreach ($validated['items'] as $item) {
                if (empty($item['product_id']) || floatval($item['quantity']) <= 0) {
                    continue;
                }
                
                $product = Product::find($item['product_id']);
                
                DeliveryReceiptItem::create([
                    'delivery_receipt_id' => $receipt->id,
                    'invoice_item_id' => $item['invoice_item_id'] ?? null,
                    'product_id' => $item['product_id'],
                    'variation_id' => $item['variation_id'] ?? null,
                    'unit_id' => $product->unit_id ?? null,
                    'description' => $item['description'] ?? $product->name,
                    'quantity' => $item['quantity'],
                    'delivered_qty' => $item['quantity'],
                ]);
            }
            
            DB::commit();
            
            return redirect()
                ->route('inventory.delivery.show', $receipt->id)
                ->with('success', 'Delivery receipt updated successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Delivery receipt update failed', ['error' => $e->getMessage()]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to update delivery receipt: ' . $e->getMessage());
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Destroy - Delete delivery receipt
    |--------------------------------------------------------------------------
    */
    
    public function destroy($id)
    {
        $this->authorize('inventory.delivery.delete');
        
        $receipt = DeliveryReceipt::find($id);
        
        if (!$receipt) {
            return response()->json(['success' => false, 'message' => 'Receipt not found'], 404);
        }
        
        if (!$receipt->canEdit()) {
            return response()->json(['success' => false, 'message' => 'Cannot delete approved/delivered receipt'], 403);
        }
        
        try {
            $receipt->delete();
            
            return response()->json(['success' => true, 'message' => 'Delivery receipt deleted successfully']);
            
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Approve - Admin approves and stock is deducted
    |--------------------------------------------------------------------------
    */
    
    public function approve(Request $request, $id)
    {
        $this->authorize('inventory.delivery.approve');
        
        $receipt = DeliveryReceipt::with(['items.product.unit', 'invoice'])->findOrFail($id);
        
        if (!$receipt->canApprove()) {
            return response()->json([
                'success' => false,
                'message' => 'This receipt cannot be approved.'
            ], 400);
        }

        DB::beginTransaction();
        
        try {
            // Update receipt status
            $receipt->update([
                'status' => DeliveryReceipt::STATUS_APPROVED,
                'approved_by' => Auth::guard('admin')->user()->name ?? 'Admin',
                'approved_at' => now(),
            ]);
            
            // Create stock movements (OUT) for each item
            foreach ($receipt->items as $item) {
                $product = $item->product;
                
                // Get current stock level
                $stockLevel = StockLevel::where('product_id', $item->product_id)
                    ->where('warehouse_id', $receipt->warehouse_id)
                    ->when($item->variation_id, fn($q) => $q->where('variation_id', $item->variation_id))
                    ->when($receipt->rack_id, fn($q) => $q->where('rack_id', $receipt->rack_id))
                    ->first();
                
                $stockBefore = $stockLevel->qty ?? 0;
                $stockAfter = $stockBefore - $item->delivered_qty;
                
                // Generate reference number
                $refNo = StockMovement::generateReferenceNo('DLV');
                
                // Create stock movement - use 'SALE' as reference_type (it's in ENUM)
                StockMovement::create([
                    'reference_no' => $refNo,
                    'product_id' => $item->product_id,
                    'variation_id' => $item->variation_id,
                    'warehouse_id' => $receipt->warehouse_id,
                    'rack_id' => $receipt->rack_id,
                    'lot_id' => null,
                    'unit_id' => $item->unit_id ?? $product->unit_id,
                    'qty' => $item->delivered_qty,
                    'base_qty' => $item->delivered_qty,
                    'stock_before' => $stockBefore,
                    'stock_after' => $stockAfter,
                    'movement_type' => 'OUT',
                    'reference_type' => 'SALE',  // IMPORTANT: Use SALE (DELIVERY not in ENUM)
                    'reference_id' => $receipt->id,
                    'reason' => 'Delivery Receipt: ' . $receipt->receipt_number,
                    'notes' => 'Invoice: ' . ($receipt->invoice->invoice_number ?? '-'),
                    'created_by' => Auth::guard('admin')->id(),
                ]);
                
                // Update stock level
                if ($stockLevel) {
                    $stockLevel->decrement('qty', $item->delivered_qty);
                } else {
                    // Create stock level with negative (shouldn't happen normally)
                    StockLevel::create([
                        'product_id' => $item->product_id,
                        'variation_id' => $item->variation_id,
                        'warehouse_id' => $receipt->warehouse_id,
                        'rack_id' => $receipt->rack_id,
                        'lot_id' => null,
                        'unit_id' => $product->unit_id,
                        'qty' => -$item->delivered_qty,
                    ]);
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Delivery receipt approved. Stock has been deducted.'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Delivery approval failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Approval failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Reject - Admin rejects the delivery receipt
    |--------------------------------------------------------------------------
    */
    
    public function reject(Request $request, $id)
    {
        $this->authorize('inventory.delivery.approve');
        
        $receipt = DeliveryReceipt::findOrFail($id);
        
        if (!$receipt->canApprove()) {
            return response()->json([
                'success' => false,
                'message' => 'This receipt cannot be rejected.'
            ], 400);
        }
        
        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:500',
        ]);

        try {
            $receipt->update([
                'status' => DeliveryReceipt::STATUS_REJECTED,
                'rejection_reason' => $validated['rejection_reason'],
                'approved_by' => Auth::guard('admin')->user()->name ?? 'Admin',
                'approved_at' => now(),
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Delivery receipt rejected.'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Rejection failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Mark Delivered - Final status after physical delivery
    |--------------------------------------------------------------------------
    */
    
    public function markDelivered(Request $request, $id)
    {
        $this->authorize('inventory.delivery.edit');
        
        $receipt = DeliveryReceipt::findOrFail($id);
        
        if (!$receipt->canMarkDelivered()) {
            return response()->json([
                'success' => false,
                'message' => 'This receipt cannot be marked as delivered.'
            ], 400);
        }

        try {
            $receipt->update([
                'status' => DeliveryReceipt::STATUS_DELIVERED,
                'delivered_by' => Auth::guard('admin')->user()->name ?? 'Staff',
                'delivered_at' => now(),
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Delivery receipt marked as delivered.'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Print - Generate printable view
    |--------------------------------------------------------------------------
    */
    
    public function print($id)
    {
        $this->authorize('inventory.delivery.read');
        
        $receipt = DeliveryReceipt::with([
            'invoice.customer',
            'warehouse',
            'rack',
            'items.product',
            'items.variation',
            'items.unit'
        ])->findOrFail($id);
        
        // Get company details
        $company = [
            'name' => \App\Models\Option::get('company_name', 'Your Company') ?? 'Your Company',
            'email' => \App\Models\Option::get('company_email', ''),
            'phone' => \App\Models\Option::get('company_phone', ''),
            'address' => \App\Models\Option::get('company_address', ''),
            'gst' => \App\Models\Option::get('company_gst', ''),
            'logo' => \App\Models\Option::get('company_logo', ''),
        ];
        
        return view('inventory::delivery.print', compact('receipt', 'company'));
    }

    /*
    |--------------------------------------------------------------------------
    | API: Get Invoice Items
    |--------------------------------------------------------------------------
    */
    
    public function getInvoiceItems($invoiceId)
    {
        $invoice = Invoice::with(['items.product', 'customer'])->find($invoiceId);
        
        if (!$invoice) {
            return response()->json(['success' => false, 'message' => 'Invoice not found'], 404);
        }
        
        $items = $invoice->items
            ->where('item_type', 'product')
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'product_id' => $item->product_id,
                    'product_name' => $item->product->name ?? $item->description,
                    'description' => $item->description,
                    'quantity' => $item->quantity,
                    'rate' => $item->rate,
                    'sku' => $item->product->sku ?? '',
                ];
            });
        
        return response()->json([
            'success' => true,
            'invoice' => [
                'id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'customer_name' => $invoice->customer->name ?? '-',
                'customer_id' => $invoice->customer_id,
            ],
            'items' => $items->values(),
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    | API: Get Racks by Warehouse
    |--------------------------------------------------------------------------
    */
    
    public function getRacksByWarehouse($warehouseId)
    {
        $racks = Rack::where('warehouse_id', $warehouseId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get(['id', 'name', 'code']);
        
        return response()->json([
            'success' => true,
            'racks' => $racks,
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    | API: Search Invoices
    |--------------------------------------------------------------------------
    */
    
    public function searchInvoices(Request $request)
    {
        $search = $request->get('q', '');
        
        $invoices = Invoice::with('customer')
            ->where(function ($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                  ->orWhereHas('customer', fn($q) => $q->where('name', 'like', "%{$search}%"));
            })
            ->whereIn('status', ['sent', 'paid', 'partially_paid'])
            ->latest()
            ->limit(20)
            ->get()
            ->map(function ($invoice) {
                return [
                    'id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'customer_name' => $invoice->customer->name ?? '-',
                    'date' => $invoice->date ? $invoice->date->format('d M Y') : '-',
                    'total' => number_format($invoice->total, 2),
                ];
            });
        
        return response()->json($invoices);
    }
}