<?php

namespace Modules\Inventory\Http\Controllers;

use Illuminate\Http\Request;
use Modules\Inventory\Models\Product;
use Modules\Inventory\Models\ProductCategory;
use Modules\Inventory\Models\Brand;
use Modules\Inventory\Models\Warehouse;
use Modules\Inventory\Models\Lot;
use Modules\Inventory\Models\StockLevel;
use Modules\Inventory\Models\StockMovement;
use Carbon\Carbon;
use DB;

class ReportController extends BaseController
{
    // ==================== 1. STOCK VALUATION REPORT ====================
    public function stockSummary(Request $request)
    {
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $categories = ProductCategory::where('is_active', true)->orderBy('name')->get();
        $brands = Brand::where('is_active', true)->orderBy('name')->get();

        $query = StockLevel::with(['product.category', 'product.brand', 'product.unit', 'warehouse'])
            ->where('qty', '>', 0);

        if ($request->warehouse_id) {
            $query->where('warehouse_id', $request->warehouse_id);
        }
        if ($request->category_id) {
            $query->whereHas('product', fn($q) => $q->where('category_id', $request->category_id));
        }
        if ($request->brand_id) {
            $query->whereHas('product', fn($q) => $q->where('brand_id', $request->brand_id));
        }

        $stockData = $query->get();

        $totalValue = 0;
        $totalQty = 0;
        $lowStockCount = 0;
        $categoryBreakdown = [];
        $warehouseBreakdown = [];

        $stockReport = $stockData->map(function ($item) use (&$totalValue, &$totalQty, &$lowStockCount, &$categoryBreakdown, &$warehouseBreakdown) {
            $value = $item->qty * ($item->product?->purchase_price ?? 0);
            $totalValue += $value;
            $totalQty += $item->qty;

            $reorderPoint = $item->product?->reorder_point ?? 10;
            if ($item->qty <= $reorderPoint) {
                $lowStockCount++;
            }

            $catName = $item->product?->category?->name ?? 'Uncategorized';
            if (!isset($categoryBreakdown[$catName])) {
                $categoryBreakdown[$catName] = ['qty' => 0, 'value' => 0];
            }
            $categoryBreakdown[$catName]['qty'] += $item->qty;
            $categoryBreakdown[$catName]['value'] += $value;

            $whName = $item->warehouse?->name ?? 'Unknown';
            if (!isset($warehouseBreakdown[$whName])) {
                $warehouseBreakdown[$whName] = ['qty' => 0, 'value' => 0];
            }
            $warehouseBreakdown[$whName]['qty'] += $item->qty;
            $warehouseBreakdown[$whName]['value'] += $value;

            $status = 'ok';
            if ($item->qty <= 0) $status = 'out';
            elseif ($item->qty <= $reorderPoint) $status = 'low';

            return (object) [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'product_name' => $item->product?->name ?? '-',
                'sku' => $item->product?->sku ?? '-',
                'category_name' => $catName,
                'brand_name' => $item->product?->brand?->name ?? '-',
                'warehouse_name' => $whName,
                'unit_name' => $item->product?->unit?->short_name ?? 'PCS',
                'qty' => $item->qty,
                'reorder_point' => $reorderPoint,
                'purchase_price' => $item->product?->purchase_price ?? 0,
                'sale_price' => $item->product?->sale_price ?? 0,
                'stock_value' => $value,
                'margin' => $item->product?->sale_price > 0 
                    ? round((($item->product->sale_price - $item->product->purchase_price) / $item->product->sale_price) * 100, 1) 
                    : 0,
                'status' => $status,
            ];
        });

        arsort($categoryBreakdown);
        arsort($warehouseBreakdown);

        $thirtyDaysAgo = Carbon::now()->subDays(30);
        $deadStockCount = Product::where('is_active', true)
            ->whereDoesntHave('stockMovements', fn($q) => $q->where('created_at', '>=', $thirtyDaysAgo))
            ->count();

        return view('inventory::reports.stock-summary', compact(
            'stockReport', 'totalValue', 'totalQty', 'lowStockCount', 'deadStockCount',
            'categoryBreakdown', 'warehouseBreakdown',
            'warehouses', 'categories', 'brands'
        ));
    }

    // ==================== 2. EXPIRY FORECAST REPORT ====================
    public function lotSummary(Request $request)
    {
        $products = Product::where('is_active', true)
            ->where('is_batch_managed', true)
            ->orderBy('name')
            ->get();

        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $days = $request->get('days', 30);

        $query = Lot::with(['product.unit', 'stockLevels.warehouse'])
            ->where('status', 'ACTIVE');

        if ($request->product_id) {
            $query->where('product_id', $request->product_id);
        }

        $lots = $query->orderBy('expiry_date', 'asc')->get();

        $expired = [];
        $critical = [];
        $warning = [];
        $upcoming = [];
        $safe = [];
        $expiredValue = 0;
        $atRiskValue = 0;

        foreach ($lots as $lot) {
            $stockQty = $lot->stockLevels->sum('qty');
            if ($stockQty <= 0) continue;

            $value = $stockQty * ($lot->purchase_price ?? $lot->product?->purchase_price ?? 0);
            $daysToExpiry = null;
            $expiryStatus = 'safe';

            if ($lot->expiry_date) {
                $daysToExpiry = Carbon::now()->startOfDay()->diffInDays($lot->expiry_date->startOfDay(), false);
                
                if ($daysToExpiry < 0) {
                    $expiryStatus = 'expired';
                    $expiredValue += $value;
                } elseif ($daysToExpiry <= 7) {
                    $expiryStatus = 'critical';
                    $atRiskValue += $value;
                } elseif ($daysToExpiry <= 30) {
                    $expiryStatus = 'warning';
                    $atRiskValue += $value;
                } elseif ($daysToExpiry <= 90) {
                    $expiryStatus = 'upcoming';
                }
            }

            $lotData = (object) [
                'id' => $lot->id,
                'lot_no' => $lot->lot_no,
                'batch_no' => $lot->batch_no,
                'product_name' => $lot->product?->name ?? '-',
                'product_sku' => $lot->product?->sku ?? '-',
                'unit_name' => $lot->product?->unit?->short_name ?? 'PCS',
                'qty' => $stockQty,
                'value' => $value,
                'manufacturing_date' => $lot->manufacturing_date?->format('d M Y'),
                'expiry_date' => $lot->expiry_date?->format('d M Y'),
                'days_to_expiry' => $daysToExpiry,
                'expiry_status' => $expiryStatus,
                'warehouse' => $lot->stockLevels->first()?->warehouse?->name ?? '-',
            ];

            switch ($expiryStatus) {
                case 'expired': $expired[] = $lotData; break;
                case 'critical': $critical[] = $lotData; break;
                case 'warning': $warning[] = $lotData; break;
                case 'upcoming': $upcoming[] = $lotData; break;
                default: $safe[] = $lotData;
            }
        }

        $stats = [
            'expired_count' => count($expired),
            'critical_count' => count($critical),
            'warning_count' => count($warning),
            'upcoming_count' => count($upcoming),
            'safe_count' => count($safe),
            'expired_value' => $expiredValue,
            'at_risk_value' => $atRiskValue,
        ];

        return view('inventory::reports.lot-summary', compact(
            'expired', 'critical', 'warning', 'upcoming', 'safe',
            'stats', 'products', 'warehouses', 'days'
        ));
    }

    // ==================== 3. ABC ANALYSIS REPORT ====================
    public function abcAnalysis(Request $request)
    {
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $categories = ProductCategory::where('is_active', true)->orderBy('name')->get();

        $query = StockLevel::with(['product.category', 'product.unit', 'warehouse'])
            ->where('qty', '>', 0);

        if ($request->warehouse_id) {
            $query->where('warehouse_id', $request->warehouse_id);
        }
        if ($request->category_id) {
            $query->whereHas('product', fn($q) => $q->where('category_id', $request->category_id));
        }

        // Group by product and calculate total value
        $stockData = $query->get()->groupBy('product_id')->map(function ($items) {
            $first = $items->first();
            $totalQty = $items->sum('qty');
            $unitPrice = $first->product?->purchase_price ?? 0;
            $totalValue = $totalQty * $unitPrice;
            
            return (object) [
                'product_id' => $first->product_id,
                'product_name' => $first->product?->name ?? '-',
                'sku' => $first->product?->sku ?? '-',
                'category_name' => $first->product?->category?->name ?? '-',
                'unit_name' => $first->product?->unit?->short_name ?? 'PCS',
                'total_qty' => $totalQty,
                'unit_price' => $unitPrice,
                'total_value' => $totalValue,
            ];
        })->sortByDesc('total_value')->values();

        $grandTotal = $stockData->sum('total_value');
        $totalItems = $stockData->count();

        // ABC Classification
        $runningTotal = 0;
        $classA = [];
        $classB = [];
        $classC = [];

        foreach ($stockData as $item) {
            $runningTotal += $item->total_value;
            $cumulativePercent = $grandTotal > 0 ? ($runningTotal / $grandTotal) * 100 : 0;
            $valuePercent = $grandTotal > 0 ? ($item->total_value / $grandTotal) * 100 : 0;
            
            $item->value_percent = round($valuePercent, 2);
            $item->cumulative_percent = round($cumulativePercent, 2);

            if ($cumulativePercent <= 80) {
                $item->abc_class = 'A';
                $classA[] = $item;
            } elseif ($cumulativePercent <= 95) {
                $item->abc_class = 'B';
                $classB[] = $item;
            } else {
                $item->abc_class = 'C';
                $classC[] = $item;
            }
        }

        $stats = [
            'total_value' => $grandTotal,
            'total_items' => $totalItems,
            'class_a_count' => count($classA),
            'class_a_value' => collect($classA)->sum('total_value'),
            'class_a_percent' => $grandTotal > 0 ? round((collect($classA)->sum('total_value') / $grandTotal) * 100, 1) : 0,
            'class_b_count' => count($classB),
            'class_b_value' => collect($classB)->sum('total_value'),
            'class_b_percent' => $grandTotal > 0 ? round((collect($classB)->sum('total_value') / $grandTotal) * 100, 1) : 0,
            'class_c_count' => count($classC),
            'class_c_value' => collect($classC)->sum('total_value'),
            'class_c_percent' => $grandTotal > 0 ? round((collect($classC)->sum('total_value') / $grandTotal) * 100, 1) : 0,
        ];

        return view('inventory::reports.abc-analysis', compact(
            'classA', 'classB', 'classC', 'stats', 'warehouses', 'categories'
        ));
    }

    // ==================== 4. DEAD STOCK REPORT ====================
    public function deadStock(Request $request)
    {
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $categories = ProductCategory::where('is_active', true)->orderBy('name')->get();
        $days = $request->get('days', 30);

        $cutoffDate = Carbon::now()->subDays($days);

        // Get products with stock but no recent movement
        $query = Product::with(['category', 'unit', 'stockLevels.warehouse'])
            ->where('is_active', true)
            ->whereHas('stockLevels', fn($q) => $q->where('qty', '>', 0));

        if ($request->category_id) {
            $query->where('category_id', $request->category_id);
        }

        $products = $query->get();

        $deadStock = [];
        $totalDeadValue = 0;

        foreach ($products as $product) {
            // Get last movement date
            $lastMovement = StockMovement::where('product_id', $product->id)
                ->orderBy('created_at', 'desc')
                ->first();

            $lastMovementDate = $lastMovement?->created_at;
            $daysSinceMovement = $lastMovementDate 
                ? Carbon::now()->diffInDays($lastMovementDate) 
                : 999; // No movement ever

            if ($daysSinceMovement >= $days) {
                $totalQty = $product->stockLevels->sum('qty');
                $stockValue = $totalQty * ($product->purchase_price ?? 0);
                $totalDeadValue += $stockValue;

                $warehouseStock = $product->stockLevels->map(fn($sl) => [
                    'warehouse' => $sl->warehouse?->name ?? '-',
                    'qty' => $sl->qty
                ])->filter(fn($ws) => $ws['qty'] > 0);

                $deadStock[] = (object) [
                    'product_id' => $product->id,
                    'product_name' => $product->name,
                    'sku' => $product->sku ?? '-',
                    'category_name' => $product->category?->name ?? '-',
                    'unit_name' => $product->unit?->short_name ?? 'PCS',
                    'total_qty' => $totalQty,
                    'unit_price' => $product->purchase_price ?? 0,
                    'stock_value' => $stockValue,
                    'last_movement' => $lastMovementDate?->format('d M Y') ?? 'Never',
                    'days_idle' => $daysSinceMovement == 999 ? 'Never moved' : $daysSinceMovement,
                    'warehouse_stock' => $warehouseStock,
                ];
            }
        }

        // Sort by value descending
        usort($deadStock, fn($a, $b) => $b->stock_value <=> $a->stock_value);

        $stats = [
            'total_dead_items' => count($deadStock),
            'total_dead_value' => $totalDeadValue,
            'days_threshold' => $days,
        ];

        return view('inventory::reports.dead-stock', compact(
            'deadStock', 'stats', 'warehouses', 'categories', 'days'
        ));
    }

    // ==================== 5. REORDER REPORT ====================
    public function reorder(Request $request)
    {
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $categories = ProductCategory::where('is_active', true)->orderBy('name')->get();

        $query = Product::with(['category', 'unit', 'stockLevels'])
            ->where('is_active', true);

        if ($request->category_id) {
            $query->where('category_id', $request->category_id);
        }

        $products = $query->get();

        $needsReorder = [];
        $totalReorderValue = 0;

        foreach ($products as $product) {
            $currentStock = $product->stockLevels->sum('qty');
            $reorderPoint = $product->reorder_point ?? 10;
            $reorderQty = $product->reorder_qty ?? 50;

            if ($currentStock <= $reorderPoint) {
                // Calculate avg daily usage (last 30 days)
                $thirtyDaysAgo = Carbon::now()->subDays(30);
                $totalOut = StockMovement::where('product_id', $product->id)
                    ->where('movement_type', 'OUT')
                    ->where('created_at', '>=', $thirtyDaysAgo)
                    ->sum('qty');
                
                $avgDailyUsage = abs($totalOut) / 30;
                $daysOfStock = $avgDailyUsage > 0 ? round($currentStock / $avgDailyUsage, 1) : 999;
                
                // Suggested order qty
                $suggestedQty = max($reorderQty, ceil($avgDailyUsage * 30)); // 30 days stock
                $orderValue = $suggestedQty * ($product->purchase_price ?? 0);
                $totalReorderValue += $orderValue;

                $status = 'low';
                if ($currentStock <= 0) $status = 'out';
                elseif ($currentStock <= $reorderPoint * 0.5) $status = 'critical';

                $needsReorder[] = (object) [
                    'product_id' => $product->id,
                    'product_name' => $product->name,
                    'sku' => $product->sku ?? '-',
                    'category_name' => $product->category?->name ?? '-',
                    'unit_name' => $product->unit?->short_name ?? 'PCS',
                    'current_stock' => $currentStock,
                    'reorder_point' => $reorderPoint,
                    'avg_daily_usage' => round($avgDailyUsage, 2),
                    'days_of_stock' => $daysOfStock,
                    'suggested_qty' => $suggestedQty,
                    'unit_price' => $product->purchase_price ?? 0,
                    'order_value' => $orderValue,
                    'status' => $status,
                ];
            }
        }

        // Sort by days of stock ascending (most urgent first)
        usort($needsReorder, fn($a, $b) => $a->days_of_stock <=> $b->days_of_stock);

        $stats = [
            'total_items' => count($needsReorder),
            'out_of_stock' => collect($needsReorder)->where('status', 'out')->count(),
            'critical' => collect($needsReorder)->where('status', 'critical')->count(),
            'low' => collect($needsReorder)->where('status', 'low')->count(),
            'total_reorder_value' => $totalReorderValue,
        ];

        return view('inventory::reports.reorder', compact(
            'needsReorder', 'stats', 'warehouses', 'categories'
        ));
    }

    // ==================== 6. STOCK AGING REPORT ====================
    public function stockAging(Request $request)
    {
        $warehouses = Warehouse::where('is_active', true)->orderBy('name')->get();
        $categories = ProductCategory::where('is_active', true)->orderBy('name')->get();

        $query = Product::with(['category', 'unit', 'stockLevels.warehouse'])
            ->where('is_active', true)
            ->whereHas('stockLevels', fn($q) => $q->where('qty', '>', 0));

        if ($request->warehouse_id) {
            $query->whereHas('stockLevels', fn($q) => $q->where('warehouse_id', $request->warehouse_id)->where('qty', '>', 0));
        }
        if ($request->category_id) {
            $query->where('category_id', $request->category_id);
        }

        $products = $query->get();

        $fresh = [];      // 0-30 days
        $normal = [];     // 31-60 days
        $aging = [];      // 61-90 days
        $stale = [];      // 90+ days

        $totalFreshValue = 0;
        $totalNormalValue = 0;
        $totalAgingValue = 0;
        $totalStaleValue = 0;

        foreach ($products as $product) {
            // Get last IN movement (receipt date)
            $lastReceipt = StockMovement::where('product_id', $product->id)
                ->where('movement_type', 'IN')
                ->orderBy('created_at', 'desc')
                ->first();

            $receiptDate = $lastReceipt?->created_at ?? $product->created_at;
            $ageInDays = Carbon::now()->diffInDays($receiptDate);

            $totalQty = $product->stockLevels->sum('qty');
            $stockValue = $totalQty * ($product->purchase_price ?? 0);

            $item = (object) [
                'product_id' => $product->id,
                'product_name' => $product->name,
                'sku' => $product->sku ?? '-',
                'category_name' => $product->category?->name ?? '-',
                'unit_name' => $product->unit?->short_name ?? 'PCS',
                'total_qty' => $totalQty,
                'unit_price' => $product->purchase_price ?? 0,
                'stock_value' => $stockValue,
                'last_receipt' => $receiptDate->format('d M Y'),
                'age_days' => $ageInDays,
            ];

            if ($ageInDays <= 30) {
                $item->age_class = 'fresh';
                $fresh[] = $item;
                $totalFreshValue += $stockValue;
            } elseif ($ageInDays <= 60) {
                $item->age_class = 'normal';
                $normal[] = $item;
                $totalNormalValue += $stockValue;
            } elseif ($ageInDays <= 90) {
                $item->age_class = 'aging';
                $aging[] = $item;
                $totalAgingValue += $stockValue;
            } else {
                $item->age_class = 'stale';
                $stale[] = $item;
                $totalStaleValue += $stockValue;
            }
        }

        $grandTotal = $totalFreshValue + $totalNormalValue + $totalAgingValue + $totalStaleValue;

        $stats = [
            'fresh_count' => count($fresh),
            'fresh_value' => $totalFreshValue,
            'fresh_percent' => $grandTotal > 0 ? round(($totalFreshValue / $grandTotal) * 100, 1) : 0,
            'normal_count' => count($normal),
            'normal_value' => $totalNormalValue,
            'normal_percent' => $grandTotal > 0 ? round(($totalNormalValue / $grandTotal) * 100, 1) : 0,
            'aging_count' => count($aging),
            'aging_value' => $totalAgingValue,
            'aging_percent' => $grandTotal > 0 ? round(($totalAgingValue / $grandTotal) * 100, 1) : 0,
            'stale_count' => count($stale),
            'stale_value' => $totalStaleValue,
            'stale_percent' => $grandTotal > 0 ? round(($totalStaleValue / $grandTotal) * 100, 1) : 0,
            'total_value' => $grandTotal,
        ];

        return view('inventory::reports.stock-aging', compact(
            'fresh', 'normal', 'aging', 'stale', 'stats', 'warehouses', 'categories'
        ));
    }
}