<?php

namespace Modules\Inventory\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\Invoice;
use App\Models\Customer;

class DeliveryReceipt extends Model
{
    protected $table = 'delivery_receipts';

    protected $fillable = [
        'receipt_number',
        'invoice_id',
        'customer_id',
        'warehouse_id',
        'rack_id',
        'delivery_date',
        'status',
        'remarks',
        'rejection_reason',
        'approved_by',
        'approved_at',
        'delivered_by',
        'delivered_at',
        'created_by',
    ];

    protected $casts = [
        'delivery_date' => 'date',
        'approved_at' => 'datetime',
        'delivered_at' => 'datetime',
    ];

    /**
     * Status constants
     */
    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';
    const STATUS_DELIVERED = 'delivered';

    /**
     * Get all statuses
     */
    public static function getStatuses(): array
    {
        return [
            self::STATUS_PENDING => 'Pending',
            self::STATUS_APPROVED => 'Approved',
            self::STATUS_REJECTED => 'Rejected',
            self::STATUS_DELIVERED => 'Delivered',
        ];
    }

    /**
     * Generate unique receipt number
     */
    public static function generateReceiptNumber(): string
    {
        $prefix = 'DR-' . date('Y') . '-';
        $prefixLength = strlen($prefix);
        
        $maxNumber = static::where('receipt_number', 'like', $prefix . '%')
            ->selectRaw('MAX(CAST(SUBSTRING(receipt_number, ?) AS UNSIGNED)) as max_num', [$prefixLength + 1])
            ->value('max_num');
        
        $nextNumber = ($maxNumber ?? 0) + 1;
        
        return $prefix . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Relationships
     */
    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function rack(): BelongsTo
    {
        return $this->belongsTo(Rack::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(DeliveryReceiptItem::class);
    }

    /**
     * Scopes
     */
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    public function scopeApproved($query)
    {
        return $query->where('status', self::STATUS_APPROVED);
    }

    public function scopeRejected($query)
    {
        return $query->where('status', self::STATUS_REJECTED);
    }

    public function scopeDelivered($query)
    {
        return $query->where('status', self::STATUS_DELIVERED);
    }

    /**
     * Check if receipt can be edited
     */
    public function canEdit(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Check if receipt can be approved
     */
    public function canApprove(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Check if receipt can be marked as delivered
     */
    public function canMarkDelivered(): bool
    {
        return $this->status === self::STATUS_APPROVED;
    }

    /**
     * Get status badge color
     */
    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            self::STATUS_PENDING => 'warning',
            self::STATUS_APPROVED => 'success',
            self::STATUS_REJECTED => 'danger',
            self::STATUS_DELIVERED => 'info',
            default => 'secondary',
        };
    }

    /**
     * Get status label
     */
    public function getStatusLabelAttribute(): string
    {
        return self::getStatuses()[$this->status] ?? ucfirst($this->status);
    }

    /**
     * Get total items count
     */
    public function getTotalItemsAttribute(): int
    {
        return $this->items->count();
    }

    /**
     * Get total quantity
     */
    public function getTotalQuantityAttribute(): float
    {
        return $this->items->sum('quantity');
    }

    /**
     * Get total delivered quantity
     */
    public function getTotalDeliveredQtyAttribute(): float
    {
        return $this->items->sum('delivered_qty');
    }
}