<?php

namespace Modules\Inventory\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class DeliveryReceiptItem extends Model
{
    protected $table = 'delivery_receipt_items';

    protected $fillable = [
        'delivery_receipt_id',
        'invoice_item_id',
        'product_id',
        'variation_id',
        'rack_id',
        'lot_id',
        'unit_id',
        'description',
        'quantity',
        'delivered_qty',
        'base_qty',
    ];

    protected $casts = [
        'quantity' => 'decimal:4',
        'delivered_qty' => 'decimal:4',
        'base_qty' => 'decimal:4',
    ];

    /**
     * Relationships
     */
    public function deliveryReceipt(): BelongsTo
    {
        return $this->belongsTo(DeliveryReceipt::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function variation(): BelongsTo
    {
        return $this->belongsTo(ProductVariation::class);
    }

    public function rack(): BelongsTo
    {
        return $this->belongsTo(Rack::class);
    }

    public function lot(): BelongsTo
    {
        return $this->belongsTo(Lot::class);
    }

    public function unit(): BelongsTo
    {
        return $this->belongsTo(Unit::class);
    }

    /**
     * Get product name with variation
     */
    public function getProductNameAttribute(): string
    {
        $name = $this->product->name ?? 'Unknown Product';
        
        if ($this->variation) {
            $name .= ' - ' . $this->variation->name;
        }
        
        return $name;
    }

    /**
     * Get unit short name
     */
    public function getUnitNameAttribute(): string
    {
        return $this->unit->short_name ?? $this->product->unit->short_name ?? 'PCS';
    }

    /**
     * Check if fully delivered
     */
    public function isFullyDelivered(): bool
    {
        return $this->delivered_qty >= $this->quantity;
    }

    /**
     * Get pending quantity
     */
    public function getPendingQtyAttribute(): float
    {
        return max(0, $this->quantity - $this->delivered_qty);
    }
}