<?php

namespace Modules\Inventory\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * ProductCategory Model - FIXED VERSION
 * 
 * Fixes:
 * - Added validation to prevent self-reference loops
 * - Added isDescendantOf() method
 * - Added getAncestors() method
 */
class ProductCategory extends Model
{
    protected $fillable = [
        'parent_id',
        'code',
        'name',
        'description',
        'sort_order',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    // ==================== RELATIONSHIPS ====================

    public function parent(): BelongsTo
    {
        return $this->belongsTo(ProductCategory::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(ProductCategory::class, 'parent_id');
    }

    public function products(): HasMany
    {
        return $this->hasMany(Product::class, 'category_id');
    }

    // ==================== ACCESSORS ====================

    /**
     * Get full category path (e.g., "Electronics > Mobile > Smartphones")
     */
    public function getFullPathAttribute(): string
    {
        $path = collect([$this->name]);
        $parent = $this->parent;
        
        while ($parent) {
            $path->prepend($parent->name);
            $parent = $parent->parent;
        }
        
        return $path->implode(' > ');
    }

    /**
     * Get depth level in hierarchy
     */
    public function getDepthAttribute(): int
    {
        $depth = 0;
        $parent = $this->parent;
        
        while ($parent) {
            $depth++;
            $parent = $parent->parent;
        }
        
        return $depth;
    }

    // ==================== VALIDATION METHODS ====================

    /**
     * Check if setting parent_id would create a circular reference
     * PREVENTS: Category A → B → C → A (loop!)
     */
    public function wouldCreateLoop(int $newParentId): bool
    {
        // Cannot be parent of itself
        if ($this->id === $newParentId) {
            return true;
        }

        // Check if new parent is a descendant of this category
        return $this->isAncestorOf($newParentId);
    }

    /**
     * Check if this category is an ancestor of given category ID
     */
    public function isAncestorOf(int $categoryId): bool
    {
        $descendantIds = $this->getAllDescendantIds();
        return in_array($categoryId, $descendantIds);
    }

    /**
     * Check if this category is a descendant of given category ID
     */
    public function isDescendantOf(int $categoryId): bool
    {
        $parent = $this->parent;
        
        while ($parent) {
            if ($parent->id === $categoryId) {
                return true;
            }
            $parent = $parent->parent;
        }
        
        return false;
    }

    /**
     * Get all descendant IDs (children, grandchildren, etc.)
     */
    public function getAllDescendantIds(): array
    {
        $ids = [];
        
        foreach ($this->children as $child) {
            $ids[] = $child->id;
            $ids = array_merge($ids, $child->getAllDescendantIds());
        }
        
        return $ids;
    }

    /**
     * Get all ancestors (parent, grandparent, etc.)
     */
    public function getAncestors(): \Illuminate\Support\Collection
    {
        $ancestors = collect();
        $parent = $this->parent;
        
        while ($parent) {
            $ancestors->push($parent);
            $parent = $parent->parent;
        }
        
        return $ancestors;
    }

    /**
     * Validate parent_id before saving
     */
    public function setParentIdAttribute($value): void
    {
        // Allow null (root category)
        if ($value === null) {
            $this->attributes['parent_id'] = null;
            return;
        }

        // If this is a new record (no ID yet), allow any parent
        if (!$this->exists) {
            $this->attributes['parent_id'] = $value;
            return;
        }

        // Prevent self-reference
        if ((int) $value === $this->id) {
            throw new \InvalidArgumentException('Category cannot be its own parent');
        }

        // Prevent circular reference
        if ($this->wouldCreateLoop((int) $value)) {
            throw new \InvalidArgumentException('Setting this parent would create a circular reference');
        }

        $this->attributes['parent_id'] = $value;
    }

    // ==================== SCOPES ====================

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('name');
    }

    // ==================== STATIC METHODS ====================

    /**
     * Get category tree for dropdowns
     */
    public static function getTree(?int $excludeId = null): array
    {
        $categories = self::query()
            ->when($excludeId, function ($q) use ($excludeId) {
                // Exclude the category and its descendants
                $category = self::find($excludeId);
                if ($category) {
                    $excludeIds = array_merge([$excludeId], $category->getAllDescendantIds());
                    $q->whereNotIn('id', $excludeIds);
                }
            })
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        return self::buildTree($categories);
    }

    /**
     * Build hierarchical tree from flat collection
     */
    private static function buildTree($categories, $parentId = null, $prefix = ''): array
    {
        $result = [];
        
        foreach ($categories->where('parent_id', $parentId) as $category) {
            $result[] = [
                'id' => $category->id,
                'name' => $prefix . $category->name,
                'code' => $category->code,
                'depth' => substr_count($prefix, '— '),
            ];
            
            $children = self::buildTree($categories, $category->id, $prefix . '— ');
            $result = array_merge($result, $children);
        }
        
        return $result;
    }
}