<?php

namespace Modules\Inventory\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Rack extends Model
{
    protected $fillable = [
        'warehouse_id',
        'code',
        'name',
        'zone',
        'aisle',
        'level',
        'storage_type',
        'max_capacity',
        'capacity_unit_id',
        'max_weight',
        'description',
        'is_active',
    ];

    protected $casts = [
        'max_capacity' => 'decimal:2',
        'max_weight' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    // Storage type options
    const STORAGE_TYPES = [
        'PALLET' => ['label' => 'Pallet Rack', 'icon' => '📦', 'color' => '#3b82f6'],
        'SHELF' => ['label' => 'Shelf', 'icon' => '🗄️', 'color' => '#8b5cf6'],
        'BIN' => ['label' => 'Bin/Box', 'icon' => '📥', 'color' => '#10b981'],
        'FLOOR' => ['label' => 'Floor Space', 'icon' => '⬜', 'color' => '#6b7280'],
        'COLD' => ['label' => 'Cold Storage', 'icon' => '❄️', 'color' => '#06b6d4'],
        'FREEZER' => ['label' => 'Freezer', 'icon' => '🧊', 'color' => '#0ea5e9'],
    ];

    // ==================== RELATIONSHIPS ====================

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function capacityUnit(): BelongsTo
    {
        return $this->belongsTo(Unit::class, 'capacity_unit_id');
    }

    public function stockLevels(): HasMany
    {
        return $this->hasMany(StockLevel::class);
    }

    public function stockMovements(): HasMany
    {
        return $this->hasMany(StockMovement::class);
    }

    // ==================== ACCESSORS ====================

    /**
     * Get full location path (Zone › Aisle › Level)
     */
    public function getFullLocationAttribute(): string
    {
        $parts = array_filter([
            $this->zone,
            $this->aisle ? "Aisle {$this->aisle}" : null,
            $this->level ? "Level {$this->level}" : null,
        ]);
        return $parts ? implode(' › ', $parts) : '-';
    }

    /**
     * Get full name with warehouse
     */
    public function getFullNameAttribute(): string
    {
        return ($this->warehouse?->name ?? 'Unknown') . ' > ' . $this->name . ' (' . $this->code . ')';
    }

    /**
     * Get capacity display with unit
     */
    public function getCapacityDisplayAttribute(): ?string
    {
        if (!$this->max_capacity) return null;
        
        $unit = $this->capacityUnit?->short_name ?? '';
        return number_format($this->max_capacity, 2) . ($unit ? ' ' . $unit : '');
    }

    /**
     * Get weight display
     */
    public function getWeightDisplayAttribute(): ?string
    {
        if (!$this->max_weight) return null;
        return number_format($this->max_weight, 2) . ' kg';
    }

    /**
     * Get storage type info
     */
    public function getStorageTypeInfoAttribute(): array
    {
        return self::STORAGE_TYPES[$this->storage_type] ?? self::STORAGE_TYPES['SHELF'];
    }

    // ==================== METHODS ====================

    /**
     * Get current stock in this rack
     */
    public function getCurrentStock($productId = null): float
    {
        $query = $this->stockLevels();
        
        if ($productId) {
            $query->where('product_id', $productId);
        }
        
        return $query->sum('qty') ?? 0;
    }

    /**
     * Check if rack has any stock
     */
    public function hasStock(): bool
    {
        return $this->stockLevels()->where('qty', '>', 0)->exists();
    }

    /**
     * Get stock count (number of different products)
     */
    public function getStockCountAttribute(): int
    {
        return $this->stockLevels()->where('qty', '>', 0)->count();
    }

    // ==================== SCOPES ====================

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeInWarehouse($query, $warehouseId)
    {
        return $query->where('warehouse_id', $warehouseId);
    }

    public function scopeInZone($query, $zone)
    {
        return $query->where('zone', $zone);
    }
}