<?php

namespace Modules\Inventory\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * Staff Model - References the staffs table
 * Used for warehouse contact person assignment
 */
class Staff extends Model
{
    protected $table = 'staffs';
    
    protected $fillable = [
        'admin_id',
        'first_name',
        'last_name',
        'phone',
        'employee_code',
        'designation',
        'status',
    ];

    protected $casts = [
        'status' => 'boolean',
    ];

    /**
     * Get the admin/user account for this staff
     */
    public function admin(): BelongsTo
    {
        // Try different possible model paths
        $modelPaths = [
            'App\Models\Admin\Admin',
            'App\Models\Admin',
            'App\Models\User',
        ];
        
        foreach ($modelPaths as $path) {
            if (class_exists($path)) {
                return $this->belongsTo($path, 'admin_id');
            }
        }
        
        // Fallback
        return $this->belongsTo('App\Models\Admin', 'admin_id');
    }

    /**
     * Get full name
     */
    public function getFullNameAttribute(): string
    {
        return trim(($this->first_name ?? '') . ' ' . ($this->last_name ?? ''));
    }

    /**
     * Get email from admin record
     */
    public function getEmailAttribute(): ?string
    {
        return $this->admin?->email;
    }

    /**
     * Scope for active staff only
     */
    public function scopeActive($query)
    {
        return $query->whereHas('admin', function($q) {
            $q->where('is_active', 1)->where('is_admin', 0);
        });
    }
}
