<?php

namespace Modules\Inventory\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Warehouse extends Model
{
    protected $fillable = [
        'name',
        'code',
        'address',
        'city',
        'state',
        'country',
        'contact_staff_id',
        'contact_person',
        'phone',
        'type',
        'is_default',
        'is_active',
    ];

    protected $casts = [
        'is_default' => 'boolean',
        'is_active' => 'boolean',
    ];

    /**
     * Get the staff member assigned as contact person
     */
    public function contactStaff(): BelongsTo
    {
        return $this->belongsTo(Staff::class, 'contact_staff_id');
    }

    /**
     * Get contact person name (from staff or fallback to text field)
     */
    public function getContactNameAttribute(): ?string
    {
        if ($this->contactStaff) {
            return $this->contactStaff->full_name;
        }
        return $this->contact_person;
    }

    /**
     * Get contact phone (from staff or fallback to text field)
     */
    public function getContactPhoneAttribute(): ?string
    {
        if ($this->contactStaff && $this->contactStaff->phone) {
            return $this->contactStaff->phone;
        }
        return $this->phone;
    }

    /**
     * Get contact email (from staff's admin record)
     */
    public function getContactEmailAttribute(): ?string
    {
        if ($this->contactStaff) {
            return $this->contactStaff->email;
        }
        return null;
    }

    public function racks(): HasMany
    {
        return $this->hasMany(Rack::class);
    }

    public function activeRacks(): HasMany
    {
        return $this->hasMany(Rack::class)->where('is_active', true);
    }

    public function stockMovements(): HasMany
    {
        return $this->hasMany(StockMovement::class);
    }

    public function stockLevels(): HasMany
    {
        return $this->hasMany(StockLevel::class);
    }

    public static function getDefault(): ?self
    {
        return static::where('is_default', true)->where('is_active', true)->first();
    }

    /**
     * Get total stock value in this warehouse
     */
    public function getTotalStockValue(): float
    {
        return $this->stockLevels()
            ->join('products', 'stock_levels.product_id', '=', 'products.id')
            ->selectRaw('SUM(stock_levels.qty * products.purchase_price) as total')
            ->value('total') ?? 0;
    }
}