<?php

namespace Modules\Inventory\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;

class InventoryServiceProvider extends ServiceProvider
{
    protected string $moduleName = 'Inventory';
    protected string $moduleNameLower = 'inventory';

    public function boot(): void
    {
        $this->registerConfig();
        $this->registerViews();
        $this->loadMigrationsFrom(module_path($this->moduleName, 'Database/Migrations'));
        
        // Auto-create storage link and inventory directories
        $this->ensureStorageSetup();
        
        // Create module roles and permissions
        $this->setupModuleRoles();
    }

    public function register(): void
    {
        $this->app->register(RouteServiceProvider::class);
        
        // Register console commands
        $this->commands([
            \Modules\Inventory\Console\Commands\CheckExpiryAlerts::class,
            \Modules\Inventory\Console\Commands\SendExpiryAlerts::class,
            \Modules\Inventory\Console\Commands\ManageModuleRoles::class,
        ]);
    }

    /**
     * Setup module-specific roles (permissions are created by ModuleController)
     */
    protected function setupModuleRoles(): void
    {
        // Only run during web requests (not console) and when tables exist
        if ($this->app->runningInConsole()) {
            return;
        }
        
        try {
            if (!Schema::hasTable('roles')) {
                return;
            }
            
            // Get Role model (Spatie)
            $roleClass = config('permission.models.role', \Spatie\Permission\Models\Role::class);
            
            // Get module roles from config
            $moduleRoles = config('inventory.roles', []);
            
            foreach ($moduleRoles as $roleConfig) {
                // Just create role if not exists - permissions are handled by ModuleController
                $roleClass::firstOrCreate(
                    ['name' => $roleConfig['name'], 'guard_name' => $roleConfig['guard'] ?? 'admin']
                );
            }
            
        } catch (\Exception $e) {
            // Silently fail - might be during installation
            Log::debug('Inventory module roles setup skipped: ' . $e->getMessage());
        }
    }

    /**
     * Ensure storage link exists and required directories are created
     */
    protected function ensureStorageSetup(): void
    {
        try {
            $publicStoragePath = public_path('storage');
            $storagePath = storage_path('app/public');
            
            // Create storage/app/public if not exists
            if (!File::isDirectory($storagePath)) {
                File::makeDirectory($storagePath, 0755, true);
            }
            
            // Create symlink if not exists (storage:link equivalent)
            if (!file_exists($publicStoragePath)) {
                @symlink($storagePath, $publicStoragePath);
            }
            
            // Create inventory directories
            $dirs = [
                $storagePath . '/inventory',
                $storagePath . '/inventory/products',
                $storagePath . '/inventory/barcodes',
            ];
            
            foreach ($dirs as $dir) {
                if (!File::isDirectory($dir)) {
                    File::makeDirectory($dir, 0755, true);
                }
            }
        } catch (\Exception $e) {
            // Silently fail - storage might already exist or permission issues
        }
    }

    protected function registerConfig(): void
    {
        $this->publishes([module_path($this->moduleName, 'Config/config.php') => config_path($this->moduleNameLower . '.php')], 'config');
        $this->mergeConfigFrom(module_path($this->moduleName, 'Config/config.php'), $this->moduleNameLower);
    }

    protected function registerViews(): void
    {
        $viewPath = resource_path('views/modules/' . $this->moduleNameLower);
        $sourcePath = module_path($this->moduleName, 'Resources/views');
        $this->publishes([$sourcePath => $viewPath], ['views', $this->moduleNameLower . '-module-views']);
        $this->loadViewsFrom(array_merge($this->getPublishableViewPaths(), [$sourcePath]), $this->moduleNameLower);
    }

    private function getPublishableViewPaths(): array
    {
        $paths = [];
        foreach (config('view.paths') as $path) {
            if (is_dir($path . '/modules/' . $this->moduleNameLower)) {
                $paths[] = $path . '/modules/' . $this->moduleNameLower;
            }
        }
        return $paths;
    }

    public function provides(): array
    {
        return [];
    }
    
    /**
     * Static method to uninstall module roles
     * Call this when uninstalling the module
     */
    public static function uninstallModuleRoles(): void
    {
        try {
            if (!Schema::hasTable('roles')) {
                return;
            }
            
            $roleClass = config('permission.models.role', \Spatie\Permission\Models\Role::class);
            
            // Get module roles from config
            $moduleRoles = config('inventory.roles', []);
            
            foreach ($moduleRoles as $roleConfig) {
                $roleClass::where('name', $roleConfig['name'])
                    ->where('guard_name', $roleConfig['guard'] ?? 'admin')
                    ->delete();
            }
            
            // Clear permission cache
            app(\Spatie\Permission\PermissionRegistrar::class)->forgetCachedPermissions();
            
        } catch (\Exception $e) {
            Log::error('Failed to uninstall Inventory module roles: ' . $e->getMessage());
        }
    }
}
