{{-- Modern Inventory Dashboard with Charts --}}

<style>
:root {
    --dash-primary: #6366f1;
    --dash-success: #10b981;
    --dash-warning: #f59e0b;
    --dash-danger: #ef4444;
    --dash-info: #06b6d4;
    --dash-purple: #8b5cf6;
    --shadow-sm: 0 1px 2px rgba(0,0,0,0.04);
    --shadow-md: 0 4px 12px rgba(0,0,0,0.08);
    --shadow-lg: 0 8px 24px rgba(0,0,0,0.12);
}

.dashboard {
    padding: 24px;
    max-width: 1600px;
    margin: 0 auto;
    background: var(--body-bg);
    min-height: 100vh;
}

/* Header */
.dash-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 32px;
    flex-wrap: wrap;
    gap: 16px;
}

.dash-header-left h1 {
    font-size: 32px;
    font-weight: 800;
    color: var(--text-primary);
    margin: 0;
    display: flex;
    align-items: center;
    gap: 14px;
}

.dash-header-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, var(--dash-primary), #4f46e5);
    border-radius: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    font-size: 28px;
    box-shadow: 0 4px 15px rgba(99, 102, 241, 0.4);
}

.dash-header-left p {
    margin: 8px 0 0 70px;
    color: var(--text-muted);
    font-size: 15px;
}

.dash-header-actions {
    display: flex;
    gap: 12px;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border-radius: 12px;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    border: none;
    cursor: pointer;
    transition: all 0.2s;
}

.btn-primary {
    background: linear-gradient(135deg, var(--dash-primary), #4f46e5);
    color: #fff;
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(99, 102, 241, 0.4);
    color: #fff;
}

.btn-secondary {
    background: var(--card-bg);
    color: var(--text-secondary);
    border: 2px solid var(--card-border);
}

.btn-secondary:hover {
    background: var(--body-bg);
    border-color: var(--text-muted);
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 20px;
    margin-bottom: 28px;
}

@media (max-width: 1200px) { .stats-grid { grid-template-columns: repeat(3, 1fr); } }
@media (max-width: 768px) { .stats-grid { grid-template-columns: repeat(2, 1fr); } }
@media (max-width: 480px) { .stats-grid { grid-template-columns: 1fr; } }

.stat-card {
    background: var(--card-bg);
    border-radius: 20px;
    padding: 24px;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--card-border);
    transition: all 0.3s;
    position: relative;
    overflow: hidden;
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-lg);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
}

.stat-card.blue::before { background: linear-gradient(90deg, #3b82f6, #60a5fa); }
.stat-card.green::before { background: linear-gradient(90deg, #10b981, #34d399); }
.stat-card.purple::before { background: linear-gradient(90deg, #8b5cf6, #a78bfa); }
.stat-card.orange::before { background: linear-gradient(90deg, #f97316, #fb923c); }
.stat-card.red::before { background: linear-gradient(90deg, #ef4444, #f87171); }

.stat-icon {
    width: 52px;
    height: 52px;
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 26px;
    margin-bottom: 16px;
}

.stat-card.blue .stat-icon { background: rgba(59, 130, 246, 0.15); }
.stat-card.green .stat-icon { background: rgba(16, 185, 129, 0.15); }
.stat-card.purple .stat-icon { background: rgba(139, 92, 246, 0.15); }
.stat-card.orange .stat-icon { background: rgba(249, 115, 22, 0.15); }
.stat-card.red .stat-icon { background: rgba(239, 68, 68, 0.15); }

.stat-value {
    font-size: 36px;
    font-weight: 800;
    color: var(--text-primary);
    line-height: 1;
    margin-bottom: 6px;
}

.stat-label {
    font-size: 13px;
    color: var(--text-muted);
    font-weight: 500;
}

.stat-badge {
    position: absolute;
    top: 16px;
    right: 16px;
    padding: 4px 10px;
    border-radius: 20px;
    font-size: 11px;
    font-weight: 700;
}

.stat-badge.alert {
    background: rgba(239, 68, 68, 0.15);
    color: #ef4444;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.6; }
}

/* Charts Row */
.charts-row {
    display: grid;
    grid-template-columns: 1fr 1fr 1fr;
    gap: 24px;
    margin-bottom: 28px;
}

@media (max-width: 1200px) { .charts-row { grid-template-columns: 1fr 1fr; } }
@media (max-width: 768px) { .charts-row { grid-template-columns: 1fr; } }

.chart-card {
    background: var(--card-bg);
    border-radius: 20px;
    padding: 24px;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--card-border);
}

.chart-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.chart-title {
    font-size: 16px;
    font-weight: 700;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: 10px;
}

.chart-title-icon {
    width: 36px;
    height: 36px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 18px;
}

.chart-container {
    position: relative;
    height: 220px;
}

.chart-center-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    text-align: center;
    pointer-events: none;
}

.chart-center-value {
    font-size: 28px;
    font-weight: 800;
    color: var(--text-primary);
}

.chart-center-label {
    font-size: 12px;
    color: var(--text-muted);
}

.chart-legend {
    display: flex;
    flex-wrap: wrap;
    gap: 12px;
    margin-top: 16px;
    padding-top: 16px;
    border-top: 1px solid var(--card-border);
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 13px;
    color: var(--text-secondary);
}

.legend-dot {
    width: 10px;
    height: 10px;
    border-radius: 50%;
}

/* Value Card - Dark themed (stays same in both modes) */
.value-card {
    background: linear-gradient(135deg, #1e293b, #334155);
    color: #fff;
    border-radius: 20px;
    padding: 28px;
    box-shadow: var(--shadow-lg);
    border: none;
}

.value-card .chart-title {
    color: #fff;
}

.value-card .chart-title-icon {
    background: rgba(255,255,255,0.15);
}

.value-amount {
    font-size: 42px;
    font-weight: 800;
    margin: 20px 0 8px;
}

.value-label {
    font-size: 14px;
    color: #94a3b8;
    margin-bottom: 24px;
}

.value-stats {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 16px;
}

.value-stat {
    background: rgba(255,255,255,0.1);
    border-radius: 12px;
    padding: 16px;
    text-align: center;
}

.value-stat-number {
    font-size: 24px;
    font-weight: 700;
}

.value-stat-label {
    font-size: 12px;
    color: #94a3b8;
    margin-top: 4px;
}

/* Quick Actions */
.quick-section {
    margin-bottom: 28px;
}

.section-header {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 16px;
}

.section-title {
    font-size: 18px;
    font-weight: 700;
    color: var(--text-primary);
}

.quick-grid {
    display: grid;
    grid-template-columns: repeat(8, 1fr);
    gap: 12px;
}

@media (max-width: 1200px) { .quick-grid { grid-template-columns: repeat(4, 1fr); } }
@media (max-width: 768px) { .quick-grid { grid-template-columns: repeat(2, 1fr); } }

.quick-btn {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 10px;
    padding: 20px 12px;
    background: var(--card-bg);
    border-radius: 16px;
    text-decoration: none;
    font-size: 12px;
    font-weight: 600;
    color: var(--text-secondary);
    box-shadow: var(--shadow-sm);
    border: 2px solid transparent;
    transition: all 0.2s;
}

.quick-btn:hover {
    transform: translateY(-3px);
    box-shadow: var(--shadow-md);
}

.quick-btn-icon {
    width: 48px;
    height: 48px;
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    color: #fff;
}

.quick-btn.blue .quick-btn-icon { background: linear-gradient(135deg, #3b82f6, #2563eb); }
.quick-btn.green .quick-btn-icon { background: linear-gradient(135deg, #10b981, #059669); }
.quick-btn.orange .quick-btn-icon { background: linear-gradient(135deg, #f97316, #ea580c); }
.quick-btn.purple .quick-btn-icon { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }
.quick-btn.cyan .quick-btn-icon { background: linear-gradient(135deg, #06b6d4, #0891b2); }
.quick-btn.teal .quick-btn-icon { background: linear-gradient(135deg, #14b8a6, #0d9488); }
.quick-btn.indigo .quick-btn-icon { background: linear-gradient(135deg, #6366f1, #4f46e5); }
.quick-btn.pink .quick-btn-icon { background: linear-gradient(135deg, #ec4899, #db2777); }

.quick-btn:hover.blue { border-color: #3b82f6; }
.quick-btn:hover.green { border-color: #10b981; }
.quick-btn:hover.orange { border-color: #f97316; }
.quick-btn:hover.purple { border-color: #8b5cf6; }
.quick-btn:hover.cyan { border-color: #06b6d4; }
.quick-btn:hover.teal { border-color: #14b8a6; }
.quick-btn:hover.indigo { border-color: #6366f1; }
.quick-btn:hover.pink { border-color: #ec4899; }

/* Tables Grid */
.tables-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 24px;
}

@media (max-width: 992px) { .tables-grid { grid-template-columns: 1fr; } }

.table-card {
    background: var(--card-bg);
    border-radius: 20px;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--card-border);
    overflow: hidden;
}

.table-header {
    padding: 20px 24px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 1px solid var(--card-border);
    background: var(--body-bg);
}

.table-title {
    font-size: 16px;
    font-weight: 700;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: 10px;
}

.table-title-icon {
    width: 36px;
    height: 36px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 18px;
}

.alert-badge {
    background: rgba(239, 68, 68, 0.15);
    color: #ef4444;
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 700;
    margin-left: 8px;
    animation: pulse 2s infinite;
}

.view-link {
    font-size: 13px;
    color: var(--dash-primary);
    text-decoration: none;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 4px;
}

.view-link:hover { text-decoration: underline; }

.table-body {
    max-height: 400px;
    overflow-y: auto;
}

.table-row {
    display: flex;
    align-items: center;
    padding: 16px 24px;
    border-bottom: 1px solid var(--card-border);
    transition: background 0.15s;
    cursor: pointer;
}

.table-row:hover { background: var(--body-bg); }
.table-row:last-child { border-bottom: none; }

.row-avatar {
    width: 44px;
    height: 44px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 18px;
    margin-right: 14px;
    flex-shrink: 0;
}

.row-avatar.in { background: rgba(16, 185, 129, 0.15); }
.row-avatar.out { background: rgba(239, 68, 68, 0.15); }
.row-avatar.transfer { background: rgba(139, 92, 246, 0.15); }
.row-avatar.return { background: rgba(6, 182, 212, 0.15); }
.row-avatar.adjust { background: rgba(20, 184, 166, 0.15); }
.row-avatar.warning { background: rgba(245, 158, 11, 0.15); }
.row-avatar.danger { background: rgba(239, 68, 68, 0.15); }
.row-avatar.critical { background: rgba(249, 115, 22, 0.15); }

.row-info { flex: 1; min-width: 0; }

.row-title {
    font-size: 14px;
    font-weight: 600;
    color: var(--text-primary);
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.row-subtitle {
    font-size: 12px;
    color: var(--text-muted);
    margin-top: 2px;
}

.row-meta {
    text-align: right;
    flex-shrink: 0;
}

.row-qty {
    font-size: 15px;
    font-weight: 700;
    color: var(--text-primary);
}

.row-qty.positive { color: #10b981; }
.row-qty.negative { color: #ef4444; }

.row-time {
    font-size: 11px;
    color: var(--text-muted);
    margin-top: 2px;
}

.stock-bar {
    width: 80px;
    height: 6px;
    background: var(--body-bg);
    border-radius: 3px;
    overflow: hidden;
    margin-top: 6px;
    border: 1px solid var(--card-border);
}

.stock-bar-fill {
    height: 100%;
    border-radius: 3px;
    transition: width 0.3s;
}

.stock-bar-fill.warning { background: linear-gradient(90deg, #f59e0b, #fbbf24); }
.stock-bar-fill.danger { background: #ef4444; }
.stock-bar-fill.critical { background: linear-gradient(90deg, #f97316, #fb923c); }

.empty-state {
    padding: 48px 24px;
    text-align: center;
    color: var(--text-muted);
}

.empty-state-icon {
    font-size: 48px;
    margin-bottom: 12px;
    opacity: 0.5;
}

.empty-state h4 {
    font-size: 16px;
    color: var(--text-secondary);
    margin: 0 0 4px;
}

.empty-state p {
    font-size: 13px;
    margin: 0;
}

.empty-state.success .empty-state-icon { color: #10b981; opacity: 1; }
.empty-state.success h4 { color: #10b981; }

/* Chart icon backgrounds using rgba for both modes */
.chart-title-icon[style*="eff6ff"] { background: rgba(59, 130, 246, 0.15) !important; }
.chart-title-icon[style*="fef2f2"] { background: rgba(239, 68, 68, 0.15) !important; }
.table-title-icon[style*="fef2f2"] { background: rgba(239, 68, 68, 0.15) !important; }
.table-title-icon[style*="eff6ff"] { background: rgba(59, 130, 246, 0.15) !important; }
</style>

<div class="dashboard">
    <!-- Header -->
    <div class="dash-header">
        <div class="dash-header-left">
            <h1>
                <div class="dash-header-icon">📦</div>
                {{ $greeting ?? 'Good Morning' }}!
            </h1>
            <p>Here's what's happening with your inventory today</p>
        </div>
        <div class="dash-header-actions">
            <a href="{{ route('inventory.stock.movements') }}" class="btn btn-secondary">📋 View History</a>
            <a href="{{ route('inventory.products.create') }}" class="btn btn-primary">➕ Add Product</a>
        </div>
    </div>

    <!-- Stats Grid -->
    <div class="stats-grid">
        <div class="stat-card blue">
            <div class="stat-icon">📦</div>
            <div class="stat-value">{{ number_format($stats['totalProducts'] ?? 0) }}</div>
            <div class="stat-label">Total Products</div>
        </div>
        <div class="stat-card green">
            <div class="stat-icon">🏭</div>
            <div class="stat-value">{{ number_format($stats['totalWarehouses'] ?? 0) }}</div>
            <div class="stat-label">Warehouses</div>
        </div>
        <div class="stat-card purple">
            <div class="stat-icon">🗄️</div>
            <div class="stat-value">{{ number_format($stats['totalRacks'] ?? 0) }}</div>
            <div class="stat-label">Storage Racks</div>
        </div>
        <div class="stat-card orange">
            <div class="stat-icon">🏷️</div>
            <div class="stat-value">{{ number_format($stats['totalCategories'] ?? 0) }}</div>
            <div class="stat-label">Categories</div>
        </div>
        <div class="stat-card red">
            <div class="stat-icon">⚠️</div>
            <div class="stat-value">{{ number_format($lowStockCount ?? 0) }}</div>
            <div class="stat-label">Low Stock Items</div>
            @if(($lowStockCount ?? 0) > 0)
                <span class="stat-badge alert">Alert!</span>
            @endif
        </div>
    </div>

    <!-- Charts Row -->
    <div class="charts-row">
        <!-- Stock Value Card -->
        <div class="value-card">
            <div class="chart-header">
                <div class="chart-title">
                    <div class="chart-title-icon">💰</div>
                    Inventory Value
                </div>
            </div>
            <div class="value-amount">₹{{ number_format($totalStockValue ?? 0, 0) }}</div>
            <div class="value-label">Total Stock Value (Purchase Price)</div>
            <div class="value-stats">
                <div class="value-stat">
                    <div class="value-stat-number">{{ number_format($totalStockQty ?? 0, 0) }}</div>
                    <div class="value-stat-label">Total Units</div>
                </div>
                <div class="value-stat">
                    <div class="value-stat-number">{{ $stats['totalBrands'] ?? 0 }}</div>
                    <div class="value-stat-label">Brands</div>
                </div>
            </div>
        </div>

        <!-- Today's Activity Donut -->
        <div class="chart-card">
            <div class="chart-header">
                <div class="chart-title">
                    <div class="chart-title-icon" style="background: linear-gradient(135deg, #eff6ff, #dbeafe);">⚡</div>
                    Today's Activity
                </div>
                <span style="font-size: 12px; color: #64748b;">{{ now()->format('d M Y') }}</span>
            </div>
            <div class="chart-container">
                <canvas id="activityChart"></canvas>
                <div class="chart-center-text">
                    <div class="chart-center-value">{{ ($todayIn ?? 0) + ($todayOut ?? 0) + ($todayTransfer ?? 0) + ($todayAdjust ?? 0) }}</div>
                    <div class="chart-center-label">Total</div>
                </div>
            </div>
            <div class="chart-legend">
                <div class="legend-item"><div class="legend-dot" style="background: #10b981;"></div> Received: {{ $todayIn ?? 0 }}</div>
                <div class="legend-item"><div class="legend-dot" style="background: #ef4444;"></div> Delivered: {{ $todayOut ?? 0 }}</div>
                <div class="legend-item"><div class="legend-dot" style="background: #8b5cf6;"></div> Transfers: {{ $todayTransfer ?? 0 }}</div>
                <div class="legend-item"><div class="legend-dot" style="background: #06b6d4;"></div> Adjustments: {{ $todayAdjust ?? 0 }}</div>
            </div>
        </div>

        <!-- Stock Status Donut -->
        <div class="chart-card">
            <div class="chart-header">
                <div class="chart-title">
                    <div class="chart-title-icon" style="background: linear-gradient(135deg, #fef2f2, #fee2e2);">📊</div>
                    Stock Status
                </div>
            </div>
            <div class="chart-container">
                <canvas id="stockStatusChart"></canvas>
                <div class="chart-center-text">
                    <div class="chart-center-value">{{ ($stockStatusSummary['total_low_stock'] ?? 0) }}</div>
                    <div class="chart-center-label">Alerts</div>
                </div>
            </div>
            <div class="chart-legend">
                <div class="legend-item"><div class="legend-dot" style="background: #ef4444;"></div> Out of Stock: {{ $stockStatusSummary['out_of_stock'] ?? 0 }}</div>
                <div class="legend-item"><div class="legend-dot" style="background: #f97316;"></div> Critical: {{ $stockStatusSummary['critical'] ?? 0 }}</div>
                <div class="legend-item"><div class="legend-dot" style="background: #f59e0b;"></div> Warning: {{ $stockStatusSummary['warning'] ?? 0 }}</div>
            </div>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="quick-section">
        <div class="section-header">
            <span class="section-title">⚡ Quick Actions</span>
        </div>
        <div class="quick-grid">
            <a href="{{ route('inventory.products.create') }}" class="quick-btn blue">
                <div class="quick-btn-icon">➕</div>
                Add Product
            </a>
            <a href="{{ route('inventory.stock.receive') }}" class="quick-btn green">
                <div class="quick-btn-icon">📥</div>
                Receive
            </a>
            <a href="{{ route('inventory.stock.deliver') }}" class="quick-btn orange">
                <div class="quick-btn-icon">📤</div>
                Deliver
            </a>
            <a href="{{ route('inventory.stock.transfer') }}" class="quick-btn purple">
                <div class="quick-btn-icon">🔄</div>
                Transfer
            </a>
            <a href="{{ route('inventory.stock.returns') }}" class="quick-btn cyan">
                <div class="quick-btn-icon">↩️</div>
                Return
            </a>
            <a href="{{ route('inventory.stock.adjustments') }}" class="quick-btn teal">
                <div class="quick-btn-icon">⚖️</div>
                Adjust
            </a>
            <a href="{{ route('inventory.reports.stock-summary') }}" class="quick-btn indigo">
                <div class="quick-btn-icon">📊</div>
                Reports
            </a>
            <a href="{{ route('inventory.settings.index') }}" class="quick-btn pink">
                <div class="quick-btn-icon">⚙️</div>
                Settings
            </a>
        </div>
    </div>

    <!-- Tables Grid -->
    <div class="tables-grid">
        <!-- Low Stock Alerts -->
        <div class="table-card">
            <div class="table-header">
                <div class="table-title">
                    <div class="table-title-icon" style="background: linear-gradient(135deg, #fef2f2, #fee2e2);">⚠️</div>
                    Low Stock Alerts
                    @if(($lowStockCount ?? 0) > 0)
                        <span class="alert-badge">{{ $lowStockCount }}</span>
                    @endif
                </div>
                <a href="{{ route('inventory.products.index') }}?filter=low_stock" class="view-link">View All →</a>
            </div>
            <div class="table-body">
                @forelse(($lowStockProducts ?? collect())->take(8) as $item)
                    @php
                        $percentage = $item->min_stock_level > 0 ? min(100, ($item->current_stock / $item->min_stock_level) * 100) : 0;
                        $statusClass = $item->current_stock <= 0 ? 'danger' : ($percentage <= 50 ? 'critical' : 'warning');
                        $statusIcon = $item->current_stock <= 0 ? '🔴' : ($percentage <= 50 ? '🟠' : '🟡');
                        $productUrl = isset($item->product_id) ? route('inventory.products.show', $item->product_id) : route('inventory.products.show', $item->id);
                    @endphp
                    <div class="table-row" onclick="window.location='{{ $productUrl }}'">
                        <div class="row-avatar {{ $statusClass }}">{{ $statusIcon }}</div>
                        <div class="row-info">
                            <div class="row-title">{{ $item->product_name ?? $item->name }}</div>
                            <div class="row-subtitle">{{ $item->sku ?? '' }} • Min: {{ $item->min_stock_level }}</div>
                            <div class="stock-bar">
                                <div class="stock-bar-fill {{ $statusClass }}" style="width: {{ $percentage }}%"></div>
                            </div>
                        </div>
                        <div class="row-meta">
                            <div class="row-qty {{ $item->current_stock <= 0 ? 'negative' : '' }}">{{ number_format($item->current_stock, 0) }}</div>
                            <div class="row-time">{{ $item->unit_name ?? 'PCS' }}</div>
                        </div>
                    </div>
                @empty
                    <div class="empty-state success">
                        <div class="empty-state-icon">✅</div>
                        <h4>All Good!</h4>
                        <p>No low stock alerts at the moment</p>
                    </div>
                @endforelse
            </div>
        </div>

        <!-- Recent Movements -->
        <div class="table-card">
            <div class="table-header">
                <div class="table-title">
                    <div class="table-title-icon" style="background: linear-gradient(135deg, #eff6ff, #dbeafe);">📋</div>
                    Recent Movements
                </div>
                <a href="{{ route('inventory.stock.movements') }}" class="view-link">View All →</a>
            </div>
            <div class="table-body">
                @forelse(($recentMovements ?? collect())->take(8) as $movement)
                    @php
                        $typeConfig = [
                            'IN' => ['icon' => '📥', 'class' => 'in', 'label' => 'Received'],
                            'OUT' => ['icon' => '📤', 'class' => 'out', 'label' => 'Delivered'],
                            'TRANSFER' => ['icon' => '🔄', 'class' => 'transfer', 'label' => 'Transfer'],
                            'RETURN' => ['icon' => '↩️', 'class' => 'return', 'label' => 'Return'],
                            'ADJUSTMENT' => ['icon' => '⚖️', 'class' => 'adjust', 'label' => 'Adjusted'],
                        ];
                        $config = $typeConfig[$movement->movement_type] ?? ['icon' => '📦', 'class' => 'in', 'label' => $movement->movement_type];
                        $isPositive = $movement->qty >= 0;
                    @endphp
                    <div class="table-row">
                        <div class="row-avatar {{ $config['class'] }}">{{ $config['icon'] }}</div>
                        <div class="row-info">
                            <div class="row-title">{{ $movement->product->name ?? 'Unknown' }}</div>
                            <div class="row-subtitle">{{ $movement->reference_no }} • {{ $movement->warehouse->name ?? '' }}</div>
                        </div>
                        <div class="row-meta">
                            <div class="row-qty {{ $isPositive ? 'positive' : 'negative' }}">
                                {{ $isPositive ? '+' : '' }}{{ number_format($movement->qty, 2) }}
                            </div>
                            <div class="row-time">{{ $movement->created_at->diffForHumans() }}</div>
                        </div>
                    </div>
                @empty
                    <div class="empty-state">
                        <div class="empty-state-icon">📋</div>
                        <h4>No Recent Activity</h4>
                        <p>Stock movements will appear here</p>
                    </div>
                @endforelse
            </div>
        </div>
    </div>
</div>

<!-- Chart.js -->
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Activity Donut Chart
    var activityCtx = document.getElementById('activityChart');
    if (activityCtx) {
        var totalActivity = {{ ($todayIn ?? 0) + ($todayOut ?? 0) + ($todayTransfer ?? 0) + ($todayAdjust ?? 0) }};
        new Chart(activityCtx, {
            type: 'doughnut',
            data: {
                labels: ['Received', 'Delivered', 'Transfers', 'Adjustments'],
                datasets: [{
                    data: totalActivity > 0 ? [{{ $todayIn ?? 0 }}, {{ $todayOut ?? 0 }}, {{ $todayTransfer ?? 0 }}, {{ $todayAdjust ?? 0 }}] : [1],
                    backgroundColor: totalActivity > 0 ? ['#10b981', '#ef4444', '#8b5cf6', '#06b6d4'] : ['#e2e8f0'],
                    borderWidth: 0,
                    cutout: '70%'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false }
                }
            }
        });
    }

    // Stock Status Donut Chart
    var stockCtx = document.getElementById('stockStatusChart');
    if (stockCtx) {
        var outOfStock = {{ $stockStatusSummary['out_of_stock'] ?? 0 }};
        var critical = {{ $stockStatusSummary['critical'] ?? 0 }};
        var warning = {{ $stockStatusSummary['warning'] ?? 0 }};
        var total = outOfStock + critical + warning;
        
        new Chart(stockCtx, {
            type: 'doughnut',
            data: {
                labels: ['Out of Stock', 'Critical', 'Warning'],
                datasets: [{
                    data: total > 0 ? [outOfStock, critical, warning] : [1],
                    backgroundColor: total > 0 ? ['#ef4444', '#f97316', '#f59e0b'] : ['#e2e8f0'],
                    borderWidth: 0,
                    cutout: '70%'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false }
                }
            }
        });
    }
});
</script>