@include('components.datatable')

<style>
    .page-container {
        padding: 20px;
        width: 100%;
        box-sizing: border-box;
    }

    .page-header {
        display: flex;
        align-items: center;
        gap: 16px;
        margin-bottom: 24px;
        max-width: 1000px;
        margin-left: auto;
        margin-right: auto;
    }
    
    .back-btn {
        width: 40px;
        height: 40px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: var(--card-bg);
        border: 1px solid var(--card-border);
        color: var(--text-muted);
        text-decoration: none;
        transition: all 0.2s;
        flex-shrink: 0;
    }
    
    .back-btn:hover {
        background: var(--body-bg);
        color: var(--text-primary);
    }
    
    .back-btn svg {
        width: 20px;
        height: 20px;
    }
    
    .page-header h1 {
        font-size: 24px;
        font-weight: 700;
        color: var(--text-primary);
        margin: 0;
    }

    .form-card {
        background: var(--card-bg);
        border: 1px solid var(--card-border);
        border-radius: 12px;
        max-width: 1000px;
        margin: 0 auto;
        width: 100%;
    }
    
    .form-card-header {
        padding: 20px 24px;
        border-bottom: 1px solid var(--card-border);
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    
    .form-card-title {
        font-size: 16px;
        font-weight: 600;
        color: var(--text-primary);
        margin: 0;
    }
    
    .receipt-number {
        font-family: monospace;
        font-size: 14px;
        font-weight: 600;
        color: var(--primary);
        background: var(--body-bg);
        padding: 6px 12px;
        border-radius: 6px;
    }
    
    .form-card-body {
        padding: 24px;
    }

    .form-section {
        margin-bottom: 32px;
        padding-bottom: 32px;
        border-bottom: 1px solid var(--card-border);
    }
    
    .form-section:last-child {
        margin-bottom: 0;
        padding-bottom: 0;
        border-bottom: none;
    }
    
    .form-section-title {
        font-size: 14px;
        font-weight: 600;
        color: var(--text-muted);
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 20px;
    }

    .form-row {
        display: grid;
        grid-template-columns: repeat(4, 1fr);
        gap: 20px;
    }
    
    .form-row.two-col {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .form-row.three-col {
        grid-template-columns: repeat(3, 1fr);
    }
    
    @media (max-width: 768px) {
        .form-row {
            grid-template-columns: 1fr;
        }
    }

    .form-group {
        margin-bottom: 20px;
    }
    
    .form-group:last-child {
        margin-bottom: 0;
    }
    
    .form-group.full-width {
        grid-column: 1 / -1;
    }
    
    .form-label {
        display: block;
        font-size: 13px;
        font-weight: 500;
        color: var(--text-primary);
        margin-bottom: 8px;
    }
    
    .form-label .required {
        color: #ef4444;
    }
    
    .form-control {
        width: 100%;
        padding: 10px 14px;
        border: 1px solid var(--card-border);
        border-radius: 8px;
        font-size: 14px;
        background: var(--card-bg);
        color: var(--text-primary);
        transition: border-color 0.2s, box-shadow 0.2s;
        box-sizing: border-box;
    }
    
    .form-control:focus {
        outline: none;
        border-color: var(--primary);
        box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
    }
    
    .form-control::placeholder {
        color: var(--text-muted);
    }
    
    .form-control:disabled {
        background: var(--body-bg);
        cursor: not-allowed;
    }
    
    textarea.form-control {
        min-height: 80px;
        resize: vertical;
    }
    
    .form-help {
        font-size: 12px;
        color: var(--text-muted);
        margin-top: 6px;
    }

    /* Items Table */
    .items-table {
        width: 100%;
        border-collapse: collapse;
    }
    
    .items-table th,
    .items-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid var(--card-border);
    }
    
    .items-table th {
        background: var(--body-bg);
        font-size: 12px;
        font-weight: 600;
        color: var(--text-muted);
        text-transform: uppercase;
    }
    
    .items-table td {
        font-size: 14px;
        color: var(--text-primary);
    }
    
    .items-table input {
        width: 100%;
        padding: 8px 10px;
        border: 1px solid var(--card-border);
        border-radius: 6px;
        font-size: 13px;
        background: var(--card-bg);
        color: var(--text-primary);
    }
    
    .items-table input:focus {
        outline: none;
        border-color: var(--primary);
    }
    
    .product-cell {
        display: flex;
        flex-direction: column;
        gap: 2px;
    }
    
    .product-name {
        font-weight: 500;
    }
    
    .product-sku {
        font-size: 12px;
        color: var(--text-muted);
        font-family: monospace;
    }

    /* Form Actions */
    .form-actions {
        display: flex;
        justify-content: flex-end;
        gap: 12px;
        margin-top: 24px;
        padding-top: 24px;
        border-top: 1px solid var(--card-border);
    }
    
    .btn {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        padding: 12px 24px;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
        border: none;
        text-decoration: none;
    }
    
    .btn-primary {
        background: linear-gradient(135deg, var(--primary), var(--primary-hover));
        color: #fff;
        box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
    }
    
    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 16px rgba(59, 130, 246, 0.4);
    }
    
    .btn-secondary {
        background: var(--body-bg);
        color: var(--text-primary);
        border: 1px solid var(--card-border);
    }
    
    .btn-secondary:hover {
        background: var(--card-border);
    }
    
    .btn svg {
        width: 18px;
        height: 18px;
    }

    /* Alert Box */
    .alert {
        padding: 12px 16px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-size: 14px;
    }
    
    .alert-warning {
        background: #fef3c7;
        color: #92400e;
        border: 1px solid #fcd34d;
    }

    #itemsSection {
        display: none;
    }
    
    #itemsSection.show {
        display: block;
    }
</style>

<div class="page-container">
    <!-- Header -->
    <div class="page-header">
        <a href="{{ route('inventory.delivery.index') }}" class="back-btn">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M15 19l-7-7 7-7"/>
            </svg>
        </a>
        <h1>Create Delivery Receipt</h1>
    </div>

    @if(session('error'))
        <div class="alert alert-warning" style="max-width: 1000px; margin: 0 auto 20px;">
            {{ session('error') }}
        </div>
    @endif

    <form action="{{ route('inventory.delivery.store') }}" method="POST" id="deliveryForm">
        @csrf
        
        <div class="form-card">
            <div class="form-card-header">
                <h2 class="form-card-title">Delivery Receipt Details</h2>
                <span class="receipt-number">{{ $receiptNumber }}</span>
            </div>
            
            <div class="form-card-body">
                <!-- Select Invoice Section -->
                <div class="form-section">
                    <div class="form-section-title">Select Invoice</div>
                    <div class="form-row two-col">
                        <div class="form-group">
                            <label class="form-label">Invoice <span class="required">*</span></label>
                            <select name="invoice_id" id="invoiceSelect" class="form-control" required>
                                <option value="">-- Select Invoice --</option>
                                @foreach($invoices as $inv)
                                    <option value="{{ $inv->id }}" 
                                            data-customer="{{ $inv->customer->name ?? 'N/A' }}"
                                            {{ (isset($invoice) && $invoice->id == $inv->id) ? 'selected' : '' }}>
                                        {{ $inv->invoice_number }} - {{ $inv->customer->name ?? 'N/A' }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Customer</label>
                            <input type="text" id="customerDisplay" class="form-control" readonly placeholder="Select invoice first">
                        </div>
                    </div>
                </div>

                <!-- Delivery Details -->
                <div class="form-section">
                    <div class="form-section-title">Delivery Information</div>
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Warehouse <span class="required">*</span></label>
                            <select name="warehouse_id" id="warehouseSelect" class="form-control" required>
                                <option value="">Select Warehouse</option>
                                @foreach($warehouses as $warehouse)
                                    <option value="{{ $warehouse->id }}" {{ old('warehouse_id') == $warehouse->id ? 'selected' : '' }}>
                                        {{ $warehouse->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Rack (Optional)</label>
                            <select name="rack_id" id="rackSelect" class="form-control">
                                <option value="">Select Rack</option>
                            </select>
                            <div class="form-help">Select warehouse first to load racks</div>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Delivery Date <span class="required">*</span></label>
                            <input type="date" name="delivery_date" class="form-control" 
                                   value="{{ old('delivery_date', date('Y-m-d')) }}" required>
                        </div>
                    </div>
                    <div class="form-group full-width">
                        <label class="form-label">Remarks</label>
                        <textarea name="remarks" class="form-control" rows="2" placeholder="Optional notes about this delivery...">{{ old('remarks') }}</textarea>
                    </div>
                </div>

                <!-- Items Section (Hidden until invoice selected) -->
                <div class="form-section" id="itemsSection">
                    <div class="form-section-title">Items to Deliver</div>
                    
                    <div style="overflow-x: auto;">
                        <table class="items-table">
                            <thead>
                                <tr>
                                    <th style="width: 40px;">#</th>
                                    <th>Product</th>
                                    <th style="width: 120px;">Invoice Qty</th>
                                    <th style="width: 120px;">Deliver Qty</th>
                                </tr>
                            </thead>
                            <tbody id="itemsBody">
                                <!-- Items will be loaded via AJAX -->
                            </tbody>
                        </table>
                    </div>
                    
                    <div id="noItemsMessage" class="alert alert-warning" style="display: none; margin-top: 16px;">
                        No product items found in this invoice.
                    </div>
                </div>

                <!-- Form Actions -->
                <div class="form-actions">
                    <a href="{{ route('inventory.delivery.index') }}" class="btn btn-secondary">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/>
                        </svg>
                        Cancel
                    </a>
                    <button type="submit" class="btn btn-primary" id="submitBtn" disabled>
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7"/>
                        </svg>
                        Create Delivery Receipt
                    </button>
                </div>
            </div>
        </div>
    </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    var invoiceSelect = document.getElementById('invoiceSelect');
    var customerDisplay = document.getElementById('customerDisplay');
    var warehouseSelect = document.getElementById('warehouseSelect');
    var rackSelect = document.getElementById('rackSelect');
    var itemsSection = document.getElementById('itemsSection');
    var itemsBody = document.getElementById('itemsBody');
    var noItemsMessage = document.getElementById('noItemsMessage');
    var submitBtn = document.getElementById('submitBtn');
    
    // Load racks when warehouse changes
    warehouseSelect.addEventListener('change', function() {
        var warehouseId = this.value;
        rackSelect.innerHTML = '<option value="">Loading...</option>';
        
        if (!warehouseId) {
            rackSelect.innerHTML = '<option value="">Select Rack</option>';
            return;
        }
        
        fetch('/admin/inventory/racks/by-warehouse/' + warehouseId)
            .then(response => response.json())
            .then(data => {
                var options = '<option value="">Select Rack</option>';
                if (data.racks && data.racks.length > 0) {
                    data.racks.forEach(function(rack) {
                        options += '<option value="' + rack.id + '">' + rack.name + (rack.code ? ' (' + rack.code + ')' : '') + '</option>';
                    });
                }
                rackSelect.innerHTML = options;
            })
            .catch(error => {
                rackSelect.innerHTML = '<option value="">Select Rack</option>';
            });
    });
    
    // When invoice changes, load items
    invoiceSelect.addEventListener('change', function() {
        var invoiceId = this.value;
        var selectedOption = this.options[this.selectedIndex];
        
        if (!invoiceId) {
            customerDisplay.value = '';
            itemsSection.classList.remove('show');
            itemsBody.innerHTML = '';
            submitBtn.disabled = true;
            return;
        }
        
        // Show customer name
        customerDisplay.value = selectedOption.dataset.customer || '';
        
        // Load invoice items via AJAX
        fetch('/admin/inventory/delivery/invoice/' + invoiceId + '/items')
            .then(response => response.json())
            .then(data => {
                if (data.success && data.items.length > 0) {
                    var html = '';
                    data.items.forEach(function(item, index) {
                        html += '<tr>';
                        html += '<td>' + (index + 1) + '</td>';
                        html += '<td>';
                        html += '<div class="product-cell">';
                        html += '<span class="product-name">' + item.product_name + '</span>';
                        if (item.sku) {
                            html += '<span class="product-sku">' + item.sku + '</span>';
                        }
                        html += '</div>';
                        html += '<input type="hidden" name="items[' + index + '][invoice_item_id]" value="' + item.id + '">';
                        html += '<input type="hidden" name="items[' + index + '][product_id]" value="' + item.product_id + '">';
                        html += '<input type="hidden" name="items[' + index + '][description]" value="' + item.description + '">';
                        html += '</td>';
                        html += '<td>' + parseFloat(item.quantity).toFixed(2) + '</td>';
                        html += '<td><input type="number" name="items[' + index + '][quantity]" value="' + item.quantity + '" min="0" max="' + item.quantity + '" step="0.01" required></td>';
                        html += '</tr>';
                    });
                    
                    itemsBody.innerHTML = html;
                    itemsSection.classList.add('show');
                    noItemsMessage.style.display = 'none';
                    submitBtn.disabled = false;
                } else {
                    itemsBody.innerHTML = '';
                    itemsSection.classList.add('show');
                    noItemsMessage.style.display = 'block';
                    submitBtn.disabled = true;
                }
            })
            .catch(error => {
                console.error('Error loading invoice items:', error);
                itemsBody.innerHTML = '<tr><td colspan="4" style="color: red;">Error loading items</td></tr>';
                itemsSection.classList.add('show');
                submitBtn.disabled = true;
            });
    });
    
    // If invoice is pre-selected (from URL param), trigger change
    if (invoiceSelect.value) {
        invoiceSelect.dispatchEvent(new Event('change'));
    }
});
</script>