@include('components.datatable')

<style>
    .page-container {
        padding: 20px;
        width: 100%;
        box-sizing: border-box;
    }

    .page-header {
        display: flex;
        align-items: center;
        gap: 16px;
        margin-bottom: 24px;
        max-width: 1000px;
        margin-left: auto;
        margin-right: auto;
    }
    
    .back-btn {
        width: 40px;
        height: 40px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: var(--card-bg);
        border: 1px solid var(--card-border);
        color: var(--text-muted);
        text-decoration: none;
        transition: all 0.2s;
        flex-shrink: 0;
    }
    
    .back-btn:hover {
        background: var(--body-bg);
        color: var(--text-primary);
    }
    
    .back-btn svg {
        width: 20px;
        height: 20px;
    }
    
    .page-header h1 {
        font-size: 24px;
        font-weight: 700;
        color: var(--text-primary);
        margin: 0;
    }

    .form-card {
        background: var(--card-bg);
        border: 1px solid var(--card-border);
        border-radius: 12px;
        max-width: 1000px;
        margin: 0 auto;
        width: 100%;
    }
    
    .form-card-header {
        padding: 20px 24px;
        border-bottom: 1px solid var(--card-border);
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    
    .form-card-title {
        font-size: 16px;
        font-weight: 600;
        color: var(--text-primary);
        margin: 0;
    }
    
    .receipt-number {
        font-family: monospace;
        font-size: 14px;
        font-weight: 600;
        color: var(--primary);
        background: var(--body-bg);
        padding: 6px 12px;
        border-radius: 6px;
    }
    
    .form-card-body {
        padding: 24px;
    }

    .form-section {
        margin-bottom: 32px;
        padding-bottom: 32px;
        border-bottom: 1px solid var(--card-border);
    }
    
    .form-section:last-child {
        margin-bottom: 0;
        padding-bottom: 0;
        border-bottom: none;
    }
    
    .form-section-title {
        font-size: 14px;
        font-weight: 600;
        color: var(--text-muted);
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 20px;
    }

    .form-row {
        display: grid;
        grid-template-columns: repeat(3, 1fr);
        gap: 20px;
    }
    
    @media (max-width: 768px) {
        .form-row {
            grid-template-columns: 1fr;
        }
    }

    .form-group {
        margin-bottom: 20px;
    }
    
    .form-group.full-width {
        grid-column: 1 / -1;
    }
    
    .form-label {
        display: block;
        font-size: 13px;
        font-weight: 500;
        color: var(--text-primary);
        margin-bottom: 8px;
    }
    
    .form-label .required {
        color: #ef4444;
    }
    
    .form-control {
        width: 100%;
        padding: 10px 14px;
        border: 1px solid var(--card-border);
        border-radius: 8px;
        font-size: 14px;
        background: var(--card-bg);
        color: var(--text-primary);
        transition: border-color 0.2s, box-shadow 0.2s;
        box-sizing: border-box;
    }
    
    .form-control:focus {
        outline: none;
        border-color: var(--primary);
        box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
    }
    
    .form-control:disabled {
        background: var(--body-bg);
        cursor: not-allowed;
    }
    
    textarea.form-control {
        min-height: 80px;
        resize: vertical;
    }

    /* Invoice Info Box */
    .invoice-info-box {
        background: linear-gradient(135deg, #dbeafe 0%, #ede9fe 100%);
        border-radius: 10px;
        padding: 16px 20px;
        margin-bottom: 24px;
        display: flex;
        align-items: center;
        gap: 16px;
    }
    
    .invoice-info-box .icon {
        width: 48px;
        height: 48px;
        background: #fff;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: var(--primary);
    }
    
    .invoice-info-box .icon svg {
        width: 24px;
        height: 24px;
    }
    
    .invoice-info-box .details h3 {
        font-size: 16px;
        font-weight: 600;
        color: #1e3a5f;
        margin: 0 0 4px 0;
    }
    
    .invoice-info-box .details p {
        font-size: 13px;
        color: #475569;
        margin: 0;
    }

    /* Items Table */
    .items-table {
        width: 100%;
        border-collapse: collapse;
    }
    
    .items-table th,
    .items-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid var(--card-border);
    }
    
    .items-table th {
        background: var(--body-bg);
        font-size: 12px;
        font-weight: 600;
        color: var(--text-muted);
        text-transform: uppercase;
    }
    
    .items-table td {
        font-size: 14px;
        color: var(--text-primary);
    }
    
    .items-table input,
    .items-table select {
        width: 100%;
        padding: 8px 10px;
        border: 1px solid var(--card-border);
        border-radius: 6px;
        font-size: 13px;
        background: var(--card-bg);
        color: var(--text-primary);
    }
    
    .items-table input:focus,
    .items-table select:focus {
        outline: none;
        border-color: var(--primary);
    }
    
    .product-cell {
        display: flex;
        flex-direction: column;
        gap: 2px;
    }
    
    .product-name {
        font-weight: 500;
    }
    
    .product-sku {
        font-size: 12px;
        color: var(--text-muted);
        font-family: monospace;
    }

    /* Form Actions */
    .form-actions {
        display: flex;
        justify-content: flex-end;
        gap: 12px;
        margin-top: 24px;
        padding-top: 24px;
        border-top: 1px solid var(--card-border);
    }
    
    .btn {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        padding: 12px 24px;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
        border: none;
        text-decoration: none;
    }
    
    .btn-primary {
        background: linear-gradient(135deg, var(--primary), var(--primary-hover));
        color: #fff;
        box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
    }
    
    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 16px rgba(59, 130, 246, 0.4);
    }
    
    .btn-secondary {
        background: var(--body-bg);
        color: var(--text-primary);
        border: 1px solid var(--card-border);
    }
    
    .btn-secondary:hover {
        background: var(--card-border);
    }
    
    .btn svg {
        width: 18px;
        height: 18px;
    }
</style>

<div class="page-container">
    <!-- Header -->
    <div class="page-header">
        <a href="{{ route('inventory.delivery.show', $receipt->id) }}" class="back-btn">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M15 19l-7-7 7-7"/>
            </svg>
        </a>
        <h1>Edit Delivery Receipt</h1>
    </div>

    @if(session('error'))
        <div style="background: #fee2e2; color: #991b1b; padding: 12px 16px; border-radius: 8px; margin-bottom: 20px; max-width: 1000px; margin-left: auto; margin-right: auto;">
            {{ session('error') }}
        </div>
    @endif

    <form action="{{ route('inventory.delivery.update', $receipt->id) }}" method="POST" id="deliveryForm">
        @csrf
        @method('PUT')
        
        <div class="form-card">
            <div class="form-card-header">
                <h2 class="form-card-title">Delivery Receipt Details</h2>
                <span class="receipt-number">{{ $receipt->receipt_number }}</span>
            </div>
            
            <div class="form-card-body">
                <!-- Invoice Info -->
                <div class="invoice-info-box">
                    <div class="icon">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                        </svg>
                    </div>
                    <div class="details">
                        <h3>Invoice: {{ $invoice->invoice_number }}</h3>
                        <p>Customer: {{ $invoice->customer->name ?? 'N/A' }} | Date: {{ $invoice->date ? $invoice->date->format('d M Y') : '-' }} | Total: ₹{{ number_format($invoice->total, 2) }}</p>
                    </div>
                </div>

                <!-- Delivery Details -->
                <div class="form-section">
                    <div class="form-section-title">Delivery Information</div>
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Warehouse <span class="required">*</span></label>
                            <select name="warehouse_id" id="warehouseSelect" class="form-control" required>
                                <option value="">Select Warehouse</option>
                                @foreach($warehouses as $warehouse)
                                    <option value="{{ $warehouse->id }}" {{ old('warehouse_id', $receipt->warehouse_id) == $warehouse->id ? 'selected' : '' }}>
                                        {{ $warehouse->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Delivery Date <span class="required">*</span></label>
                            <input type="date" name="delivery_date" class="form-control" 
                                   value="{{ old('delivery_date', $receipt->delivery_date ? $receipt->delivery_date->format('Y-m-d') : '') }}" required>
                        </div>
                    </div>
                    <div class="form-group full-width">
                        <label class="form-label">Remarks</label>
                        <textarea name="remarks" class="form-control" rows="2" placeholder="Optional notes about this delivery...">{{ old('remarks', $receipt->remarks) }}</textarea>
                    </div>
                </div>

                <!-- Items Section -->
                <div class="form-section">
                    <div class="form-section-title">Items to Deliver</div>
                    
                    <div style="overflow-x: auto;">
                        <table class="items-table">
                            <thead>
                                <tr>
                                    <th style="width: 40px;">#</th>
                                    <th>Product</th>
                                    <th style="width: 100px;">Invoice Qty</th>
                                    <th style="width: 120px;">Deliver Qty</th>
                                    <th style="width: 150px;">Rack (Optional)</th>
                                </tr>
                            </thead>
                            <tbody id="itemsBody">
                                @foreach($receipt->items as $index => $item)
                                    @php
                                        $invoiceItem = $invoiceItems->firstWhere('id', $item->invoice_item_id);
                                        $invoiceQty = $invoiceItem ? $invoiceItem->quantity : $item->quantity;
                                    @endphp
                                    <tr>
                                        <td>{{ $index + 1 }}</td>
                                        <td>
                                            <div class="product-cell">
                                                <span class="product-name">{{ $item->product->name ?? $item->description }}</span>
                                                <span class="product-sku">{{ $item->product->sku ?? '' }}</span>
                                            </div>
                                            <input type="hidden" name="items[{{ $index }}][invoice_item_id]" value="{{ $item->invoice_item_id }}">
                                            <input type="hidden" name="items[{{ $index }}][product_id]" value="{{ $item->product_id }}">
                                            <input type="hidden" name="items[{{ $index }}][description]" value="{{ $item->description }}">
                                        </td>
                                        <td>{{ number_format($invoiceQty, 2) }}</td>
                                        <td>
                                            <input type="number" name="items[{{ $index }}][quantity]" 
                                                   value="{{ $item->quantity }}" 
                                                   min="0" max="{{ $invoiceQty }}" step="0.01" required>
                                        </td>
                                        <td>
                                            <select name="items[{{ $index }}][rack_id]" class="rack-select" data-selected="{{ $item->rack_id }}">
                                                <option value="">Select Rack</option>
                                            </select>
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Form Actions -->
                <div class="form-actions">
                    <a href="{{ route('inventory.delivery.show', $receipt->id) }}" class="btn btn-secondary">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/>
                        </svg>
                        Cancel
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7"/>
                        </svg>
                        Update Delivery Receipt
                    </button>
                </div>
            </div>
        </div>
    </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    var warehouseSelect = document.getElementById('warehouseSelect');
    
    function loadRacks(warehouseId) {
        var rackSelects = document.querySelectorAll('.rack-select');
        
        rackSelects.forEach(function(select) {
            var selectedValue = select.dataset.selected || '';
            select.innerHTML = '<option value="">Loading...</option>';
            
            if (warehouseId) {
                fetch('/admin/inventory/racks/by-warehouse/' + warehouseId)
                    .then(response => response.json())
                    .then(data => {
                        var options = '<option value="">Select Rack</option>';
                        if (data.racks) {
                            data.racks.forEach(function(rack) {
                                var selected = rack.id == selectedValue ? 'selected' : '';
                                options += '<option value="' + rack.id + '" ' + selected + '>' + rack.name + '</option>';
                            });
                        }
                        select.innerHTML = options;
                    })
                    .catch(error => {
                        select.innerHTML = '<option value="">Select Rack</option>';
                    });
            } else {
                select.innerHTML = '<option value="">Select Rack</option>';
            }
        });
    }
    
    if (warehouseSelect) {
        warehouseSelect.addEventListener('change', function() {
            // Clear saved selections when warehouse changes
            document.querySelectorAll('.rack-select').forEach(function(s) {
                s.dataset.selected = '';
            });
            loadRacks(this.value);
        });
        
        // Load on init
        if (warehouseSelect.value) {
            loadRacks(warehouseSelect.value);
        }
    }
});
</script>