{{-- Product View - Clean Desktop + Unique Mobile UI --}}
<style>
:root {
    --shadow-sm: 0 1px 2px rgba(0,0,0,0.04);
    --shadow-md: 0 4px 12px rgba(0,0,0,0.08);
}

.product-page {
    padding: 24px;
    max-width: 1400px;
    margin: 0 auto;
    background: var(--body-bg);
    min-height: 100vh;
}

/* ==================== DESKTOP STYLES ==================== */
.page-header {
    display: flex;
    align-items: center;
    gap: 20px;
    padding: 24px;
    background: var(--card-bg);
    border-radius: 20px;
    box-shadow: var(--shadow-md);
    margin-bottom: 24px;
    border: 1px solid var(--card-border);
}

.back-btn {
    width: 48px;
    height: 48px;
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: var(--body-bg);
    border: 1px solid var(--card-border);
    color: var(--text-muted);
    text-decoration: none;
    flex-shrink: 0;
}

.back-btn:hover { background: var(--card-border); }
.back-btn svg { width: 22px; height: 22px; }

.product-avatar {
    width: 80px;
    height: 80px;
    border-radius: 16px;
    background: linear-gradient(135deg, #6366f1, #4f46e5);
    display: flex;
    align-items: center;
    justify-content: center;
    color: #fff;
    font-size: 28px;
    font-weight: 800;
    flex-shrink: 0;
    overflow: hidden;
}

.product-avatar img { width: 100%; height: 100%; object-fit: cover; }
.product-info { flex: 1; min-width: 0; }
.product-name { font-size: 26px; font-weight: 800; color: var(--text-primary); margin: 0 0 8px; }
.product-meta { display: flex; align-items: center; gap: 10px; flex-wrap: wrap; }
.meta-sku { font-family: monospace; font-size: 13px; color: var(--text-muted); background: var(--body-bg); padding: 6px 12px; border-radius: 8px; border: 1px solid var(--card-border); }

.badge { padding: 5px 12px; border-radius: 20px; font-size: 11px; font-weight: 600; }
.badge-green { background: rgba(16, 185, 129, 0.15); color: #10b981; }
.badge-red { background: rgba(239, 68, 68, 0.15); color: #ef4444; }
.badge-purple { background: rgba(139, 92, 246, 0.15); color: #8b5cf6; }
.badge-amber { background: rgba(245, 158, 11, 0.15); color: #f59e0b; }

.header-actions { display: flex; gap: 10px; flex-shrink: 0; }

.btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border-radius: 12px;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    border: none;
    cursor: pointer;
}

.btn-primary { background: linear-gradient(135deg, #6366f1, #4f46e5); color: #fff; }
.btn-success { background: linear-gradient(135deg, #10b981, #059669); color: #fff; }

.stats-row { display: grid; grid-template-columns: repeat(4, 1fr); gap: 16px; margin-bottom: 24px; }

.stat-card {
    background: var(--card-bg);
    border-radius: 16px;
    padding: 24px;
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--card-border);
    display: flex;
    align-items: center;
    gap: 16px;
}

.stat-icon { width: 56px; height: 56px; border-radius: 14px; display: flex; align-items: center; justify-content: center; font-size: 26px; }
.stat-icon.stock { background: rgba(16, 185, 129, 0.15); }
.stat-icon.value { background: rgba(59, 130, 246, 0.15); }
.stat-icon.purchase { background: rgba(245, 158, 11, 0.15); }
.stat-icon.sale { background: rgba(139, 92, 246, 0.15); }

.stat-value { font-size: 28px; font-weight: 800; color: var(--text-primary); }
.stat-value.green { color: #10b981; }
.stat-value.blue { color: #3b82f6; }
.stat-value.amber { color: #f59e0b; }
.stat-value.purple { color: #8b5cf6; }
.stat-label { font-size: 13px; color: var(--text-muted); margin-top: 4px; }
.stat-sub { font-size: 11px; color: var(--text-muted); margin-top: 2px; opacity: 0.7; }

.main-layout { display: grid; grid-template-columns: 1fr 320px; gap: 24px; }

.card { background: var(--card-bg); border-radius: 16px; box-shadow: var(--shadow-sm); border: 1px solid var(--card-border); overflow: hidden; margin-bottom: 20px; }
.card-header { padding: 18px 24px; border-bottom: 1px solid var(--card-border); display: flex; justify-content: space-between; align-items: center; background: var(--body-bg); }
.card-title { font-size: 16px; font-weight: 700; color: var(--text-primary); margin: 0; display: flex; align-items: center; gap: 10px; }
.card-title .count { background: rgba(99, 102, 241, 0.15); color: #6366f1; padding: 3px 10px; border-radius: 20px; font-size: 12px; }
.card-body { padding: 24px; }
.card-body.no-pad { padding: 0; }

.info-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; }
.info-item label { display: block; font-size: 12px; color: var(--text-muted); margin-bottom: 4px; text-transform: uppercase; }
.info-item span { font-size: 14px; font-weight: 600; color: var(--text-primary); }

.flags-row { display: flex; gap: 10px; flex-wrap: wrap; margin-top: 20px; padding-top: 20px; border-top: 1px solid var(--card-border); }
.flag { padding: 8px 14px; border-radius: 10px; font-size: 12px; font-weight: 600; }
.flag.on { background: rgba(16, 185, 129, 0.15); color: #10b981; }
.flag.off { background: var(--body-bg); color: var(--text-muted); border: 1px solid var(--card-border); }

.data-table { width: 100%; border-collapse: collapse; }
.data-table th, .data-table td { padding: 14px 20px; text-align: left; border-bottom: 1px solid var(--card-border); font-size: 13px; color: var(--text-primary); }
.data-table th { background: var(--body-bg); font-weight: 600; color: var(--text-muted); text-transform: uppercase; font-size: 11px; }
.data-table tfoot { background: var(--body-bg); font-weight: 700; }

.mv-badge { padding: 5px 12px; border-radius: 8px; font-size: 11px; font-weight: 600; }
.mv-badge.in { background: rgba(16, 185, 129, 0.15); color: #10b981; }
.mv-badge.out { background: rgba(239, 68, 68, 0.15); color: #ef4444; }
.mv-badge.transfer { background: rgba(139, 92, 246, 0.15); color: #8b5cf6; }
.mv-badge.return { background: rgba(6, 182, 212, 0.15); color: #06b6d4; }
.mv-badge.adjust { background: rgba(20, 184, 166, 0.15); color: #14b8a6; }

.qty-positive { color: #10b981; font-weight: 700; }
.qty-negative { color: #ef4444; font-weight: 700; }

.action-btn { display: flex; align-items: center; gap: 12px; width: 100%; padding: 14px 16px; background: var(--card-bg); border: 1px solid var(--card-border); border-radius: 12px; text-decoration: none; font-size: 14px; font-weight: 600; color: var(--text-secondary); margin-bottom: 10px; }
.action-btn:hover { border-color: #6366f1; background: rgba(99, 102, 241, 0.05); }
.action-btn .icon { width: 40px; height: 40px; border-radius: 10px; display: flex; align-items: center; justify-content: center; font-size: 18px; }
.action-btn .icon.green { background: rgba(16, 185, 129, 0.15); }
.action-btn .icon.red { background: rgba(239, 68, 68, 0.15); }
.action-btn .icon.purple { background: rgba(139, 92, 246, 0.15); }
.action-btn .icon.blue { background: rgba(59, 130, 246, 0.15); }
.action-btn .icon.amber { background: rgba(245, 158, 11, 0.15); }

.summary-item { display: flex; justify-content: space-between; padding: 14px 20px; border-bottom: 1px solid var(--card-border); }
.summary-label { font-size: 13px; color: var(--text-muted); }
.summary-value { font-size: 14px; font-weight: 700; color: var(--text-primary); }
.summary-value.green { color: #10b981; }
.summary-value.amber { color: #f59e0b; }
.summary-value.purple { color: #8b5cf6; }

.reorder-status { padding: 16px 20px; border-radius: 12px; margin: 16px 20px 20px; font-size: 14px; font-weight: 700; }
.reorder-status.ok { background: rgba(16, 185, 129, 0.15); color: #10b981; }
.reorder-status.low { background: rgba(239, 68, 68, 0.15); color: #ef4444; }
.reorder-status.warning { background: rgba(245, 158, 11, 0.15); color: #f59e0b; }

/* Scroll Containers */
.scroll-box { max-height: 400px; overflow-y: auto; }

.variations-container { max-height: 400px; overflow-y: auto; padding: 20px; }
.variations-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(280px, 1fr)); gap: 16px; }
.variation-card { background: var(--body-bg); border: 1px solid var(--card-border); border-radius: 12px; padding: 16px; }
.variation-sku { font-size: 11px; color: var(--text-muted); font-family: monospace; margin-bottom: 8px; }
.variation-attrs { display: flex; gap: 6px; flex-wrap: wrap; margin-bottom: 12px; }
.attr-chip { display: inline-flex; align-items: center; gap: 4px; padding: 4px 10px; background: var(--card-bg); border: 1px solid var(--card-border); border-radius: 20px; font-size: 11px; color: var(--text-secondary); }
.attr-color { width: 12px; height: 12px; border-radius: 50%; border: 1px solid var(--card-border); }
.variation-footer { display: flex; justify-content: space-between; padding-top: 12px; border-top: 1px solid var(--card-border); }
.variation-price { font-size: 16px; font-weight: 700; color: #6366f1; }
.variation-stock { font-size: 13px; color: var(--text-secondary); }
.variation-stock.in { color: #10b981; font-weight: 600; }
.variation-stock.out { color: #ef4444; font-weight: 600; }

.empty-state { text-align: center; padding: 48px 24px; color: var(--text-muted); }
.empty-state-icon { font-size: 48px; margin-bottom: 12px; opacity: 0.5; }

.loading { text-align: center; padding: 40px; }
.spinner { width: 32px; height: 32px; border: 3px solid var(--card-border); border-top-color: #6366f1; border-radius: 50%; animation: spin 0.8s linear infinite; margin: 0 auto 12px; }
@keyframes spin { to { transform: rotate(360deg); } }

.view-link { font-size: 13px; color: #6366f1; text-decoration: none; font-weight: 600; }

.desktop-only { display: block; }
.mobile-only { display: none; }

/* ==================== MOBILE STYLES ==================== */
@media (max-width: 768px) {
    .desktop-only { display: none !important; }
    .mobile-only { display: block !important; }
    
    .product-page { padding: 0; background: var(--body-bg); }
    
    /* Mobile Header - App Style */
    .page-header {
        flex-direction: column;
        gap: 0;
        padding: 0;
        border-radius: 0;
        background: linear-gradient(135deg, #6366f1 0%, #4f46e5 100%);
        color: #fff;
        margin-bottom: 0;
        padding-bottom: 60px;
        border: none;
    }
    
    .m-header-top {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 16px;
        width: 100%;
    }
    
    .back-btn {
        width: 36px;
        height: 36px;
        background: rgba(255,255,255,0.2);
        color: #fff;
        border-radius: 10px;
        border: none;
    }
    
    .m-edit-btn {
        padding: 8px 16px;
        background: rgba(255,255,255,0.2);
        color: #fff;
        border-radius: 10px;
        font-size: 13px;
        font-weight: 600;
        text-decoration: none;
    }
    
    .m-product-center {
        text-align: center;
        padding: 0 16px 20px;
    }
    
    .product-avatar {
        width: 80px;
        height: 80px;
        margin: 0 auto 12px;
        border: 3px solid rgba(255,255,255,0.3);
        background: var(--card-bg);
    }
    
    .product-avatar img { border-radius: 12px; }
    
    .product-name {
        font-size: 22px;
        color: #fff;
        margin-bottom: 4px;
    }
    
    .m-sku {
        font-family: monospace;
        font-size: 12px;
        color: rgba(255,255,255,0.8);
        margin-bottom: 10px;
    }
    
    .m-badges {
        display: flex;
        justify-content: center;
        gap: 6px;
        flex-wrap: wrap;
    }
    
    .m-badges .badge {
        font-size: 10px;
        padding: 4px 10px;
    }
    
    .header-actions { display: none; }
    .product-info { display: none; }
    .meta-sku { display: none; }
    .product-meta { display: none; }
    
    /* Mobile Stats - Floating Card */
    .stats-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 12px;
        margin: -40px 12px 12px;
        position: relative;
        z-index: 10;
    }
    
    .stat-card {
        padding: 16px;
        border-radius: 14px;
        flex-direction: column;
        align-items: flex-start;
        gap: 8px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }
    
    .stat-icon { width: 36px; height: 36px; font-size: 16px; border-radius: 10px; }
    .stat-value { font-size: 22px; }
    .stat-label { font-size: 11px; margin-top: 0; }
    .stat-sub { display: none; }
    
    /* Mobile Main Content */
    .main-layout { display: block; padding: 0 12px 12px; }
    
    /* Mobile Cards */
    .card { border-radius: 14px; margin-bottom: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.06); }
    .card-header { padding: 14px 16px; }
    .card-title { font-size: 14px; }
    .card-title .count { font-size: 10px; padding: 2px 8px; }
    .card-body { padding: 16px; }
    
    .info-grid { grid-template-columns: 1fr 1fr; gap: 14px; }
    .info-item label { font-size: 10px; }
    .info-item span { font-size: 13px; }
    
    .flags-row { gap: 6px; margin-top: 14px; padding-top: 14px; }
    .flag { padding: 6px 10px; font-size: 10px; border-radius: 8px; }
    
    /* Mobile Quick Actions - Compact Grid */
    .m-actions {
        display: grid;
        grid-template-columns: repeat(5, 1fr);
        gap: 8px;
        padding: 12px;
    }
    
    .m-action {
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 4px;
        padding: 10px 4px;
        background: var(--card-bg);
        border-radius: 12px;
        text-decoration: none;
        box-shadow: 0 2px 8px rgba(0,0,0,0.06);
        border: 1px solid var(--card-border);
    }
    
    .m-action-icon {
        width: 32px;
        height: 32px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 14px;
    }
    
    .m-action-icon.green { background: rgba(16, 185, 129, 0.15); }
    .m-action-icon.red { background: rgba(239, 68, 68, 0.15); }
    .m-action-icon.purple { background: rgba(139, 92, 246, 0.15); }
    .m-action-icon.blue { background: rgba(59, 130, 246, 0.15); }
    .m-action-icon.amber { background: rgba(245, 158, 11, 0.15); }
    
    .m-action-text { font-size: 9px; font-weight: 600; color: var(--text-muted); }
    
    /* Mobile Scroll Boxes */
    .m-scroll-box {
        max-height: 250px;
        overflow-y: auto;
        padding: 12px;
        background: var(--body-bg);
    }
    
    /* Mobile Stock Item */
    .m-stock-item {
        background: var(--card-bg);
        border-radius: 10px;
        padding: 12px;
        margin-bottom: 8px;
        box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        border: 1px solid var(--card-border);
    }
    
    .m-stock-item:last-child { margin-bottom: 0; }
    
    .m-stock-top {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        margin-bottom: 6px;
    }
    
    .m-stock-loc { font-size: 13px; font-weight: 600; color: var(--text-primary); }
    .m-stock-rack { font-size: 11px; color: var(--text-muted); }
    .m-stock-qty { font-size: 18px; font-weight: 700; }
    
    .m-stock-var {
        display: flex;
        gap: 4px;
        margin-bottom: 6px;
    }
    
    .m-stock-var .attr-chip { font-size: 9px; padding: 2px 6px; }
    .m-stock-var .attr-color { width: 8px; height: 8px; }
    
    .m-stock-bottom {
        display: flex;
        justify-content: space-between;
        padding-top: 8px;
        border-top: 1px dashed var(--card-border);
        font-size: 11px;
        color: var(--text-muted);
    }
    
    .m-stock-val { font-weight: 600; color: #10b981; }
    
    /* Mobile Movement Item */
    .m-mv-item {
        background: var(--card-bg);
        border-radius: 10px;
        padding: 12px;
        margin-bottom: 8px;
        box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        border: 1px solid var(--card-border);
    }
    
    .m-mv-item:last-child { margin-bottom: 0; }
    
    .m-mv-top {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 6px;
    }
    
    .m-mv-type {
        padding: 3px 8px;
        border-radius: 6px;
        font-size: 9px;
        font-weight: 700;
    }
    
    .m-mv-type.in { background: rgba(16, 185, 129, 0.15); color: #10b981; }
    .m-mv-type.out { background: rgba(239, 68, 68, 0.15); color: #ef4444; }
    .m-mv-type.transfer { background: rgba(139, 92, 246, 0.15); color: #8b5cf6; }
    .m-mv-type.return { background: rgba(6, 182, 212, 0.15); color: #06b6d4; }
    .m-mv-type.adjust { background: rgba(20, 184, 166, 0.15); color: #14b8a6; }
    
    .m-mv-qty { font-size: 16px; font-weight: 700; }
    
    .m-mv-ref { font-family: monospace; font-size: 10px; color: var(--text-muted); }
    
    .m-mv-bottom {
        display: flex;
        justify-content: space-between;
        font-size: 10px;
        color: var(--text-muted);
        margin-top: 4px;
    }
    
    /* Mobile Variations */
    .variations-container { max-height: 280px; padding: 12px; background: var(--body-bg); }
    .variations-grid { grid-template-columns: 1fr; gap: 8px; }
    .variation-card { padding: 12px; background: var(--card-bg); }
    .variation-sku { font-size: 9px; }
    .variation-attrs { margin-bottom: 8px; }
    .variation-attrs .attr-chip { font-size: 9px; padding: 2px 6px; }
    .variation-footer { padding-top: 8px; }
    .variation-price { font-size: 14px; }
    .variation-stock { font-size: 11px; }
    
    /* Mobile Summary */
    .m-summary {
        background: var(--card-bg);
        border-radius: 14px;
        margin: 0 12px 12px;
        padding: 16px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.06);
        border: 1px solid var(--card-border);
    }
    
    .m-summary-row {
        display: flex;
        justify-content: space-between;
        padding: 8px 0;
        border-bottom: 1px solid var(--card-border);
        font-size: 13px;
    }
    
    .m-summary-row:last-child { border-bottom: none; }
    .m-summary-label { color: var(--text-muted); }
    .m-summary-val { font-weight: 700; color: var(--text-primary); }
    .m-summary-val.green { color: #10b981; }
    .m-summary-val.purple { color: #8b5cf6; }
    
    .m-reorder {
        margin-top: 12px;
        padding: 12px;
        border-radius: 10px;
        font-size: 12px;
        font-weight: 600;
        text-align: center;
    }
    
    .m-reorder.ok { background: rgba(16, 185, 129, 0.15); color: #10b981; }
    .m-reorder.warning { background: rgba(245, 158, 11, 0.15); color: #f59e0b; }
    .m-reorder.low { background: rgba(239, 68, 68, 0.15); color: #ef4444; }
    
    .empty-state { padding: 24px; }
    .empty-state-icon { font-size: 32px; }
    .empty-state h4 { font-size: 13px; color: var(--text-muted); }
}

@media (max-width: 400px) {
    .m-actions { grid-template-columns: repeat(3, 1fr); gap: 6px; }
    .m-action { padding: 8px 2px; }
    .m-action-icon { width: 28px; height: 28px; font-size: 12px; }
    .m-action-text { font-size: 8px; }
    .stats-row { gap: 8px; margin: -40px 8px 8px; }
    .stat-card { padding: 12px; }
    .stat-value { font-size: 18px; }
}
</style>

@php
    $unitName = $product->unit->short_name ?? 'PCS';
    $stockStatus = $totalStock <= 0 ? 'out' : ($totalStock < ($product->min_stock_level ?? 0) ? 'low' : 'ok');
@endphp

<div class="product-page">
    <!-- ==================== HEADER ==================== -->
    <div class="page-header">
        <!-- Mobile Header -->
        <div class="m-header-top mobile-only">
            <a href="{{ route('inventory.products.index') }}" class="back-btn">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M15 19l-7-7 7-7"/></svg>
            </a>
            <a href="{{ route('inventory.products.edit', $product->id) }}" class="m-edit-btn">✏️ Edit</a>
        </div>
        
        <div class="m-product-center mobile-only">
            <div class="product-avatar">
                @if($product->primary_image_url)
                    <img src="{{ $product->primary_image_url }}" alt="">
                @else
                    <span style="color:#6366f1;">{{ strtoupper(substr($product->name, 0, 2)) }}</span>
                @endif
            </div>
            <h1 class="product-name">{{ $product->name }}</h1>
            <div class="m-sku">{{ $product->sku }}</div>
            <div class="m-badges">
                <span class="badge {{ $product->is_active ? 'badge-green' : 'badge-red' }}">{{ $product->is_active ? '✓ Active' : '✗ Off' }}</span>
                @if($product->has_variants)<span class="badge badge-purple">🎨 Variants</span>@endif
                @if($product->is_batch_managed)<span class="badge badge-amber">📦 Batch</span>@endif
                <span class="badge {{ $stockStatus === 'ok' ? 'badge-green' : ($stockStatus === 'low' ? 'badge-amber' : 'badge-red') }}">
                    {{ $stockStatus === 'ok' ? '✓ Stock' : ($stockStatus === 'low' ? '⚠ Low' : '✗ Out') }}
                </span>
            </div>
        </div>
        
        <!-- Desktop Header -->
        <a href="{{ route('inventory.products.index') }}" class="back-btn desktop-only">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M15 19l-7-7 7-7"/></svg>
        </a>
        <div class="product-avatar desktop-only">
            @if($product->primary_image_url)<img src="{{ $product->primary_image_url }}" alt="">@else{{ strtoupper(substr($product->name, 0, 2)) }}@endif
        </div>
        <div class="product-info desktop-only">
            <h1 class="product-name">{{ $product->name }}</h1>
            <div class="product-meta">
                <span class="meta-sku">{{ $product->sku }}</span>
                <span class="meta-sku" style="background: rgba(139, 92, 246, 0.1); border-color: rgba(139, 92, 246, 0.2);">
                    📋 {{ $product->model_number }}
                </span>
                <span class="badge {{ $product->is_active ? 'badge-green' : 'badge-red' }}">{{ $product->is_active ? '✓ Active' : '✗ Inactive' }}</span>
                @if($product->has_variants)<span class="badge badge-purple">🎨 Variants</span>@endif
                @if($product->is_batch_managed)<span class="badge badge-amber">📦 Batch</span>@endif
                <span class="badge {{ $stockStatus === 'ok' ? 'badge-green' : ($stockStatus === 'low' ? 'badge-amber' : 'badge-red') }}">
                    {{ $stockStatus === 'ok' ? '✓ In Stock' : ($stockStatus === 'low' ? '⚠ Low Stock' : '✗ Out of Stock') }}
                </span>
            </div>
        </div>
        <div class="header-actions desktop-only">
            <a href="{{ route('inventory.stock.receive') }}?product_id={{ $product->id }}" class="btn btn-success">📥 Add Stock</a>
            <a href="{{ route('inventory.products.edit', $product->id) }}" class="btn btn-primary">✏️ Edit</a>
        </div>
    </div>

    <!-- ==================== STATS ==================== -->
    <div class="stats-row">
        <div class="stat-card">
            <div class="stat-icon stock">📦</div>
            <div>
                <div class="stat-value green">{{ number_format($totalStock, 2) }}</div>
                <div class="stat-label">Stock ({{ $unitName }})</div>
                <div class="stat-sub">Total</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon value">💰</div>
            <div>
                <div class="stat-value blue">₹{{ number_format($stockValue, 0) }}</div>
                <div class="stat-label">Stock Value</div>
                <div class="stat-sub">@ ₹{{ number_format($product->purchase_price, 2) }}</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon purchase">🏷️</div>
            <div>
                <div class="stat-value amber">₹{{ number_format($product->purchase_price, 2) }}</div>
                <div class="stat-label">Purchase</div>
                <div class="stat-sub">Per {{ $unitName }}</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon sale">💵</div>
            <div>
                <div class="stat-value purple">₹{{ number_format($product->sale_price, 2) }}</div>
                <div class="stat-label">Sale Price</div>
                <div class="stat-sub">{{ $product->mrp ? 'MRP ₹'.$product->mrp : '' }}</div>
            </div>
        </div>
    </div>

    <!-- Mobile Quick Actions -->
    <div class="m-actions mobile-only">
        <a href="{{ route('inventory.stock.receive') }}?product_id={{ $product->id }}" class="m-action">
            <span class="m-action-icon green">📥</span>
            <span class="m-action-text">Receive</span>
        </a>
        <a href="{{ route('inventory.stock.deliver') }}?product_id={{ $product->id }}" class="m-action">
            <span class="m-action-icon red">📤</span>
            <span class="m-action-text">Deliver</span>
        </a>
        <a href="{{ route('inventory.stock.transfer') }}?product_id={{ $product->id }}" class="m-action">
            <span class="m-action-icon purple">🔄</span>
            <span class="m-action-text">Transfer</span>
        </a>
        <a href="{{ route('inventory.stock.returns') }}?product_id={{ $product->id }}" class="m-action">
            <span class="m-action-icon blue">↩️</span>
            <span class="m-action-text">Return</span>
        </a>
        <a href="{{ route('inventory.stock.adjustments') }}?product_id={{ $product->id }}" class="m-action">
            <span class="m-action-icon amber">⚖️</span>
            <span class="m-action-text">Adjust</span>
        </a>
    </div>

    <!-- ==================== MAIN LAYOUT ==================== -->
    <div class="main-layout">
        <div>
            <!-- Product Details -->
            <div class="card">
                <div class="card-header"><h3 class="card-title">📋 Product Details</h3></div>
                <div class="card-body">
                    <div class="info-grid">
                        <div class="info-item"><label>Category</label><span>{{ $product->category->name ?? '-' }}</span></div>
                        <div class="info-item"><label>Sub Category</label><span>{{ $product->brand->name ?? '-' }}</span></div>
                        <div class="info-item"><label>Unit</label><span>{{ $unitName }}</span></div>
                        <div class="info-item"><label>HSN</label><span>{{ $product->hsn_code ?? '-' }}</span></div>
                        <div class="info-item"><label>Model Number</label><span>{{ $product->model_number }}</span></div>
                        <div class="info-item"><label>Min Stock</label><span>{{ number_format($product->min_stock_level ?? 0, 0) }}</span></div>
                        <div class="info-item"><label>Max Stock</label><span>{{ number_format($product->max_stock_level ?? 0, 0) }}</span></div>
                        @if($product->tax1)<div class="info-item"><label>{{ $product->tax1->name }}</label><span>{{ $product->tax1->rate }}%</span></div>@endif
                        <div class="info-item"><label>Profit</label><span>{{ $product->default_profit_rate ?? 0 }}%</span></div>
                        @if($product->mrp)<div class="info-item"><label>MRP</label><span>₹{{ number_format($product->mrp, 2) }}</span></div>@endif
                    </div>
                    <div class="flags-row">
                        <span class="flag {{ $product->can_be_sold ? 'on' : 'off' }}">✓ Sell</span>
                        <span class="flag {{ $product->can_be_purchased ? 'on' : 'off' }}">✓ Purchase</span>
                        <span class="flag {{ $product->track_inventory ? 'on' : 'off' }}">✓ Track</span>
                        <span class="flag {{ $product->is_batch_managed ? 'on' : 'off' }}">✓ Batch</span>
                    </div>
                </div>
            </div>

            <!-- Stock by Location -->
            <div class="card">
                <div class="card-header"><h3 class="card-title">🏭 Stock by Location <span class="count">{{ $stockDetails->count() }}</span></h3></div>
                
                <!-- Desktop Table -->
                <div class="card-body no-pad desktop-only">
                    @if($stockDetails->count() > 0)
                    <div class="scroll-box">
                    <table class="data-table">
                        <thead><tr>@if($product->has_variants)<th>Variation</th>@endif<th>Location</th><th>Lot</th><th style="text-align:right;">Qty</th><th style="text-align:right;">Value</th></tr></thead>
                        <tbody>
                            @foreach($stockDetails as $stock)
                            <tr>
                                @if($product->has_variants)
                                <td>@if($stock->variation)<div style="display:flex;gap:4px;flex-wrap:wrap;">@foreach($stock->variation->attributeValues as $av)<span class="attr-chip">@if($av->attribute && $av->attribute->type === 'color')<span class="attr-color" style="background:{{ $av->color_code ?? '#ccc' }};"></span>@endif{{ $av->value }}</span>@endforeach</div><div style="font-size:10px;color:#94a3b8;">{{ $stock->variation->sku }}</div>@else - @endif</td>
                                @endif
                                <td><div style="font-weight:600;">{{ $stock->warehouse->name ?? '-' }}</div>@if($stock->rack)<div style="font-size:11px;color:#64748b;">{{ $stock->rack->code }}</div>@endif</td>
                                <td>{{ $stock->lot ? $stock->lot->lot_number : '-' }}</td>
                                <td style="text-align:right;"><span class="{{ $stock->qty > 0 ? 'qty-positive' : 'qty-negative' }}">{{ number_format($stock->qty, 2) }}</span></td>
                                <td style="text-align:right;color:#059669;font-weight:600;">₹{{ number_format($stock->qty * $product->purchase_price, 0) }}</td>
                            </tr>
                            @endforeach
                        </tbody>
                        <tfoot><tr><td colspan="{{ $product->has_variants ? 3 : 2 }}">Total</td><td style="text-align:right;">{{ number_format($totalStock, 2) }}</td><td style="text-align:right;color:#059669;">₹{{ number_format($stockValue, 0) }}</td></tr></tfoot>
                    </table>
                    </div>
                    @else
                    <div class="empty-state"><div class="empty-state-icon">📦</div><h4>No Stock</h4></div>
                    @endif
                </div>
                
                <!-- Mobile Scroll Cards -->
                <div class="m-scroll-box mobile-only">
                    @forelse($stockDetails as $stock)
                    <div class="m-stock-item">
                        @if($product->has_variants && $stock->variation)
                        <div class="m-stock-var">
                            @foreach($stock->variation->attributeValues as $av)
                            <span class="attr-chip">@if($av->attribute && $av->attribute->type === 'color')<span class="attr-color" style="background:{{ $av->color_code ?? '#ccc' }};"></span>@endif{{ $av->value }}</span>
                            @endforeach
                        </div>
                        @endif
                        <div class="m-stock-top">
                            <div>
                                <div class="m-stock-loc">{{ $stock->warehouse->name ?? '-' }}</div>
                                @if($stock->rack)<div class="m-stock-rack">{{ $stock->rack->code }} {{ $stock->rack->name }}</div>@endif
                            </div>
                            <div class="m-stock-qty {{ $stock->qty > 0 ? 'qty-positive' : 'qty-negative' }}">{{ number_format($stock->qty, 2) }}</div>
                        </div>
                        <div class="m-stock-bottom">
                            <span>{{ $stock->lot ? $stock->lot->lot_number : 'No Lot' }}</span>
                            <span class="m-stock-val">₹{{ number_format($stock->qty * $product->purchase_price, 0) }}</span>
                        </div>
                    </div>
                    @empty
                    <div class="empty-state" style="padding:20px;"><div class="empty-state-icon">📦</div><h4>No Stock</h4></div>
                    @endforelse
                </div>
            </div>

            <!-- Variations -->
            @if($product->has_variants)
            <div class="card">
                <div class="card-header"><h3 class="card-title">🎨 Variations <span class="count" id="variationCount">...</span></h3></div>
                <div class="variations-container" id="variationsContainer">
                    <div class="loading"><div class="spinner"></div><p style="font-size:12px;color:#94a3b8;">Loading...</p></div>
                </div>
            </div>
            @endif

            <!-- Stock Movements -->
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">📊 Stock Movements</h3>
                    <a href="{{ route('inventory.stock.movements') }}?product_id={{ $product->id }}" class="view-link">View All →</a>
                </div>
                
                <!-- Desktop Table -->
                <div class="card-body no-pad desktop-only">
                    @if($recentMovements->count() > 0)
                    <div class="scroll-box">
                    <table class="data-table">
                        <thead><tr><th>Date</th><th>Type</th><th>Reference</th><th>Location</th><th style="text-align:right;">Qty</th><th>Reason</th></tr></thead>
                        <tbody>
                            @foreach($recentMovements as $mv)
                            @php $tc = match($mv->movement_type) { 'IN'=>'in', 'OUT'=>'out', 'TRANSFER'=>'transfer', 'RETURN'=>'return', default=>'adjust' }; $ti = match($mv->movement_type) { 'IN'=>'📥', 'OUT'=>'📤', 'TRANSFER'=>'🔄', 'RETURN'=>'↩️', default=>'⚖️' }; @endphp
                            <tr>
                                <td><div>{{ $mv->created_at->format('d M Y') }}</div><div style="font-size:11px;color:#94a3b8;">{{ $mv->created_at->format('h:i A') }}</div></td>
                                <td><span class="mv-badge {{ $tc }}">{{ $ti }} {{ $mv->movement_type }}</span></td>
                                <td style="font-family:monospace;font-size:12px;">{{ $mv->reference_no }}</td>
                                <td>{{ $mv->warehouse->name ?? '' }}@if($mv->rack) / {{ $mv->rack->code }}@endif</td>
                                <td style="text-align:right;"><span class="{{ $mv->qty >= 0 ? 'qty-positive' : 'qty-negative' }}">{{ $mv->qty >= 0 ? '+' : '' }}{{ number_format($mv->qty, 2) }}</span></td>
                                <td style="font-size:12px;color:#64748b;">{{ Str::limit($mv->reason, 15) }}</td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                    </div>
                    @else
                    <div class="empty-state"><div class="empty-state-icon">📋</div><h4>No Movements</h4></div>
                    @endif
                </div>
                
                <!-- Mobile Scroll Cards -->
                <div class="m-scroll-box mobile-only">
                    @forelse($recentMovements as $mv)
                    @php $tc = match($mv->movement_type) { 'IN'=>'in', 'OUT'=>'out', 'TRANSFER'=>'transfer', 'RETURN'=>'return', default=>'adjust' }; $ti = match($mv->movement_type) { 'IN'=>'📥', 'OUT'=>'📤', 'TRANSFER'=>'🔄', 'RETURN'=>'↩️', default=>'⚖️' }; @endphp
                    <div class="m-mv-item">
                        <div class="m-mv-top">
                            <span class="m-mv-type {{ $tc }}">{{ $ti }} {{ $mv->movement_type }}</span>
                            <span class="m-mv-qty {{ $mv->qty >= 0 ? 'qty-positive' : 'qty-negative' }}">{{ $mv->qty >= 0 ? '+' : '' }}{{ number_format($mv->qty, 2) }}</span>
                        </div>
                        <div class="m-mv-ref">{{ $mv->reference_no }}</div>
                        <div class="m-mv-bottom">
                            <span>{{ $mv->warehouse->name ?? '' }}@if($mv->rack) / {{ $mv->rack->code }}@endif</span>
                            <span>{{ $mv->created_at->format('d M, h:i A') }}</span>
                        </div>
                    </div>
                    @empty
                    <div class="empty-state" style="padding:20px;"><div class="empty-state-icon">📋</div><h4>No Movements</h4></div>
                    @endforelse
                </div>
            </div>
        </div>

        <!-- Sidebar Desktop -->
        <div class="desktop-only">
            <div class="card">
                <div class="card-header"><h3 class="card-title">⚡ Quick Actions</h3></div>
                <div class="card-body">
                    <a href="{{ route('inventory.stock.receive') }}?product_id={{ $product->id }}" class="action-btn"><span class="icon green">📥</span>Receive Stock</a>
                    <a href="{{ route('inventory.stock.deliver') }}?product_id={{ $product->id }}" class="action-btn"><span class="icon red">📤</span>Deliver Stock</a>
                    <a href="{{ route('inventory.stock.transfer') }}?product_id={{ $product->id }}" class="action-btn"><span class="icon purple">🔄</span>Transfer Stock</a>
                    <a href="{{ route('inventory.stock.returns') }}?product_id={{ $product->id }}" class="action-btn"><span class="icon blue">↩️</span>Process Return</a>
                    <a href="{{ route('inventory.stock.adjustments') }}?product_id={{ $product->id }}" class="action-btn"><span class="icon amber">⚖️</span>Adjust Stock</a>
                </div>
            </div>
            <div class="card">
                <div class="card-header"><h3 class="card-title">📈 Summary</h3></div>
                <div class="card-body no-pad">
                    <div class="summary-item"><span class="summary-label">Qty</span><span class="summary-value">{{ number_format($totalStock, 2) }} {{ $unitName }}</span></div>
                    <div class="summary-item"><span class="summary-label">Purchase</span><span class="summary-value amber">₹{{ number_format($stockValue, 0) }}</span></div>
                    <div class="summary-item"><span class="summary-label">Sale</span><span class="summary-value green">₹{{ number_format($saleValue, 0) }}</span></div>
                    <div class="summary-item"><span class="summary-label">Profit</span><span class="summary-value purple">₹{{ number_format($saleValue - $stockValue, 0) }}</span></div>
                </div>
            </div>
            <div class="card">
                <div class="card-header"><h3 class="card-title">🔔 Reorder</h3></div>
                <div class="card-body no-pad">
                    <div class="summary-item"><span class="summary-label">Min</span><span class="summary-value">{{ number_format($product->min_stock_level ?? 0, 0) }}</span></div>
                    <div class="summary-item"><span class="summary-label">Max</span><span class="summary-value">{{ number_format($product->max_stock_level ?? 0, 0) }}</span></div>
                    @php $rq = max(0, ($product->max_stock_level ?? 0) - $totalStock); @endphp
                    <div class="reorder-status {{ $stockStatus === 'ok' ? 'ok' : ($stockStatus === 'low' ? 'warning' : 'low') }}">
                        @if($stockStatus === 'ok') ✅ Stock OK @elseif($stockStatus === 'low') ⚠️ Reorder {{ number_format($rq, 0) }} @else 🔴 Out! @endif
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Mobile Summary -->
    <div class="m-summary mobile-only">
        <div class="m-summary-row"><span class="m-summary-label">Sale Value</span><span class="m-summary-val green">₹{{ number_format($saleValue, 0) }}</span></div>
        <div class="m-summary-row"><span class="m-summary-label">Profit</span><span class="m-summary-val purple">₹{{ number_format($saleValue - $stockValue, 0) }}</span></div>
        @php $rq = max(0, ($product->max_stock_level ?? 0) - $totalStock); @endphp
        <div class="m-reorder {{ $stockStatus === 'ok' ? 'ok' : ($stockStatus === 'low' ? 'warning' : 'low') }}">
            @if($stockStatus === 'ok') ✅ Stock OK @elseif($stockStatus === 'low') ⚠️ Reorder {{ number_format($rq, 0) }} {{ $unitName }} @else 🔴 Out of Stock @endif
        </div>
    </div>
</div>

@if($product->has_variants)
<script>
document.addEventListener('DOMContentLoaded', function() {
    fetch('{{ route("inventory.products.variations", $product->id) }}')
        .then(r => r.json())
        .then(data => {
            var c = document.getElementById('variationsContainer'), cnt = document.getElementById('variationCount');
            if (data.success && data.variations && data.variations.length > 0) {
                cnt.textContent = data.variations.length;
                var h = '<div class="variations-grid">';
                data.variations.forEach(v => {
                    var sc = v.stock_qty > 0 ? 'in' : 'out', st = v.stock_qty > 0 ? parseFloat(v.stock_qty).toFixed(0) + ' pcs' : 'Out';
                    h += '<div class="variation-card"><div class="variation-sku">' + v.sku + '</div>';
                    if (v.attributes && v.attributes.length > 0) {
                        h += '<div class="variation-attrs">';
                        v.attributes.forEach(a => { h += '<span class="attr-chip">' + (a.color_code ? '<span class="attr-color" style="background:' + a.color_code + '"></span>' : '') + a.value + '</span>'; });
                        h += '</div>';
                    }
                    h += '<div class="variation-footer"><span class="variation-price">₹' + parseFloat(v.sale_price || 0).toFixed(0) + '</span><span class="variation-stock ' + sc + '">' + st + '</span></div></div>';
                });
                h += '</div>';
                c.innerHTML = h;
            } else { cnt.textContent = '0'; c.innerHTML = '<div class="empty-state"><div class="empty-state-icon">🎨</div><h4>No Variations</h4></div>'; }
        }).catch(() => { document.getElementById('variationsContainer').innerHTML = '<div class="empty-state"><div class="empty-state-icon">⚠️</div><h4>Error</h4></div>'; });
});
</script>
@endif