{{-- Visual Rack Layout - Warehouse Floor Plan View --}}

<style>
    .rack-layout-container { padding: 20px; max-width: 1400px; margin: 0 auto; }
    
    .rack-layout-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 24px; flex-wrap: wrap; gap: 16px; }
    .rack-layout-title { font-size: 24px; font-weight: 700; color: var(--text-primary); display: flex; align-items: center; gap: 12px; }
    .rack-layout-title svg { width: 28px; height: 28px; color: #7c3aed; }
    
    .rack-layout-controls { display: flex; gap: 12px; align-items: center; flex-wrap: wrap; }
    
    .warehouse-tabs { display: flex; gap: 8px; background: var(--card-bg); border: 1px solid var(--card-border); border-radius: 10px; padding: 6px; }
    .warehouse-tab { padding: 10px 18px; border-radius: 8px; font-size: 13px; font-weight: 600; cursor: pointer; color: var(--text-muted); transition: all 0.2s; border: none; background: transparent; }
    .warehouse-tab:hover { color: var(--text-primary); background: var(--body-bg); }
    .warehouse-tab.active { background: linear-gradient(135deg, #7c3aed, #6d28d9); color: #fff; }
    
    .view-toggle { display: flex; gap: 4px; background: var(--card-bg); border: 1px solid var(--card-border); border-radius: 8px; padding: 4px; }
    .view-toggle-btn { padding: 8px 14px; border: none; background: transparent; border-radius: 6px; cursor: pointer; font-size: 12px; font-weight: 500; color: var(--text-muted); transition: all 0.15s; }
    .view-toggle-btn:hover { color: var(--text-primary); }
    .view-toggle-btn.active { background: #7c3aed; color: #fff; }
    
    /* Stats Bar */
    .rack-stats-bar { display: flex; gap: 12px; margin-bottom: 24px; flex-wrap: wrap; }
    .rack-stat-card { background: var(--card-bg); border: 1px solid var(--card-border); border-radius: 10px; padding: 16px 20px; display: flex; align-items: center; gap: 14px; min-width: 160px; }
    .rack-stat-icon { width: 44px; height: 44px; border-radius: 10px; display: flex; align-items: center; justify-content: center; font-size: 20px; }
    .rack-stat-icon.purple { background: #ede9fe; }
    .rack-stat-icon.green { background: #d1fae5; }
    .rack-stat-icon.orange { background: #ffedd5; }
    .rack-stat-icon.red { background: #fee2e2; }
    .rack-stat-value { font-size: 24px; font-weight: 700; color: var(--text-primary); }
    .rack-stat-label { font-size: 12px; color: var(--text-muted); margin-top: 2px; }
    
    /* Legend */
    .capacity-legend { display: flex; gap: 16px; margin-bottom: 20px; padding: 12px 16px; background: var(--card-bg); border: 1px solid var(--card-border); border-radius: 10px; flex-wrap: wrap; }
    .legend-item { display: flex; align-items: center; gap: 8px; font-size: 12px; color: var(--text-muted); }
    .legend-color { width: 16px; height: 16px; border-radius: 4px; }
    .legend-color.empty { background: #e5e7eb; }
    .legend-color.low { background: linear-gradient(135deg, #22c55e, #4ade80); }
    .legend-color.medium { background: linear-gradient(135deg, #eab308, #facc15); }
    .legend-color.high { background: linear-gradient(135deg, #f97316, #fb923c); }
    .legend-color.full { background: linear-gradient(135deg, #ef4444, #f87171); }
    
    /* Floor Plan Grid */
    .floor-plan-container { background: var(--card-bg); border: 1px solid var(--card-border); border-radius: 16px; padding: 24px; overflow-x: auto; }
    
    .zone-section { margin-bottom: 32px; }
    .zone-section:last-child { margin-bottom: 0; }
    
    .zone-header { display: flex; align-items: center; gap: 12px; margin-bottom: 16px; padding-bottom: 12px; border-bottom: 2px dashed var(--card-border); }
    .zone-badge { padding: 6px 14px; background: #ede9fe; color: #6d28d9; border-radius: 20px; font-size: 13px; font-weight: 600; }
    .zone-name { font-size: 16px; font-weight: 600; color: var(--text-primary); }
    .zone-stats { margin-left: auto; font-size: 12px; color: var(--text-muted); }
    
    /* Rack Grid */
    .rack-floor-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(140px, 1fr)); gap: 12px; }
    
    .rack-floor-item { background: #fff; border: 2px solid var(--card-border); border-radius: 12px; padding: 14px; cursor: pointer; transition: all 0.2s; position: relative; }
    .rack-floor-item:hover { transform: translateY(-4px); box-shadow: 0 8px 24px rgba(0,0,0,0.12); border-color: #7c3aed; }
    
    .rack-floor-header { display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 10px; }
    .rack-floor-code { font-size: 15px; font-weight: 700; color: #7c3aed; }
    .rack-floor-location { font-size: 10px; color: var(--text-muted); background: var(--body-bg); padding: 2px 6px; border-radius: 4px; }
    
    .rack-floor-name { font-size: 12px; color: var(--text-muted); margin-bottom: 12px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
    
    .rack-floor-capacity { margin-bottom: 8px; }
    .rack-floor-bar { height: 8px; background: #e5e7eb; border-radius: 4px; overflow: hidden; }
    .rack-floor-fill { height: 100%; border-radius: 4px; transition: width 0.3s; }
    .rack-floor-fill.empty { background: #e5e7eb; }
    .rack-floor-fill.low { background: linear-gradient(90deg, #22c55e, #4ade80); }
    .rack-floor-fill.medium { background: linear-gradient(90deg, #eab308, #facc15); }
    .rack-floor-fill.high { background: linear-gradient(90deg, #f97316, #fb923c); }
    .rack-floor-fill.full { background: linear-gradient(90deg, #ef4444, #f87171); }
    
    .rack-floor-stats { display: flex; justify-content: space-between; font-size: 11px; }
    .rack-floor-percent { font-weight: 700; }
    .rack-floor-percent.empty { color: #6b7280; }
    .rack-floor-percent.low { color: #16a34a; }
    .rack-floor-percent.medium { color: #ca8a04; }
    .rack-floor-percent.high { color: #ea580c; }
    .rack-floor-percent.full { color: #dc2626; }
    .rack-floor-qty { color: var(--text-muted); }
    
    .rack-floor-products { margin-top: 10px; padding-top: 10px; border-top: 1px dashed var(--card-border); }
    .rack-floor-products-count { font-size: 11px; color: var(--text-muted); display: flex; align-items: center; gap: 4px; }
    
    /* List View */
    .rack-list-view { display: none; }
    .rack-list-view.show { display: block; }
    
    .rack-list-table { width: 100%; border-collapse: collapse; }
    .rack-list-table th { text-align: left; padding: 12px 16px; font-size: 11px; font-weight: 600; color: var(--text-muted); text-transform: uppercase; border-bottom: 2px solid var(--card-border); }
    .rack-list-table td { padding: 14px 16px; border-bottom: 1px solid var(--card-border); font-size: 13px; }
    .rack-list-table tr:hover { background: var(--body-bg); }
    
    .rack-list-code { font-weight: 700; color: #7c3aed; }
    .rack-list-location-cell { font-size: 12px; color: var(--text-muted); }
    
    .rack-list-bar { width: 100px; height: 8px; background: #e5e7eb; border-radius: 4px; overflow: hidden; }
    .rack-list-bar-fill { height: 100%; border-radius: 4px; }
    
    /* Rack Detail Modal */
    .rack-detail-modal { position: fixed; inset: 0; background: rgba(0,0,0,0.5); display: none; align-items: center; justify-content: center; z-index: 1000; padding: 20px; }
    .rack-detail-modal.show { display: flex; }
    
    .rack-detail-content { background: var(--card-bg); border-radius: 16px; width: 100%; max-width: 500px; max-height: 90vh; overflow-y: auto; }
    
    .rack-detail-header { padding: 20px 24px; border-bottom: 1px solid var(--card-border); display: flex; justify-content: space-between; align-items: center; }
    .rack-detail-title { font-size: 18px; font-weight: 700; color: var(--text-primary); }
    .rack-detail-close { width: 36px; height: 36px; border: none; background: var(--body-bg); border-radius: 8px; cursor: pointer; display: flex; align-items: center; justify-content: center; color: var(--text-muted); }
    .rack-detail-close:hover { background: var(--card-border); }
    
    .rack-detail-body { padding: 24px; }
    
    .rack-detail-info { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; margin-bottom: 20px; }
    .rack-detail-info-item { background: var(--body-bg); padding: 14px; border-radius: 10px; }
    .rack-detail-info-label { font-size: 11px; color: var(--text-muted); margin-bottom: 4px; text-transform: uppercase; }
    .rack-detail-info-value { font-size: 16px; font-weight: 700; color: var(--text-primary); }
    
    .rack-detail-capacity { margin-bottom: 20px; }
    .rack-detail-capacity-header { display: flex; justify-content: space-between; margin-bottom: 8px; font-size: 13px; }
    .rack-detail-capacity-bar { height: 12px; background: #e5e7eb; border-radius: 6px; overflow: hidden; }
    .rack-detail-capacity-fill { height: 100%; border-radius: 6px; }
    
    .rack-detail-products { margin-top: 20px; }
    .rack-detail-products-title { font-size: 14px; font-weight: 600; margin-bottom: 12px; color: var(--text-primary); }
    .rack-detail-product-item { display: flex; align-items: center; gap: 12px; padding: 10px; background: var(--body-bg); border-radius: 8px; margin-bottom: 8px; }
    .rack-detail-product-image { width: 40px; height: 40px; border-radius: 6px; object-fit: cover; background: #fff; }
    .rack-detail-product-name { font-size: 13px; font-weight: 500; color: var(--text-primary); }
    .rack-detail-product-qty { margin-left: auto; font-weight: 700; color: #7c3aed; }
    
    .rack-detail-actions { display: flex; gap: 10px; margin-top: 20px; padding-top: 20px; border-top: 1px solid var(--card-border); }
    .rack-detail-btn { flex: 1; padding: 12px; border: none; border-radius: 8px; font-size: 13px; font-weight: 600; cursor: pointer; display: flex; align-items: center; justify-content: center; gap: 6px; transition: all 0.2s; }
    .rack-detail-btn.primary { background: linear-gradient(135deg, #7c3aed, #6d28d9); color: #fff; }
    .rack-detail-btn.secondary { background: var(--body-bg); color: var(--text-primary); }
    
    /* Empty State */
    .rack-empty-state { text-align: center; padding: 60px 20px; color: var(--text-muted); }
    .rack-empty-icon { font-size: 48px; margin-bottom: 16px; }
    .rack-empty-text { font-size: 16px; margin-bottom: 8px; }
    .rack-empty-hint { font-size: 13px; }
</style>

<div class="rack-layout-container">
    <!-- Header -->
    <div class="rack-layout-header">
        <div class="rack-layout-title">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"/>
            </svg>
            Rack Layout
        </div>
        
        <div class="rack-layout-controls">
            <div class="warehouse-tabs" id="warehouseTabs">
                @foreach($warehouses as $index => $wh)
                    <button type="button" class="warehouse-tab {{ $index === 0 ? 'active' : '' }}" data-id="{{ $wh->id }}" onclick="loadWarehouseLayout({{ $wh->id }}, this)">
                        {{ $wh->name }}
                    </button>
                @endforeach
            </div>
            
            <div class="view-toggle">
                <button type="button" class="view-toggle-btn active" onclick="setLayoutView('grid', this)">▦ Grid</button>
                <button type="button" class="view-toggle-btn" onclick="setLayoutView('list', this)">☰ List</button>
            </div>
        </div>
    </div>
    
    <!-- Stats Bar -->
    <div class="rack-stats-bar" id="rackStatsBar">
        <div class="rack-stat-card">
            <div class="rack-stat-icon purple">📦</div>
            <div>
                <div class="rack-stat-value" id="statTotalRacks">0</div>
                <div class="rack-stat-label">Total Racks</div>
            </div>
        </div>
        <div class="rack-stat-card">
            <div class="rack-stat-icon green">✓</div>
            <div>
                <div class="rack-stat-value" id="statAvailable">0</div>
                <div class="rack-stat-label">Available (&lt;70%)</div>
            </div>
        </div>
        <div class="rack-stat-card">
            <div class="rack-stat-icon orange">⚠</div>
            <div>
                <div class="rack-stat-value" id="statNearFull">0</div>
                <div class="rack-stat-label">Near Full (70-90%)</div>
            </div>
        </div>
        <div class="rack-stat-card">
            <div class="rack-stat-icon red">🚫</div>
            <div>
                <div class="rack-stat-value" id="statFull">0</div>
                <div class="rack-stat-label">Full (&gt;90%)</div>
            </div>
        </div>
    </div>
    
    <!-- Legend -->
    <div class="capacity-legend">
        <span style="font-weight:600;color:var(--text-primary);margin-right:8px;">Capacity:</span>
        <div class="legend-item"><div class="legend-color empty"></div> Empty (0%)</div>
        <div class="legend-item"><div class="legend-color low"></div> Low (1-40%)</div>
        <div class="legend-item"><div class="legend-color medium"></div> Medium (40-70%)</div>
        <div class="legend-item"><div class="legend-color high"></div> High (70-90%)</div>
        <div class="legend-item"><div class="legend-color full"></div> Full (90%+)</div>
    </div>
    
    <!-- Floor Plan -->
    <div class="floor-plan-container">
        <div id="gridView">
            <!-- Zones and racks populated by JS -->
        </div>
        
        <div class="rack-list-view" id="listView">
            <table class="rack-list-table">
                <thead>
                    <tr>
                        <th>Code</th>
                        <th>Name</th>
                        <th>Zone</th>
                        <th>Location</th>
                        <th>Capacity</th>
                        <th>Stock</th>
                        <th>Products</th>
                    </tr>
                </thead>
                <tbody id="rackListBody"></tbody>
            </table>
        </div>
        
        <div class="rack-empty-state" id="rackEmptyState" style="display:none;">
            <div class="rack-empty-icon">📦</div>
            <div class="rack-empty-text">No racks in this warehouse</div>
            <div class="rack-empty-hint">Add racks to start organizing your inventory</div>
        </div>
    </div>
</div>

<!-- Rack Detail Modal -->
<div class="rack-detail-modal" id="rackDetailModal">
    <div class="rack-detail-content">
        <div class="rack-detail-header">
            <div class="rack-detail-title" id="modalRackTitle">Rack Details</div>
            <button type="button" class="rack-detail-close" onclick="closeRackModal()">
                <svg width="20" height="20" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/>
                </svg>
            </button>
        </div>
        <div class="rack-detail-body">
            <div class="rack-detail-info">
                <div class="rack-detail-info-item">
                    <div class="rack-detail-info-label">Code</div>
                    <div class="rack-detail-info-value" id="modalRackCode">-</div>
                </div>
                <div class="rack-detail-info-item">
                    <div class="rack-detail-info-label">Zone</div>
                    <div class="rack-detail-info-value" id="modalRackZone">-</div>
                </div>
                <div class="rack-detail-info-item">
                    <div class="rack-detail-info-label">Aisle</div>
                    <div class="rack-detail-info-value" id="modalRackAisle">-</div>
                </div>
                <div class="rack-detail-info-item">
                    <div class="rack-detail-info-label">Level</div>
                    <div class="rack-detail-info-value" id="modalRackLevel">-</div>
                </div>
            </div>
            
            <div class="rack-detail-capacity">
                <div class="rack-detail-capacity-header">
                    <span>Capacity Usage</span>
                    <span id="modalCapacityText">0 / 0</span>
                </div>
                <div class="rack-detail-capacity-bar">
                    <div class="rack-detail-capacity-fill" id="modalCapacityBar" style="width:0%"></div>
                </div>
            </div>
            
            <div class="rack-detail-products">
                <div class="rack-detail-products-title">Products in this Rack</div>
                <div id="modalProductsList">
                    <div style="text-align:center;padding:20px;color:var(--text-muted);">Loading...</div>
                </div>
            </div>
            
            <div class="rack-detail-actions">
                <a href="#" class="rack-detail-btn secondary" id="modalEditLink">✏️ Edit Rack</a>
                <a href="{{ route('inventory.stock.receive') }}" class="rack-detail-btn primary">📥 Receive Stock</a>
            </div>
        </div>
    </div>
</div>

<script>
var currentWarehouseId = {{ $warehouses->first()->id ?? 0 }};
var currentView = 'grid';
var racksData = [];

document.addEventListener('DOMContentLoaded', function() {
    if (currentWarehouseId) {
        loadWarehouseLayout(currentWarehouseId);
    }
});

function loadWarehouseLayout(warehouseId, tabEl) {
    currentWarehouseId = warehouseId;
    
    // Update tab selection
    if (tabEl) {
        document.querySelectorAll('.warehouse-tab').forEach(t => t.classList.remove('active'));
        tabEl.classList.add('active');
    }
    
    // Fetch rack data
    fetch('{{ url("admin/inventory/racks/layout") }}/' + warehouseId)
        .then(r => r.json())
        .then(data => {
            racksData = data.racks || [];
            updateStats(racksData);
            renderLayout(racksData);
        });
}

function updateStats(racks) {
    var total = racks.length;
    var available = 0, nearFull = 0, full = 0;
    
    racks.forEach(r => {
        var percent = r.max_capacity > 0 ? (r.current_stock / r.max_capacity) * 100 : 0;
        if (percent >= 90) full++;
        else if (percent >= 70) nearFull++;
        else available++;
    });
    
    document.getElementById('statTotalRacks').textContent = total;
    document.getElementById('statAvailable').textContent = available;
    document.getElementById('statNearFull').textContent = nearFull;
    document.getElementById('statFull').textContent = full;
}

function renderLayout(racks) {
    var gridView = document.getElementById('gridView');
    var listBody = document.getElementById('rackListBody');
    var emptyState = document.getElementById('rackEmptyState');
    
    if (racks.length === 0) {
        gridView.innerHTML = '';
        listBody.innerHTML = '';
        emptyState.style.display = 'block';
        return;
    }
    
    emptyState.style.display = 'none';
    
    // Group by zone
    var zones = {};
    racks.forEach(r => {
        var zone = r.zone || 'Unassigned';
        if (!zones[zone]) zones[zone] = [];
        zones[zone].push(r);
    });
    
    // Render Grid View
    gridView.innerHTML = '';
    Object.keys(zones).forEach(zoneName => {
        var zoneRacks = zones[zoneName];
        var zoneTotal = zoneRacks.reduce((sum, r) => sum + (r.current_stock || 0), 0);
        
        var section = document.createElement('div');
        section.className = 'zone-section';
        section.innerHTML = '<div class="zone-header"><span class="zone-badge">' + zoneName + '</span><span class="zone-name">' + zoneRacks.length + ' Racks</span><span class="zone-stats">Total Stock: ' + zoneTotal + ' units</span></div>';
        
        var grid = document.createElement('div');
        grid.className = 'rack-floor-grid';
        
        zoneRacks.forEach(rack => {
            var cap = getCapacity(rack);
            var item = document.createElement('div');
            item.className = 'rack-floor-item';
            item.onclick = function() { openRackModal(rack); };
            
            var location = [rack.aisle ? 'A' + rack.aisle : null, rack.level ? 'L' + rack.level : null].filter(Boolean).join('-');
            
            item.innerHTML = 
                '<div class="rack-floor-header">' +
                    '<span class="rack-floor-code">' + rack.code + '</span>' +
                    (location ? '<span class="rack-floor-location">' + location + '</span>' : '') +
                '</div>' +
                '<div class="rack-floor-name">' + rack.name + '</div>' +
                '<div class="rack-floor-capacity">' +
                    '<div class="rack-floor-bar"><div class="rack-floor-fill ' + cap.class + '" style="width:' + cap.percent + '%"></div></div>' +
                '</div>' +
                '<div class="rack-floor-stats">' +
                    '<span class="rack-floor-percent ' + cap.class + '">' + cap.percent + '%</span>' +
                    '<span class="rack-floor-qty">' + (rack.current_stock || 0) + '/' + (rack.max_capacity || '∞') + '</span>' +
                '</div>' +
                '<div class="rack-floor-products">' +
                    '<div class="rack-floor-products-count">📦 ' + (rack.products_count || 0) + ' products</div>' +
                '</div>';
            
            grid.appendChild(item);
        });
        
        section.appendChild(grid);
        gridView.appendChild(section);
    });
    
    // Render List View
    listBody.innerHTML = '';
    racks.forEach(rack => {
        var cap = getCapacity(rack);
        var location = [rack.aisle ? 'Aisle ' + rack.aisle : null, rack.level ? 'Level ' + rack.level : null].filter(Boolean).join(', ');
        
        var row = document.createElement('tr');
        row.style.cursor = 'pointer';
        row.onclick = function() { openRackModal(rack); };
        
        row.innerHTML = 
            '<td><span class="rack-list-code">' + rack.code + '</span></td>' +
            '<td>' + rack.name + '</td>' +
            '<td>' + (rack.zone || '-') + '</td>' +
            '<td class="rack-list-location-cell">' + (location || '-') + '</td>' +
            '<td><div class="rack-list-bar"><div class="rack-list-bar-fill ' + cap.class + '" style="width:' + cap.percent + '%"></div></div></td>' +
            '<td>' + (rack.current_stock || 0) + ' / ' + (rack.max_capacity || '∞') + '</td>' +
            '<td>' + (rack.products_count || 0) + '</td>';
        
        listBody.appendChild(row);
    });
}

function getCapacity(rack) {
    var current = parseFloat(rack.current_stock) || 0;
    var max = parseFloat(rack.max_capacity) || 0;
    
    if (max === 0) return { percent: 0, class: 'empty' };
    
    var percent = Math.min(100, Math.round((current / max) * 100));
    var cls = 'empty';
    if (percent >= 90) cls = 'full';
    else if (percent >= 70) cls = 'high';
    else if (percent >= 40) cls = 'medium';
    else if (percent > 0) cls = 'low';
    
    return { percent: percent, class: cls };
}

function setLayoutView(view, btn) {
    currentView = view;
    document.querySelectorAll('.view-toggle-btn').forEach(b => b.classList.remove('active'));
    btn.classList.add('active');
    
    document.getElementById('gridView').style.display = view === 'grid' ? 'block' : 'none';
    document.getElementById('listView').classList.toggle('show', view === 'list');
}

function openRackModal(rack) {
    var modal = document.getElementById('rackDetailModal');
    var cap = getCapacity(rack);
    
    document.getElementById('modalRackTitle').textContent = rack.code + ' - ' + rack.name;
    document.getElementById('modalRackCode').textContent = rack.code;
    document.getElementById('modalRackZone').textContent = rack.zone || '-';
    document.getElementById('modalRackAisle').textContent = rack.aisle || '-';
    document.getElementById('modalRackLevel').textContent = rack.level || '-';
    document.getElementById('modalCapacityText').textContent = (rack.current_stock || 0) + ' / ' + (rack.max_capacity || '∞');
    document.getElementById('modalCapacityBar').style.width = cap.percent + '%';
    document.getElementById('modalCapacityBar').className = 'rack-detail-capacity-fill ' + cap.class;
    document.getElementById('modalEditLink').href = '{{ url("admin/inventory/racks") }}/' + rack.id + '/edit';
    
    // Load products
    document.getElementById('modalProductsList').innerHTML = '<div style="text-align:center;padding:20px;color:var(--text-muted);">Loading products...</div>';
    
    fetch('{{ url("admin/inventory/racks") }}/' + rack.id + '/products')
        .then(r => r.json())
        .then(products => {
            var list = document.getElementById('modalProductsList');
            if (products.length === 0) {
                list.innerHTML = '<div style="text-align:center;padding:20px;color:var(--text-muted);">No products in this rack</div>';
                return;
            }
            
            list.innerHTML = '';
            products.forEach(p => {
                var item = document.createElement('div');
                item.className = 'rack-detail-product-item';
                item.innerHTML = 
                    (p.image ? '<img src="' + p.image + '" class="rack-detail-product-image" alt="">' : '<div class="rack-detail-product-image" style="display:flex;align-items:center;justify-content:center;background:#ede9fe;color:#7c3aed;font-weight:700;">' + p.name.substring(0,2).toUpperCase() + '</div>') +
                    '<div class="rack-detail-product-name">' + p.name + '<br><small style="color:var(--text-muted);">' + p.sku + '</small></div>' +
                    '<div class="rack-detail-product-qty">' + p.qty + ' ' + (p.unit || 'PCS') + '</div>';
                list.appendChild(item);
            });
        });
    
    modal.classList.add('show');
}

function closeRackModal() {
    document.getElementById('rackDetailModal').classList.remove('show');
}

// Close modal on outside click
document.getElementById('rackDetailModal').addEventListener('click', function(e) {
    if (e.target === this) closeRackModal();
});
</script>
