{{-- Stock Returns - Modern Green Theme (Adds Stock Back) --}}

<style>
    :root {
        --return-primary: #10b981;
        --return-dark: #059669;
        --return-light: #d1fae5;
        --return-bg: #ecfdf5;
        --shadow-sm: 0 1px 2px rgba(0,0,0,0.05);
        --shadow-md: 0 4px 12px rgba(0,0,0,0.08);
        --shadow-lg: 0 8px 24px rgba(0,0,0,0.12);
    }
    
    .page-container { padding: 24px; max-width: 900px; margin: 0 auto; }
    
    .page-header { 
        display: flex; align-items: center; gap: 16px; margin-bottom: 28px;
        padding: 20px 24px; background: linear-gradient(135deg, var(--return-bg), var(--return-light));
        border-radius: 16px; border: 1px solid var(--return-primary);
    }
    .back-btn { 
        width: 44px; height: 44px; border-radius: 12px; display: flex; align-items: center; justify-content: center; 
        background: #fff; border: 2px solid var(--return-primary); color: var(--return-dark); 
        text-decoration: none; transition: all 0.2s; flex-shrink: 0; box-shadow: var(--shadow-sm);
    }
    .back-btn:hover { background: var(--return-primary); color: #fff; transform: translateX(-2px); }
    .back-btn svg { width: 20px; height: 20px; }
    .page-header h1 { 
        font-size: 26px; font-weight: 800; color: var(--return-dark); margin: 0; 
        display: flex; align-items: center; gap: 12px; 
    }
    .page-header-icon {
        width: 48px; height: 48px; background: var(--return-primary); border-radius: 12px;
        display: flex; align-items: center; justify-content: center; color: #fff; font-size: 24px;
        box-shadow: var(--shadow-md);
    }
    .page-header-text { display: flex; flex-direction: column; }
    .page-header-subtitle { font-size: 14px; font-weight: 500; color: var(--return-dark); opacity: 0.8; }
    
    .alert { padding: 16px 20px; border-radius: 12px; margin-bottom: 20px; display: flex; align-items: center; gap: 12px; }
    .alert-success { background: var(--return-light); border: 1px solid var(--return-primary); color: var(--return-dark); }
    .alert-error { background: #fef2f2; border: 1px solid #fca5a5; color: #991b1b; }
    
    .form-card { 
        background: #fff; border-radius: 20px; overflow: hidden; 
        box-shadow: var(--shadow-lg); border: 1px solid #e5e7eb;
    }
    .form-card-header { 
        padding: 24px 28px; 
        background: linear-gradient(135deg, var(--return-primary), var(--return-dark)); 
        position: relative; overflow: hidden;
    }
    .form-card-header::before {
        content: ''; position: absolute; top: -50%; right: -20%; width: 200px; height: 200px;
        background: rgba(255,255,255,0.1); border-radius: 50%;
    }
    .form-card-title { 
        font-size: 18px; font-weight: 700; color: #fff; margin: 0; 
        display: flex; align-items: center; gap: 12px; position: relative; z-index: 1;
    }
    .form-card-body { padding: 28px; }
    
    .form-section { margin-bottom: 32px; padding-bottom: 32px; border-bottom: 2px dashed #e5e7eb; }
    .form-section:last-child { margin-bottom: 0; padding-bottom: 0; border-bottom: none; }
    .form-section-title { 
        font-size: 14px; font-weight: 700; color: var(--return-dark); 
        text-transform: uppercase; letter-spacing: 0.5px; margin-bottom: 20px;
        display: flex; align-items: center; gap: 8px;
    }
    
    .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; }
    @media (max-width: 640px) { .form-row { grid-template-columns: 1fr; } }
    .form-group { margin-bottom: 20px; }
    .form-group:last-child { margin-bottom: 0; }
    .form-label { display: block; font-size: 13px; font-weight: 600; color: #374151; margin-bottom: 8px; }
    .form-label .required { color: #ef4444; margin-left: 2px; }
    .form-control { 
        width: 100%; padding: 14px 18px; border: 2px solid #e5e7eb; border-radius: 12px; 
        font-size: 15px; background: #f9fafb; color: #111827; box-sizing: border-box; 
        transition: all 0.2s; font-weight: 500;
    }
    .form-control:focus { outline: none; border-color: var(--return-primary); background: #fff; box-shadow: 0 0 0 4px rgba(16, 185, 129, 0.15); }
    .form-control::placeholder { color: #9ca3af; font-weight: 400; }
    textarea.form-control { min-height: 100px; resize: vertical; }
    
    /* Product Search */
    .product-search-container { position: relative; }
    .product-search { 
        width: 100%; padding: 16px 20px 16px 54px; border: 2px solid #e5e7eb; border-radius: 14px; 
        font-size: 16px; background: #f9fafb; transition: all 0.2s; box-sizing: border-box; font-weight: 500;
    }
    .product-search:focus { outline: none; border-color: var(--return-primary); background: #fff; box-shadow: 0 0 0 4px rgba(16, 185, 129, 0.12); }
    .product-search-icon { position: absolute; left: 20px; top: 50%; transform: translateY(-50%); width: 22px; height: 22px; color: #9ca3af; pointer-events: none; }
    
    .product-dropdown { 
        position: absolute; top: 100%; left: 0; right: 0; background: #fff; 
        border: 2px solid #e5e7eb; border-radius: 14px; box-shadow: var(--shadow-lg); 
        z-index: 100; max-height: 380px; overflow-y: auto; display: none; margin-top: 8px; 
    }
    .product-dropdown.show { display: block; }
    
    .product-item { 
        display: flex; align-items: center; gap: 16px; padding: 16px 18px; 
        cursor: pointer; transition: all 0.15s; border-bottom: 1px solid #f3f4f6; 
    }
    .product-item:last-child { border-bottom: none; }
    .product-item:hover { background: var(--return-bg); }
    .product-item-image { width: 52px; height: 52px; border-radius: 12px; object-fit: cover; border: 2px solid #e5e7eb; flex-shrink: 0; }
    .product-item-placeholder { 
        width: 52px; height: 52px; border-radius: 12px; 
        background: linear-gradient(135deg, var(--return-primary), var(--return-dark)); 
        display: flex; align-items: center; justify-content: center; 
        color: #fff; font-weight: 700; font-size: 16px; flex-shrink: 0; 
    }
    .product-item-info { flex: 1; min-width: 0; }
    .product-item-name { font-weight: 600; color: #111827; font-size: 15px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
    .product-item-meta { display: flex; align-items: center; gap: 8px; margin-top: 6px; flex-wrap: wrap; }
    .product-item-sku { font-size: 12px; color: #6b7280; font-family: monospace; background: #f3f4f6; padding: 2px 8px; border-radius: 4px; }
    .product-item-badge { font-size: 11px; font-weight: 600; padding: 4px 10px; border-radius: 20px; }
    .product-item-badge.batch { background: #fef3c7; color: #92400e; }
    .product-item-badge.variant { background: #dbeafe; color: #1e40af; }
    .product-item-stock { text-align: right; flex-shrink: 0; }
    .product-item-stock-value { font-size: 18px; font-weight: 800; color: var(--return-primary); }
    .product-item-stock-label { font-size: 11px; color: #6b7280; }
    .product-no-results { padding: 32px; text-align: center; color: #6b7280; display: none; }
    
    /* Selected Product */
    .selected-product { 
        display: none; background: linear-gradient(135deg, var(--return-bg), var(--return-light)); 
        border: 2px solid var(--return-primary); border-radius: 16px; padding: 20px; margin-top: 16px; 
    }
    .selected-product.show { display: block; animation: slideIn 0.3s ease; }
    @keyframes slideIn { from { opacity: 0; transform: translateY(-10px); } to { opacity: 1; transform: translateY(0); } }
    
    .selected-product-header { display: flex; align-items: center; gap: 18px; }
    .selected-product-header img { width: 72px; height: 72px; border-radius: 14px; object-fit: cover; border: 3px solid #fff; box-shadow: var(--shadow-md); }
    .selected-product-header .placeholder { 
        width: 72px; height: 72px; border-radius: 14px; 
        background: linear-gradient(135deg, var(--return-primary), var(--return-dark)); 
        display: flex; align-items: center; justify-content: center; 
        color: #fff; font-weight: 700; font-size: 22px; box-shadow: var(--shadow-md);
    }
    .selected-product-info { flex: 1; }
    .selected-product-name { font-size: 20px; font-weight: 700; color: var(--return-dark); }
    .selected-product-sku { font-size: 13px; color: var(--return-primary); margin-top: 4px; font-family: monospace; }
    .selected-product-badges { display: flex; gap: 8px; margin-top: 10px; flex-wrap: wrap; }
    .selected-product-clear { 
        width: 40px; height: 40px; border-radius: 12px; border: none; 
        background: #fff; color: var(--return-dark); cursor: pointer; 
        display: flex; align-items: center; justify-content: center; 
        transition: all 0.2s; flex-shrink: 0; box-shadow: var(--shadow-sm);
    }
    .selected-product-clear:hover { background: #fee2e2; color: #dc2626; transform: scale(1.05); }
    
    /* Price Display */
    .selected-product-prices { 
        display: grid; grid-template-columns: repeat(4, 1fr); gap: 12px; 
        margin-top: 20px; padding-top: 20px; border-top: 2px dashed var(--return-primary); 
    }
    @media (max-width: 640px) { .selected-product-prices { grid-template-columns: repeat(2, 1fr); } }
    .price-box { background: #fff; border-radius: 12px; padding: 14px; text-align: center; box-shadow: var(--shadow-sm); }
    .price-box-label { font-size: 11px; color: #6b7280; text-transform: uppercase; font-weight: 600; margin-bottom: 6px; }
    .price-box-value { font-size: 18px; font-weight: 800; color: var(--return-dark); }
    .price-box-source { font-size: 10px; color: var(--return-primary); margin-top: 4px; }
    
    /* Variation Selection */
    .variation-section { display: none; margin-top: 20px; padding-top: 20px; border-top: 2px dashed var(--return-primary); }
    .variation-section.show { display: block; }
    .variation-section-title { font-size: 14px; font-weight: 700; color: var(--return-dark); margin-bottom: 14px; }
    .variation-section.show { display: block; }
    .variation-section-title { font-size: 13px; font-weight: 600; color: #0e7490; margin-bottom: 12px; }
    
    .variation-grid { display: flex; flex-wrap: wrap; gap: 10px; }
    .variation-chip { display: flex; align-items: center; gap: 8px; padding: 10px 14px; background: #fff; border: 2px solid #d1d5db; border-radius: 10px; cursor: pointer; transition: all 0.15s; }
    .variation-chip:hover { border-color: #0891b2; }
    .variation-chip.selected { border-color: #0891b2; background: #cffafe; }
    .variation-chip-color { width: 20px; height: 20px; border-radius: 50%; border: 2px solid rgba(0,0,0,0.1); flex-shrink: 0; }
    .variation-chip-image { width: 32px; height: 32px; border-radius: 6px; object-fit: cover; border: 1px solid #d1d5db; }
    .variation-chip-name { font-size: 13px; font-weight: 500; color: var(--text-primary); }
    .variation-chip-stock { font-size: 11px; color: var(--text-muted); margin-left: 8px; }
    .variation-chip-stock.low { color: #dc2626; }
    .variation-chip-price { font-size: 11px; font-weight: 600; color: #0891b2; margin-left: 4px; background: #cffafe; padding: 2px 6px; border-radius: 4px; }
    
    /* Lot Section */
    .lot-section { display: none; background: #fffbeb; border: 2px solid #fcd34d; border-radius: 14px; padding: 18px; margin-top: 16px; }
    .lot-section.show { display: block; }
    .lot-section-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 14px; }
    .lot-section-title { font-size: 14px; font-weight: 700; color: #92400e; }
    
    .lot-list { max-height: 220px; overflow-y: auto; }
    .lot-item { display: flex; align-items: center; gap: 14px; padding: 12px 14px; background: #fff; border: 2px solid #e5e7eb; border-radius: 10px; margin-bottom: 8px; cursor: pointer; transition: all 0.15s; }
    .lot-item:last-child { margin-bottom: 0; }
    .lot-item:hover { border-color: #f59e0b; }
    .lot-item.selected { border-color: #f59e0b; background: #fef3c7; }
    
    .lot-item-icon { width: 40px; height: 40px; border-radius: 10px; display: flex; align-items: center; justify-content: center; font-size: 18px; flex-shrink: 0; }
    .lot-item-icon.ok { background: #d1fae5; }
    .lot-item-icon.warn { background: #fef3c7; }
    .lot-item-icon.bad { background: #fee2e2; }
    
    .lot-item-info { flex: 1; min-width: 0; }
    .lot-item-header { display: flex; align-items: center; gap: 8px; margin-bottom: 4px; flex-wrap: wrap; }
    .lot-item-no { font-weight: 700; font-size: 13px; color: var(--text-primary); }
    .lot-item-batch { font-size: 11px; color: var(--text-muted); background: #f3f4f6; padding: 2px 6px; border-radius: 4px; }
    .lot-item-expiry { font-size: 10px; padding: 3px 8px; border-radius: 10px; font-weight: 600; margin-left: auto; }
    .lot-item-expiry.ok { background: #d1fae5; color: #065f46; }
    .lot-item-expiry.warn { background: #fef3c7; color: #92400e; }
    .lot-item-expiry.bad { background: #fee2e2; color: #991b1b; }
    
    .lot-item-meta { display: flex; gap: 12px; font-size: 11px; color: var(--text-muted); }
    
    .lot-item-prices { text-align: right; flex-shrink: 0; }
    .lot-item-price { font-size: 14px; font-weight: 700; color: #ea580c; }
    .lot-item-price-label { font-size: 10px; color: var(--text-muted); }
    .lot-item-stock { font-size: 12px; font-weight: 600; color: var(--text-primary); margin-top: 4px; }
    
    .lot-no-items { text-align: center; padding: 20px; color: var(--text-muted); font-size: 13px; }
    
    /* Selected Lot Display */
    .selected-lot { display: none; background: #fff; border: 2px solid #f59e0b; border-radius: 12px; padding: 14px; margin-top: 12px; }
    .selected-lot.show { display: flex; align-items: center; gap: 14px; flex-wrap: wrap; }
    .selected-lot-icon { width: 48px; height: 48px; border-radius: 10px; background: #fef3c7; display: flex; align-items: center; justify-content: center; font-size: 22px; flex-shrink: 0; }
    .selected-lot-info { flex: 1; min-width: 150px; }
    .selected-lot-no { font-size: 15px; font-weight: 700; color: var(--text-primary); }
    .selected-lot-meta { display: flex; gap: 12px; margin-top: 4px; font-size: 11px; color: var(--text-muted); flex-wrap: wrap; }
    .selected-lot-prices { display: flex; gap: 16px; }
    .selected-lot-price { text-align: center; }
    .selected-lot-price-value { font-size: 16px; font-weight: 700; color: #ea580c; }
    .selected-lot-price-label { font-size: 10px; color: var(--text-muted); }
    .selected-lot-clear { padding: 8px; border: none; background: transparent; color: #92400e; cursor: pointer; border-radius: 8px; }
    .selected-lot-clear:hover { background: #fef3c7; }
    
    /* Current Stock Display */
    .current-stock-display { display: none; background: var(--body-bg); border: 1px solid var(--card-border); border-radius: 10px; padding: 14px; margin-top: 12px; }
    .current-stock-display.show { display: flex; align-items: center; gap: 16px; }
    .current-stock-label { font-size: 12px; color: var(--text-muted); }
    .current-stock-value { font-size: 22px; font-weight: 700; color: #0891b2; }
    
    /* Form Actions */
    .form-actions { display: flex; gap: 12px; padding-top: 24px; border-top: 1px solid var(--card-border); margin-top: 24px; }
    .btn { display: inline-flex; align-items: center; justify-content: center; gap: 8px; padding: 14px 28px; border-radius: 10px; font-size: 14px; font-weight: 600; cursor: pointer; border: none; text-decoration: none; transition: all 0.2s; }
    .btn-primary { background: linear-gradient(135deg, var(--return-primary), var(--return-dark)); color: #fff; box-shadow: var(--shadow-md); }
    .btn-primary:hover { transform: translateY(-3px); box-shadow: 0 8px 25px rgba(16, 185, 129, 0.35); }
    .btn-secondary { background: var(--body-bg); color: var(--text-primary); border: 1px solid var(--card-border); }
    .btn-secondary:hover { background: var(--card-border); }
    
    .alert { padding: 14px 18px; border-radius: 10px; margin-bottom: 20px; font-size: 14px; }
    .alert-success { background: #d1fae5; color: #065f46; border: 1px solid #a7f3d0; }
    .alert-error { background: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }
</style>

<div class="page-container">
    <div class="page-header">
        <a href="{{ route('inventory.stock.movements') }}" class="back-btn">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M15 19l-7-7 7-7"/></svg>
        </a>
        <h1>
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6"/></svg>
            Stock Returns
        </h1>
    </div>
    
    @if(session('success'))<div class="alert alert-success">{{ session('success') }}</div>@endif
    @if(session('error'))<div class="alert alert-error">{{ session('error') }}</div>@endif
    @if($errors->any())
    <div class="alert alert-error">
        <strong>Please fix the following errors:</strong>
        <ul style="margin: 8px 0 0 20px;">
            @foreach($errors->all() as $error)
                <li>{{ $error }}</li>
            @endforeach
        </ul>
    </div>
    @endif
    
    
    
    <div class="form-card">
        <div class="form-card-header">
            <h3 class="form-card-title">↩️ Process Stock Return</h3>
        </div>
        <div class="form-card-body">
            <form action="{{ route('inventory.stock.returns.store') }}" method="POST" id="mainForm">
                @csrf
                <input type="hidden" name="product_id" id="product_id">
                <input type="hidden" name="variation_id" id="variation_id">
                <input type="hidden" name="lot_id" id="lot_id">
                
                <!-- Product Selection -->
                <div class="form-section">
                    <div class="form-section-title">📦 Select Product</div>
                    
                    <div class="product-search-container">
                        <svg class="product-search-icon" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/></svg>
                        <input type="text" class="product-search" id="productSearch" placeholder="Search by name or SKU..." autocomplete="off">
                        
                        <div class="product-dropdown" id="productDropdown">
                            @foreach($products as $product)
                            @php
                                $img = $product->images->where('is_primary', true)->first();
                                if(!$img) $img = $product->images->first();
                                $imgUrl = $img ? asset('storage/' . $img->image_path) : null;
                                $totalStock = $product->stockLevels->sum('qty');
                                
                                $variationsData = [];
                                foreach($product->variations as $v) {
                                    $vImg = $v->image_path ? asset('storage/' . $v->image_path) : null;
                                    $vColor = null;
                                    if($v->attributeValues && $v->attributeValues->count() > 0) {
                                        $firstAttr = $v->attributeValues->first();
                                        $vColor = $firstAttr ? $firstAttr->color_code : null;
                                    }
                                    $vName = $v->variation_name;
                                    if (!$vName && $v->attributeValues) {
                                        $parts = [];
                                        foreach($v->attributeValues as $av) { $parts[] = $av->value; }
                                        $vName = implode(' / ', $parts);
                                    }
                                    if (!$vName) $vName = $v->sku;
                                    
                                    $variationsData[] = [
                                        'id' => $v->id, 'sku' => $v->sku, 'name' => $vName,
                                        'stock' => $v->stockLevels ? $v->stockLevels->sum('qty') : 0,
                                        'purchase_price' => $v->purchase_price, 'sale_price' => $v->sale_price,
                                        'effective_purchase' => $v->purchase_price ?? $product->purchase_price ?? 0,
                                        'effective_sale' => $v->sale_price ?? $product->sale_price ?? 0,
                                        'image' => $vImg, 'color' => $vColor
                                    ];
                                }
                                $hasRealVariants = count($variationsData) > 0;
                            @endphp
                            <div class="product-item" 
                                 data-id="{{ $product->id }}"
                                 data-name="{{ $product->name }}"
                                 data-sku="{{ $product->sku }}"
                                 data-barcode="{{ $product->barcode }}"
                                 data-image="{{ $imgUrl }}"
                                 data-unit="{{ $product->unit->short_name ?? 'PCS' }}"
                                 data-unit-id="{{ $product->unit_id }}"
                                 data-purchase="{{ $product->purchase_price ?? 0 }}"
                                 data-sale="{{ $product->sale_price ?? 0 }}"
                                 data-batch="{{ $product->is_batch_managed ? '1' : '0' }}"
                                 data-variants="{{ $hasRealVariants ? '1' : '0' }}"
                                 data-stock="{{ $totalStock }}"
                                 data-variations="{{ base64_encode(json_encode($variationsData)) }}">
                                @if($imgUrl)
                                    <img src="{{ $imgUrl }}" alt="" class="product-item-image">
                                @else
                                    <div class="product-item-placeholder">{{ strtoupper(substr($product->name, 0, 2)) }}</div>
                                @endif
                                <div class="product-item-info">
                                    <div class="product-item-name">{{ $product->name }}</div>
                                    <div class="product-item-meta">
                                        <span class="product-item-sku">{{ $product->sku }}</span>
                                        @if($product->is_batch_managed)<span class="product-item-badge batch">Batch</span>@endif
                                        @if($hasRealVariants)<span class="product-item-badge variant">{{ count($variationsData) }} Variants</span>@endif
                                    </div>
                                </div>
                                <div class="product-item-stock">
                                    <div class="product-item-stock-value">{{ number_format($totalStock, 0) }}</div>
                                    <div class="product-item-stock-label">{{ $product->unit->short_name ?? 'PCS' }}</div>
                                </div>
                            </div>
                            @endforeach
                            <div class="product-no-results" id="noProductsMsg">No products found</div>
                        </div>
                    </div>
                    
                    <!-- Selected Product Display -->
                    <div class="selected-product" id="selectedProduct">
                        <div class="selected-product-header">
                            <div id="selectedProductImage"></div>
                            <div class="selected-product-info">
                                <div class="selected-product-name" id="selectedProductName"></div>
                                <div class="selected-product-sku" id="selectedProductSku"></div>
                                <div class="selected-product-badges" id="selectedProductBadges"></div>
                            </div>
                            <button type="button" class="selected-product-clear" onclick="clearProduct()">
                                <svg width="20" height="20" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/></svg>
                            </button>
                        </div>
                        
                        <!-- Variation Selection -->
                        <div class="variation-section" id="variationSection">
                            <div class="variation-section-title">🎨 Select Variation: <span class="required">*</span></div>
                            <div class="variation-grid" id="variationGrid"></div>
                        </div>
                        
                        <!-- Price Display -->
                        <div class="selected-product-prices">
                            <div class="price-box">
                                <div class="price-box-label">Purchase</div>
                                <div class="price-box-value" id="pricePurchase">₹0</div>
                                <div class="price-box-source" id="pricePurchaseSource">Product</div>
                            </div>
                            <div class="price-box">
                                <div class="price-box-label">Sale</div>
                                <div class="price-box-value" id="priceSale">₹0</div>
                                <div class="price-box-source" id="priceSaleSource">Product</div>
                            </div>
                            <div class="price-box">
                                <div class="price-box-label">Stock</div>
                                <div class="price-box-value" id="priceStock">0</div>
                                <div class="price-box-source" id="priceStockUnit">PCS</div>
                            </div>
                            <div class="price-box">
                                <div class="price-box-label">Unit</div>
                                <div class="price-box-value" id="priceUnit">-</div>
                                <div class="price-box-source">Base Unit</div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Lot Selection (FEFO) -->
                    <div class="lot-section" id="lotSection">
                        <div class="lot-section-header">
                            <div class="lot-section-title">🏷️ Select Lot (FEFO - First Expiry First Out)</div>
                        </div>
                        <div class="lot-list" id="lotList"></div>
                    </div>
                    
                    <!-- Selected Lot Display -->
                    <div class="selected-lot" id="selectedLot">
                        <div class="selected-lot-icon">🏷️</div>
                        <div class="selected-lot-info">
                            <div class="selected-lot-no" id="selectedLotNo"></div>
                            <div class="selected-lot-meta" id="selectedLotMeta"></div>
                        </div>
                        <div class="selected-lot-prices">
                            <div class="selected-lot-price">
                                <div class="selected-lot-price-value" id="selectedLotPurchase">₹0</div>
                                <div class="selected-lot-price-label">Purchase</div>
                            </div>
                            <div class="selected-lot-price">
                                <div class="selected-lot-price-value" id="selectedLotSale">₹0</div>
                                <div class="selected-lot-price-label">Sale</div>
                            </div>
                        </div>
                        <button type="button" class="selected-lot-clear" onclick="clearLot()">
                            <svg width="18" height="18" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/></svg>
                        </button>
                    </div>
                </div>
                
                <!-- Location Section -->
                <div class="form-section">
                    <div class="form-section-title">📍 Return Location</div>
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Warehouse <span class="required">*</span></label>
                            <select name="warehouse_id" id="warehouse_id" class="form-control" required>
                                <option value="">Select warehouse...</option>
                                @foreach($warehouses as $wh)
                                    <option value="{{ $wh->id }}" {{ $wh->is_default ? 'selected' : '' }}>{{ $wh->name }} {{ $wh->is_default ? '(Default)' : '' }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Rack / Location</label>
                            <select name="rack_id" id="rack_id" class="form-control">
                                <option value="">Select rack (optional)...</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="current-stock-display" id="currentStockDisplay">
                        <div>
                            <div class="current-stock-label">Current Stock at Location</div>
                            <div class="current-stock-value" id="currentStockValue">0 PCS</div>
                        </div>
                    </div>
                </div>
                
                <!-- Quantity Section -->
                <div class="form-section">
                    <div class="form-section-title">🔢 Quantity & Details</div>
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Return Quantity <span class="required">*</span></label>
                            <input type="number" name="qty" id="qty" class="form-control" step="any" min="0.001" placeholder="Enter quantity" required>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Unit <span class="required">*</span></label>
                            <select name="unit_id" id="unit_id" class="form-control" required>
                                <option value="">Select unit...</option>
                                @foreach($units as $u)<option value="{{ $u->id }}">{{ $u->name }} ({{ $u->short_name }})</option>@endforeach
                            </select>
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Return Reason <span class="required">*</span></label>
                            <select name="reason" id="reason" class="form-control" required>
                                <option value="">Select reason...</option>
                                <option value="CUSTOMER_RETURN">Customer Return</option>
                                <option value="DAMAGED">Damaged Goods</option>
                                <option value="WRONG_ITEM">Wrong Item Shipped</option>
                                <option value="DEFECTIVE">Defective Product</option>
                                <option value="OTHER">Other</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Reference Number</label>
                            <input type="text" name="reference_no" class="form-control" placeholder="Return slip, RMA number, etc.">
                        </div>
                    </div>
                </div>
                
                <!-- Notes -->
                <div class="form-section">
                    <div class="form-group">
                        <label class="form-label">Notes</label>
                        <textarea name="notes" class="form-control" placeholder="Additional notes about this return..."></textarea>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">↩️ Process Return</button>
                    <a href="{{ route('inventory.stock.movements') }}" class="btn btn-secondary">Cancel</a>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
var selectedProduct = null;
var selectedVariation = null;
var selectedLotData = null;
var productVariations = [];
var productPrices = { purchase: 0, sale: 0 };
var baseUnit = 'PCS';
var requiresVariation = false;
var isBatchManaged = false;
var lots = [];

// Product Search
document.getElementById('productSearch').addEventListener('input', function() {
    var search = this.value.toLowerCase();
    var dropdown = document.getElementById('productDropdown');
    var items = dropdown.querySelectorAll('.product-item');
    var found = 0;
    
    items.forEach(function(item) {
        var name = (item.dataset.name || '').toLowerCase();
        var sku = (item.dataset.sku || '').toLowerCase();
        var barcode = (item.dataset.barcode || '').toLowerCase();
        
        if (name.indexOf(search) !== -1 || sku.indexOf(search) !== -1 || barcode.indexOf(search) !== -1) {
            item.style.display = '';
            found++;
        } else {
            item.style.display = 'none';
        }
    });
    
    document.getElementById('noProductsMsg').style.display = found === 0 ? 'block' : 'none';
    dropdown.classList.add('show');
});

document.getElementById('productSearch').addEventListener('focus', function() {
    document.getElementById('productDropdown').classList.add('show');
});

document.querySelectorAll('.product-item').forEach(function(item) {
    item.addEventListener('click', function() {
        selectProduct(this);
    });
});

document.addEventListener('click', function(e) {
    if (!e.target.closest('.product-search-container')) {
        document.getElementById('productDropdown').classList.remove('show');
    }
});

function selectProduct(item) {
    selectedProduct = item;
    selectedVariation = null;
    selectedLotData = null;
    
    document.getElementById('product_id').value = item.dataset.id;
    document.getElementById('variation_id').value = '';
    document.getElementById('lot_id').value = '';
    document.getElementById('unit_id').value = item.dataset.unitId || '';
    document.getElementById('productSearch').style.display = 'none';
    document.getElementById('productDropdown').classList.remove('show');
    
    // Update selected display
    var imgHtml = item.dataset.image 
        ? '<img src="' + item.dataset.image + '" alt="">'
        : '<div class="placeholder">' + item.dataset.name.substring(0,2).toUpperCase() + '</div>';
    document.getElementById('selectedProductImage').innerHTML = imgHtml;
    document.getElementById('selectedProductName').textContent = item.dataset.name;
    document.getElementById('selectedProductSku').textContent = 'SKU: ' + item.dataset.sku;
    
    var badges = '';
    if (item.dataset.batch === '1') badges += '<span class="product-item-badge batch">Batch Managed</span>';
    if (item.dataset.variants === '1') badges += '<span class="product-item-badge variant">Has Variants</span>';
    document.getElementById('selectedProductBadges').innerHTML = badges;
    
    productPrices = { purchase: parseFloat(item.dataset.purchase) || 0, sale: parseFloat(item.dataset.sale) || 0 };
    baseUnit = item.dataset.unit || 'PCS';
    isBatchManaged = item.dataset.batch === '1';
    requiresVariation = item.dataset.variants === '1';
    
    updatePriceDisplay(productPrices.purchase, productPrices.sale, 'Product', 'Product');
    document.getElementById('priceStock').textContent = item.dataset.stock || '0';
    document.getElementById('priceStockUnit').textContent = baseUnit;
    document.getElementById('priceUnit').textContent = baseUnit;
    
    document.getElementById('selectedProduct').classList.add('show');
    
    // Handle variations
    var hasVariantsFlag = item.dataset.variants === '1';
    if (hasVariantsFlag) {
        try {
            var b64 = item.dataset.variations || '';
            productVariations = JSON.parse(atob(b64));
        } catch(e) {
            console.log('Base64 decode failed, trying AJAX...');
            productVariations = [];
        }
        
        // If parsing worked and we have variations
        if (productVariations.length > 0) {
            requiresVariation = true;
            renderVariations();
            document.getElementById('variationSection').classList.add('show');
        } else {
            // Try AJAX fallback
            loadVariationsViaAjax(item.dataset.id);
        }
    } else {
        document.getElementById('variationSection').classList.remove('show');
        productVariations = [];
        requiresVariation = false;
    }
    
    // Handle lots
    if (isBatchManaged) {
        loadLots();
    } else {
        document.getElementById('lotSection').classList.remove('show');
    }
    loadCurrentStock();
    loadRacks();
}

// AJAX fallback to load variations
function loadVariationsViaAjax(productId) {
    fetch('{{ url("admin/inventory/products") }}/' + productId + '/variations')
        .then(function(r) { return r.json(); })
        .then(function(data) {
            console.log('AJAX loaded variations:', data);
            
            if (data.success && data.variations && data.variations.length > 0) {
                productVariations = data.variations.map(function(v) {
                    var name = v.variation_name;
                    if (!name && v.attributes && v.attributes.length > 0) {
                        name = v.attributes.map(function(a) { return a.value; }).join(' / ');
                    }
                    if (!name) name = v.sku;
                    
                    var color = null;
                    if (v.attributes && v.attributes.length > 0) {
                        v.attributes.forEach(function(a) {
                            if (a.color_code) color = a.color_code;
                        });
                    }
                    
                    return {
                        id: v.id, sku: v.sku, name: name,
                        stock: v.stock_qty || 0,
                        purchase_price: v.purchase_price,
                        sale_price: v.sale_price,
                        effective_purchase: v.purchase_price || productPrices.purchase,
                        effective_sale: v.sale_price || productPrices.sale,
                        image: v.image_url, color: color
                    };
                });
                
                requiresVariation = true;
                renderVariations();
                document.getElementById('variationSection').classList.add('show');
            } else {
                // No variations found - product has flag but no actual variations
                requiresVariation = false;
                document.getElementById('variationSection').classList.remove('show');
            }
        })
        .catch(function(err) {
            console.error('Failed to load variations:', err);
            requiresVariation = false;
        });
}

function renderVariations() {
    var grid = document.getElementById('variationGrid');
    var html = '';
    
    productVariations.forEach(function(v) {
        var colorHtml = v.color ? '<div class="variation-chip-color" style="background:' + v.color + '"></div>' : '';
        var imgHtml = v.image ? '<img src="' + v.image + '" class="variation-chip-image">' : '';
        var stockClass = v.stock < 10 ? 'low' : '';
        var price = v.effective_sale || v.sale_price || productPrices.sale;
        
        html += '<div class="variation-chip" data-id="' + v.id + '" onclick="selectVariation(' + v.id + ')">';
        html += colorHtml + imgHtml;
        html += '<span class="variation-chip-name">' + v.name + '</span>';
        html += '<span class="variation-chip-stock ' + stockClass + '">' + v.stock + '</span>';
        if (price > 0) html += '<span class="variation-chip-price">₹' + price + '</span>';
        html += '</div>';
    });
    
    grid.innerHTML = html;
}

function selectVariation(id) {
    selectedVariation = productVariations.find(function(v) { return v.id == id; });
    document.getElementById('variation_id').value = id;
    
    document.querySelectorAll('.variation-chip').forEach(function(c) {
        c.classList.remove('selected');
    });
    document.querySelector('.variation-chip[data-id="' + id + '"]').classList.add('selected');
    
    if (selectedVariation) {
        var purchase = selectedVariation.purchase_price || productPrices.purchase;
        var sale = selectedVariation.sale_price || productPrices.sale;
        var source = selectedVariation.sale_price ? 'Variation' : 'Product';
        updatePriceDisplay(purchase, sale, source, source);
        document.getElementById('priceStock').textContent = selectedVariation.stock || '0';
    }
    
    if (isBatchManaged) loadLots();
    loadCurrentStock();
}

function updatePriceDisplay(purchase, sale, purchaseSource, saleSource) {
    document.getElementById('pricePurchase').textContent = '₹' + (purchase || 0).toLocaleString();
    document.getElementById('priceSale').textContent = '₹' + (sale || 0).toLocaleString();
    document.getElementById('pricePurchaseSource').textContent = purchaseSource;
    document.getElementById('priceSaleSource').textContent = saleSource;
}

function loadLots() {
    var productId = document.getElementById('product_id').value;
    var warehouseId = document.getElementById('warehouse_id').value;
    var variationId = document.getElementById('variation_id').value;
    
    if (!productId) return;
    
    var url = '{{ url("admin/inventory/stock/product-lots") }}?product_id=' + productId;
    if (warehouseId) url += '&warehouse_id=' + warehouseId;
    if (variationId) url += '&variation_id=' + variationId;
    
    fetch(url)
        .then(function(r) { return r.json(); })
        .then(function(data) {
            lots = data.lots || [];
            renderLots();
            document.getElementById('lotSection').classList.add('show');
        });
}

function renderLots() {
    var list = document.getElementById('lotList');
    
    if (lots.length === 0) {
        list.innerHTML = '<div class="lot-no-items">No lots available</div>';
        return;
    }
    
    var html = '';
    lots.forEach(function(lot) {
        var expClass = 'ok', icon = '✅';
        if (lot.days_until_expiry !== null) {
            if (lot.days_until_expiry < 0) { expClass = 'bad'; icon = '🚫'; }
            else if (lot.days_until_expiry <= 30) { expClass = 'warn'; icon = '⚠️'; }
        }
        
        html += '<div class="lot-item" data-id="' + lot.id + '" onclick="selectLot(' + lot.id + ')">';
        html += '<div class="lot-item-icon ' + expClass + '">' + icon + '</div>';
        html += '<div class="lot-item-info">';
        html += '<div class="lot-item-header">';
        html += '<span class="lot-item-no">' + lot.lot_number + '</span>';
        if (lot.batch_number) html += '<span class="lot-item-batch">' + lot.batch_number + '</span>';
        if (lot.expiry_date) html += '<span class="lot-item-expiry ' + expClass + '">' + lot.expiry_display + '</span>';
        html += '</div>';
        html += '<div class="lot-item-meta">';
        if (lot.mfg_date) html += '<span>Mfg: ' + lot.mfg_date + '</span>';
        html += '</div></div>';
        html += '<div class="lot-item-prices">';
        html += '<div class="lot-item-price">₹' + (lot.sale_price || lot.purchase_price || 0) + '</div>';
        html += '<div class="lot-item-stock">' + lot.qty + ' ' + baseUnit + '</div>';
        html += '</div></div>';
    });
    
    list.innerHTML = html;
}

function selectLot(id) {
    selectedLotData = lots.find(function(l) { return l.id == id; });
    document.getElementById('lot_id').value = id;
    
    document.querySelectorAll('.lot-item').forEach(function(i) { i.classList.remove('selected'); });
    document.querySelector('.lot-item[data-id="' + id + '"]').classList.add('selected');
    
    if (selectedLotData) {
        document.getElementById('selectedLotNo').textContent = selectedLotData.lot_number;
        var meta = [];
        if (selectedLotData.batch_number) meta.push('Batch: ' + selectedLotData.batch_number);
        if (selectedLotData.expiry_date) meta.push('Exp: ' + selectedLotData.expiry_display);
        meta.push('Stock: ' + selectedLotData.qty + ' ' + baseUnit);
        document.getElementById('selectedLotMeta').innerHTML = meta.join(' • ');
        
        document.getElementById('selectedLotPurchase').textContent = '₹' + (selectedLotData.purchase_price || 0);
        document.getElementById('selectedLotSale').textContent = '₹' + (selectedLotData.sale_price || 0);
        document.getElementById('selectedLot').classList.add('show');
        
        // Update price display with lot prices
        var purchase = selectedLotData.purchase_price || (selectedVariation ? selectedVariation.effective_purchase : productPrices.purchase);
        var sale = selectedLotData.sale_price || (selectedVariation ? selectedVariation.effective_sale : productPrices.sale);
        updatePriceDisplay(purchase, sale, selectedLotData.purchase_price ? 'Lot' : 'Product', selectedLotData.sale_price ? 'Lot' : 'Product');
    }
}

function clearLot() {
    selectedLotData = null;
    document.getElementById('lot_id').value = '';
    document.querySelectorAll('.lot-item').forEach(function(i) { i.classList.remove('selected'); });
    document.getElementById('selectedLot').classList.remove('show');
}

function clearProduct() {
    selectedProduct = null;
    selectedVariation = null;
    selectedLotData = null;
    productVariations = [];
    lots = [];
    
    document.getElementById('product_id').value = '';
    document.getElementById('variation_id').value = '';
    document.getElementById('lot_id').value = '';
    document.getElementById('productSearch').style.display = '';
    document.getElementById('productSearch').value = '';
    document.getElementById('selectedProduct').classList.remove('show');
    document.getElementById('variationSection').classList.remove('show');
    document.getElementById('lotSection').classList.remove('show');
    document.getElementById('selectedLot').classList.remove('show');
}

var loadingRacks = false;
function loadRacks() {
    if (loadingRacks) return;
    
    var warehouseId = document.getElementById('warehouse_id').value;
    var rackSelect = document.getElementById('rack_id');
    
    rackSelect.innerHTML = '<option value="">Select rack (optional)...</option>';
    
    if (warehouseId) {
        loadingRacks = true;
        fetch('{{ url("admin/inventory/racks/by-warehouse") }}/' + warehouseId)
            .then(function(r) { return r.json(); })
            .then(function(racks) {
                rackSelect.innerHTML = '<option value="">Select rack (optional)...</option>';
                racks.forEach(function(rack) {
                    var opt = document.createElement('option');
                    opt.value = rack.id;
                    opt.textContent = rack.code + ' - ' + rack.name;
                    rackSelect.appendChild(opt);
                });
                loadingRacks = false;
            })
            .catch(function() {
                loadingRacks = false;
            });
    }
}

document.getElementById('warehouse_id').addEventListener('change', function() {
    loadRacks();
    if (isBatchManaged) loadLots();
    loadCurrentStock();
});

document.getElementById('rack_id').addEventListener('change', loadCurrentStock);

function loadCurrentStock() {
    var productId = document.getElementById('product_id').value;
    var warehouseId = document.getElementById('warehouse_id').value;
    var rackId = document.getElementById('rack_id').value;
    var variationId = document.getElementById('variation_id').value;
    var lotId = document.getElementById('lot_id').value;
    
    if (!productId || !warehouseId) {
        document.getElementById('currentStockDisplay').classList.remove('show');
        return;
    }
    
    var url = '{{ url("admin/inventory/stock/check") }}?product_id=' + productId + '&warehouse_id=' + warehouseId;
    if (rackId) url += '&rack_id=' + rackId;
    if (variationId) url += '&variation_id=' + variationId;
    if (lotId) url += '&lot_id=' + lotId;
    
    fetch(url)
        .then(function(r) { return r.json(); })
        .then(function(data) {
            document.getElementById('currentStockValue').textContent = (data.base_stock || 0) + ' ' + (data.base_unit || baseUnit);
            document.getElementById('currentStockDisplay').classList.add('show');
        })
        .catch(function() {
            document.getElementById('currentStockDisplay').classList.remove('show');
        });
}
// Form validation
document.getElementById('mainForm').addEventListener('submit', function(e) {
    if (!document.getElementById('product_id').value) {
        e.preventDefault();
        alert('Please select a product.');
        return false;
    }
    if (requiresVariation && !document.getElementById('variation_id').value) {
        e.preventDefault();
        alert('Please select a variation.');
        return false;
    }
    if (!document.getElementById('unit_id').value) {
        e.preventDefault();
        alert('Please select a unit.');
        return false;
    }
    return true;
});

//  or SKU handler
// Barcode handler
if (typeof window.onBarcodeScanned === 'undefined') {
    window.onBarcodeScanned = function(code) {
        var items = document.querySelectorAll('.product-item');
        for (var i = 0; i < items.length; i++) {
            if (items[i].dataset.barcode === code || items[i].dataset.sku === code) {
                selectProduct(items[i]);
                return;
            }
        }
        alert('Product not found: ' + code);
    };
}

// Initialize
loadRacks();
loadCurrentStock();
// Auto-select preselected product from URL on page load
(function() {
    @if(isset($preselectedProduct) && $preselectedProduct)
    var preselectedId = '{{ $preselectedProduct->id }}';
    var productItem = document.querySelector('.product-item[data-id="' + preselectedId + '"]');
    if (productItem) {
        selectProduct(productItem);
    }
    @endif
})();
</script>
