<?php

namespace Modules\Inventory\Services;

use Modules\Inventory\Models\Lot;
use Modules\Inventory\Models\Product;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ExpiryAlertService
{
    /**
     * Alert thresholds in days
     */
    const CRITICAL_DAYS = 7;
    const WARNING_DAYS = 30;
    const INFO_DAYS = 90;

    /**
     * Check all lots and send expiry alerts
     * Call this from a scheduled command (daily)
     */
    public static function checkAndSendAlerts(): array
    {
        $results = [
            'critical' => [],
            'warning' => [],
            'info' => [],
            'expired' => [],
            'emails_sent' => 0,
        ];

        // Get all active lots with expiry dates
        $lots = Lot::with(['product.images', 'product.unit', 'stockLevels.warehouse'])
            ->where('status', 'ACTIVE')
            ->whereNotNull('expiry_date')
            ->whereHas('stockLevels', function ($q) {
                $q->where('qty', '>', 0);
            })
            ->orderBy('expiry_date', 'asc')
            ->get();

        $today = Carbon::today();

        foreach ($lots as $lot) {
            $daysToExpiry = $today->diffInDays($lot->expiry_date, false);
            
            $lotData = [
                'lot' => $lot,
                'days' => $daysToExpiry,
                'product_name' => $lot->product->name ?? 'Unknown',
                'lot_no' => $lot->lot_no,
                'batch_no' => $lot->batch_no,
                'expiry_date' => $lot->expiry_date->format('d M Y'),
                'stock' => $lot->current_stock,
                'warehouses' => $lot->stockLevels->map(fn($sl) => [
                    'name' => $sl->warehouse->name ?? 'Unknown',
                    'qty' => $sl->qty,
                ])->toArray(),
            ];

            if ($daysToExpiry < 0) {
                // Already expired
                $results['expired'][] = $lotData;
                
                // Auto-mark as expired if not already
                if ($lot->status !== 'EXPIRED') {
                    $lot->update(['status' => 'EXPIRED']);
                }
            } elseif ($daysToExpiry <= self::CRITICAL_DAYS) {
                $results['critical'][] = $lotData;
            } elseif ($daysToExpiry <= self::WARNING_DAYS) {
                $results['warning'][] = $lotData;
            } elseif ($daysToExpiry <= self::INFO_DAYS) {
                $results['info'][] = $lotData;
            }
        }

        // Send email alerts if there are critical or warning items
        if (!empty($results['critical']) || !empty($results['warning']) || !empty($results['expired'])) {
            $emailSent = self::sendAlertEmail($results);
            if ($emailSent) {
                $results['emails_sent'] = 1;
            }
        }

        return $results;
    }

    /**
     * Send expiry alert email
     */
    protected static function sendAlertEmail(array $results): bool
    {
        try {
            // Get admin emails from settings or config
            $adminEmails = self::getAlertRecipients();
            
            if (empty($adminEmails)) {
                Log::warning('ExpiryAlertService: No recipients configured for expiry alerts');
                return false;
            }

            $data = [
                'critical' => $results['critical'],
                'warning' => $results['warning'],
                'expired' => $results['expired'],
                'critical_count' => count($results['critical']),
                'warning_count' => count($results['warning']),
                'expired_count' => count($results['expired']),
                'generated_at' => now()->format('d M Y, h:i A'),
            ];

            // Send email
            Mail::send('inventory::emails.expiry-alert', $data, function ($message) use ($adminEmails, $data) {
                $message->to($adminEmails)
                    ->subject('⚠️ Inventory Expiry Alert - ' . ($data['critical_count'] + $data['expired_count']) . ' items need attention');
            });

            Log::info('ExpiryAlertService: Alert email sent', [
                'recipients' => $adminEmails,
                'critical' => $data['critical_count'],
                'warning' => $data['warning_count'],
                'expired' => $data['expired_count'],
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('ExpiryAlertService: Failed to send email', [
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Get alert recipients from settings
     */
    protected static function getAlertRecipients(): array
    {
        // Try to get from inventory settings
        $settings = \DB::table('inventory_settings')
            ->where('key', 'expiry_alert_emails')
            ->value('value');
        
        if ($settings) {
            $emails = json_decode($settings, true);
            if (is_array($emails)) {
                return $emails;
            }
            // Single email as string
            return [$settings];
        }

        // Fallback: Try to get admin email from users table
        $adminEmail = \DB::table('users')
            ->where('is_admin', true)
            ->orWhere('role', 'admin')
            ->value('email');
        
        if ($adminEmail) {
            return [$adminEmail];
        }

        // Last fallback: config
        $configEmail = config('inventory.expiry_alert_email');
        if ($configEmail) {
            return is_array($configEmail) ? $configEmail : [$configEmail];
        }

        return [];
    }

    /**
     * Get expiring lots summary for dashboard
     */
    public static function getExpiringSummary(int $days = 30): array
    {
        $today = Carbon::today();
        $endDate = $today->copy()->addDays($days);

        $lots = Lot::with(['product.images', 'product.unit'])
            ->where('status', 'ACTIVE')
            ->whereNotNull('expiry_date')
            ->whereBetween('expiry_date', [$today, $endDate])
            ->whereHas('stockLevels', function ($q) {
                $q->where('qty', '>', 0);
            })
            ->orderBy('expiry_date', 'asc')
            ->limit(20)
            ->get();

        return [
            'lots' => $lots,
            'count' => $lots->count(),
            'total_value' => $lots->sum(function ($lot) {
                return ($lot->purchase_price ?? $lot->product->purchase_price ?? 0) * $lot->current_stock;
            }),
        ];
    }

    /**
     * Get expired lots that need attention
     */
    public static function getExpiredLots(): array
    {
        $lots = Lot::with(['product.images', 'product.unit', 'stockLevels.warehouse'])
            ->where(function ($q) {
                $q->where('status', 'EXPIRED')
                    ->orWhere(function ($q2) {
                        $q2->where('status', 'ACTIVE')
                            ->whereNotNull('expiry_date')
                            ->where('expiry_date', '<', Carbon::today());
                    });
            })
            ->whereHas('stockLevels', function ($q) {
                $q->where('qty', '>', 0);
            })
            ->orderBy('expiry_date', 'asc')
            ->get();

        return [
            'lots' => $lots,
            'count' => $lots->count(),
            'total_qty' => $lots->sum('current_stock'),
            'total_value' => $lots->sum(function ($lot) {
                return ($lot->purchase_price ?? $lot->product->purchase_price ?? 0) * $lot->current_stock;
            }),
        ];
    }

    /**
     * Generate expiry report data
     */
    public static function generateReport(): array
    {
        $today = Carbon::today();

        // Group by expiry timeframe
        $critical = Lot::where('status', 'ACTIVE')
            ->whereNotNull('expiry_date')
            ->whereBetween('expiry_date', [$today, $today->copy()->addDays(self::CRITICAL_DAYS)])
            ->whereHas('stockLevels', fn($q) => $q->where('qty', '>', 0))
            ->count();

        $warning = Lot::where('status', 'ACTIVE')
            ->whereNotNull('expiry_date')
            ->whereBetween('expiry_date', [$today->copy()->addDays(self::CRITICAL_DAYS + 1), $today->copy()->addDays(self::WARNING_DAYS)])
            ->whereHas('stockLevels', fn($q) => $q->where('qty', '>', 0))
            ->count();

        $info = Lot::where('status', 'ACTIVE')
            ->whereNotNull('expiry_date')
            ->whereBetween('expiry_date', [$today->copy()->addDays(self::WARNING_DAYS + 1), $today->copy()->addDays(self::INFO_DAYS)])
            ->whereHas('stockLevels', fn($q) => $q->where('qty', '>', 0))
            ->count();

        $expired = Lot::where(function ($q) use ($today) {
                $q->where('status', 'EXPIRED')
                    ->orWhere(function ($q2) use ($today) {
                        $q2->where('status', 'ACTIVE')
                            ->whereNotNull('expiry_date')
                            ->where('expiry_date', '<', $today);
                    });
            })
            ->whereHas('stockLevels', fn($q) => $q->where('qty', '>', 0))
            ->count();

        return [
            'critical' => $critical,
            'warning' => $warning,
            'info' => $info,
            'expired' => $expired,
            'total_at_risk' => $critical + $warning + $expired,
            'thresholds' => [
                'critical' => self::CRITICAL_DAYS,
                'warning' => self::WARNING_DAYS,
                'info' => self::INFO_DAYS,
            ],
        ];
    }
}
