<?php

namespace Modules\Inventory\Services;

use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Module;
use App\Models\Menu;
use App\Models\Permission;
use App\Services\Admin\CoreMenuService;

/**
 * Inventory Module Service
 * 
 * Provides module configuration, menu definitions, and lifecycle hooks
 * Called by ModuleController during install/uninstall
 */
class InventoryModuleService
{
    /**
     * Module Configuration
     */
    public static function config(): array
    {
        return [
            'name' => 'Inventory',
            'alias' => 'inventory',
            'version' => '1.6.0',
            'description' => 'Complete Inventory Management System with Products, Warehouses, Stock, Lots and more',
            'is_core' => false,
        ];
    }

    /**
     * Module Menus Definition
     */
    public static function menus(): array
    {
        return [
            [
                'menu_name' => 'Inventory',
                'slug' => 'inventory',
                'icon' => 'fa fa-boxes',
                'route' => null,
                'category' => 'inventory',
                'menu_visibility' => 'Admin',
                'sort_order' => 50,
                'actions' => ['read'],
                'children' => [
                    [
                        'menu_name' => 'Dashboard',
                        'slug' => 'dashboard',
                        'icon' => 'fa fa-chart-pie',
                        'route' => 'inventory.dashboard',
                        'sort_order' => 1,
                        'actions' => ['read'],
                    ],
                    [
                        'menu_name' => 'Products',
                        'slug' => 'products',
                        'icon' => 'fa fa-box',
                        'route' => 'inventory.products.index',
                        'sort_order' => 2,
                        'actions' => ['read', 'create', 'edit', 'delete', 'export', 'import'],
                    ],
                    [
                        'menu_name' => 'Warehouses',
                        'slug' => 'warehouses',
                        'icon' => 'fa fa-warehouse',
                        'route' => 'inventory.warehouses.index',
                        'sort_order' => 3,
                        'actions' => ['read', 'create', 'edit', 'delete'],
                    ],
                    [
                        'menu_name' => 'Racks',
                        'slug' => 'racks',
                        'icon' => 'fa fa-pallet',
                        'route' => 'inventory.racks.index',
                        'sort_order' => 4,
                        'actions' => ['read', 'create', 'edit', 'delete'],
                    ],
                    [
                        'menu_name' => 'Stock Movements',
                        'slug' => 'stock',
                        'icon' => 'fa fa-exchange-alt',
                        'route' => 'inventory.stock.movements',
                        'sort_order' => 5,
                        'actions' => ['read', 'create', 'edit'],
                    ],
                    [
                        'menu_name' => 'Delivery Receipts',
                        'slug' => 'delivery',
                        'icon' => 'fa fa-truck',
                        'route' => 'inventory.delivery.index',
                        'sort_order' => 6,
                        'actions' => ['read', 'create', 'edit', 'delete', 'approve', 'export'],
                    ],
                    [
                        'menu_name' => 'Lots & Batches',
                        'slug' => 'lots',
                        'icon' => 'fa fa-layer-group',
                        'route' => 'inventory.lots.index',
                        'sort_order' => 6,
                        'actions' => ['read', 'create', 'edit', 'delete'],
                    ],
                    [
                        'menu_name' => 'Reports',
                        'slug' => 'reports',
                        'icon' => 'fa fa-file-alt',
                        'route' => 'inventory.reports.stock-summary',
                        'sort_order' => 7,
                        'actions' => ['read', 'export'],
                    ],
                    [
                        'menu_name' => 'Settings',
                        'slug' => 'settings',
                        'icon' => 'fa fa-cog',
                        'route' => 'inventory.settings.index',
                        'sort_order' => 8,
                        'actions' => ['read', 'edit'],
                    ],
                ],
            ],
        ];
    }

   /*
    |--------------------------------------------------------------------------
    | Registration Methods
    |--------------------------------------------------------------------------
    */

    public static function register(): Module
    {
        return DB::transaction(function () {
            $config = self::config();
            $module = Module::updateOrCreate(
                ['alias' => $config['alias']],
                [
                    'name' => $config['name'],
                    'description' => $config['description'],
                    'version' => $config['version'],
                    'is_active' => true,
                    'is_installed' => true,
                    'is_core' => $config['is_core'] ?? false,
                    'sort_order' => $config['sort_order'] ?? 0,
                    'installed_at' => now(),
                ]
            );

            self::registerMenusAndPermissions($module);

            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
            
            if (class_exists('App\Services\Admin\CoreMenuService')) {
                CoreMenuService::clearCache();
            }

            return $module;
        });
    }

    protected static function registerMenusAndPermissions(Module $module): void
    {
        $moduleAlias = self::config()['alias'];
        foreach (self::menus() as $menuData) {
            self::createMenuWithPermissions($menuData, $module, $moduleAlias, null);
        }
    }

    protected static function createMenuWithPermissions(array $menuData, Module $module, string $moduleAlias, ?int $parentId): Menu
    {
        $permissionName = "{$moduleAlias}.{$menuData['slug']}.read";
        $menu = Menu::updateOrCreate(
            ['module_id' => $module->id, 'slug' => $menuData['slug']],
            [
                'parent_id' => $parentId,
                'menu_name' => $menuData['menu_name'],
                'icon' => $menuData['icon'] ?? null,
                'route' => $menuData['route'] ?? null,
                'category' => $menuData['category'] ?? 'core',
                'permission_name' => $permissionName,
                'menu_visibility' => 'Admin',
                'sort_order' => $menuData['sort_order'] ?? 0,
                'is_active' => true,
            ]
        );

        foreach ($menuData['actions'] ?? ['read'] as $action) {
            self::createPermission($moduleAlias, $menuData['slug'], $action, $module->id);
        }

        if (!empty($menuData['children'])) {
            foreach ($menuData['children'] as $childData) {
                $childData['category'] = $menuData['category'] ?? 'core';
                self::createMenuWithPermissions($childData, $module, $moduleAlias, $menu->id);
            }
        }

        return $menu;
    }

    protected static function createPermission(string $moduleAlias, string $menuSlug, string $action, int $moduleId): void
    {
        $permissionName = "{$moduleAlias}.{$menuSlug}.{$action}";
        $actionNames = [
            'read' => 'View',
            'create' => 'Create',
            'edit' => 'Edit',
            'delete' => 'Delete',
            'export' => 'Export',
            'import' => 'Import',
            'approve' => 'Approve',
            'print' => 'Print',
        ];

        Permission::updateOrCreate(
            ['name' => $permissionName, 'guard_name' => 'admin'],
            [
                'module_id' => $moduleId,
                'action_name' => $actionNames[$action] ?? ucfirst($action),
                'sort_order' => array_search($action, array_keys($actionNames)) ?: 0,
            ]
        );
    }

    public static function uninstall(): void
    {
        $module = Module::where('alias', self::config()['alias'])->first();
        if (!$module) return;

        DB::transaction(function () use ($module) {
            Permission::where('module_id', $module->id)->delete();
            Menu::where('module_id', $module->id)->delete();
            $module->update(['is_active' => false, 'is_installed' => false]);
            
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
            
            if (class_exists('App\Services\Admin\CoreMenuService')) {
                CoreMenuService::clearCache();
            }
        });
    }

    /*
    |--------------------------------------------------------------------------
    | Helper Methods
    |--------------------------------------------------------------------------
    */

    /**
     * Get module alias
     */
    public static function getAlias(): string
    {
        return self::config()['alias'];
    }

    /**
     * Get module name
     */
    public static function getName(): string
    {
        return self::config()['name'];
    }
}