<?php

namespace Modules\Inventory\Services;

use Illuminate\Support\Collection;
use Modules\Inventory\Models\Product;
use Modules\Inventory\Models\ProductVariation;
use Modules\Inventory\Models\StockLevel;

/**
 * LowStockService - FIXED VERSION
 * 
 * Changes:
 * - Now uses available_qty (qty - reserved_qty) instead of just qty
 * - Added getAvailableQty helper method
 * - Proper handling of reserved stock
 */
class LowStockService
{
    /**
     * Get available quantity (qty - reserved_qty)
     */
    private static function getAvailableQty($query): float
    {
        $result = $query->selectRaw('COALESCE(SUM(qty), 0) - COALESCE(SUM(reserved_qty), 0) as available')
            ->value('available');
        return (float) ($result ?? 0);
    }

    /**
     * Check single product/variation for low stock
     * FIXED: Now considers reserved_qty
     */
    public static function checkAndNotifyProduct(int $productId, ?int $variationId = null): ?array
    {
        $product = Product::find($productId);
        if (!$product || !$product->track_inventory || $product->min_stock_level <= 0) {
            return null;
        }
        
        $stockQuery = StockLevel::where('product_id', $productId);
        if ($variationId) {
            $stockQuery->where('variation_id', $variationId);
        } elseif (!$product->has_variants) {
            $stockQuery->whereNull('variation_id');
        }
        
        // FIXED: Use available qty (qty - reserved_qty)
        $stockData = (clone $stockQuery)->selectRaw('COALESCE(SUM(qty), 0) as total_qty, COALESCE(SUM(reserved_qty), 0) as total_reserved')->first();
        $currentStock = (float) $stockData->total_qty;
        $reservedStock = (float) $stockData->total_reserved;
        $availableStock = $currentStock - $reservedStock;
        
        if ($availableStock > $product->min_stock_level) {
            return null;
        }
        
        return [
            'product_id' => $productId,
            'variation_id' => $variationId,
            'name' => $product->name,
            'sku' => $product->sku,
            'current_stock' => $currentStock,
            'reserved_stock' => $reservedStock,
            'available_stock' => $availableStock,
            'min_stock_level' => $product->min_stock_level,
            'shortage' => $product->min_stock_level - $availableStock,
        ];
    }

    /**
     * Get all low stock products
     * FIXED: Now considers reserved_qty
     */
    public static function getLowStockProducts(int $limit = 50): Collection
    {
        $products = Product::where('is_active', true)
            ->where('track_inventory', true)
            ->where('min_stock_level', '>', 0)
            ->where('has_variants', false)
            ->get(['id', 'name', 'sku', 'min_stock_level', 'unit_id']);
        
        $result = collect();
        
        foreach ($products as $product) {
            // FIXED: Calculate available stock (qty - reserved_qty)
            $stockData = StockLevel::where('product_id', $product->id)
                ->whereNull('variation_id')
                ->selectRaw('COALESCE(SUM(qty), 0) as total_qty, COALESCE(SUM(reserved_qty), 0) as total_reserved')
                ->first();
            
            $currentStock = (float) $stockData->total_qty;
            $reservedStock = (float) $stockData->total_reserved;
            $availableStock = $currentStock - $reservedStock;
            
            if ($availableStock <= $product->min_stock_level) {
                $result->push((object)[
                    'id' => $product->id,
                    'product_id' => $product->id,
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'variation_name' => null,
                    'min_stock_level' => $product->min_stock_level,
                    'current_stock' => $currentStock,
                    'reserved_stock' => $reservedStock,
                    'available_stock' => $availableStock,
                    'shortage' => $product->min_stock_level - $availableStock,
                    'type' => 'product',
                    'unit_id' => $product->unit_id,
                ]);
            }
        }
        
        return $result->sortByDesc('shortage')->take($limit)->values();
    }

    /**
     * Get all low stock variations
     * FIXED: Now considers reserved_qty
     */
    public static function getLowStockVariations(int $limit = 50): Collection
    {
        $products = Product::where('is_active', true)
            ->where('track_inventory', true)
            ->where('min_stock_level', '>', 0)
            ->where('has_variants', true)
            ->with(['variations' => function($q) {
                $q->where('is_active', true);
            }])
            ->get();
        
        $result = collect();
        
        foreach ($products as $product) {
            foreach ($product->variations as $variation) {
                // FIXED: Calculate available stock (qty - reserved_qty)
                $stockData = StockLevel::where('product_id', $product->id)
                    ->where('variation_id', $variation->id)
                    ->selectRaw('COALESCE(SUM(qty), 0) as total_qty, COALESCE(SUM(reserved_qty), 0) as total_reserved')
                    ->first();
                
                $currentStock = (float) $stockData->total_qty;
                $reservedStock = (float) $stockData->total_reserved;
                $availableStock = $currentStock - $reservedStock;
                
                if ($availableStock <= $product->min_stock_level) {
                    $result->push((object)[
                        'id' => $variation->id,
                        'product_id' => $product->id,
                        'name' => $product->name,
                        'sku' => $variation->sku,
                        'variation_name' => $variation->variation_name,
                        'min_stock_level' => $product->min_stock_level,
                        'current_stock' => $currentStock,
                        'reserved_stock' => $reservedStock,
                        'available_stock' => $availableStock,
                        'shortage' => $product->min_stock_level - $availableStock,
                        'type' => 'variation',
                        'unit_id' => $product->unit_id,
                    ]);
                }
            }
        }
        
        return $result->sortByDesc('shortage')->take($limit)->values();
    }

    /**
     * Get all low stock items
     */
    public static function getAllLowStockItems(int $limit = 50): Collection
    {
        $products = self::getLowStockProducts(100);
        $variations = self::getLowStockVariations(100);
        
        return $products->concat($variations)
            ->sortByDesc('shortage')
            ->take($limit)
            ->values();
    }

    /**
     * Get low stock count
     */
    public static function getLowStockCount(): int
    {
        return self::getAllLowStockItems(1000)->count();
    }

    /**
     * Check and return all low stock items
     */
    public static function checkAndNotify(?int $userId = null): array
    {
        return self::getAllLowStockItems(100)->toArray();
    }

    /**
     * Get out of stock items (available_stock <= 0)
     */
    public static function getOutOfStockItems(int $limit = 50): Collection
    {
        return self::getAllLowStockItems(200)
            ->filter(fn($item) => $item->available_stock <= 0)
            ->take($limit)
            ->values();
    }

    /**
     * Get stock status summary
     */
    public static function getStockStatusSummary(): array
    {
        $allLowStock = self::getAllLowStockItems(1000);
        
        return [
            'total_low_stock' => $allLowStock->count(),
            'out_of_stock' => $allLowStock->filter(fn($i) => $i->available_stock <= 0)->count(),
            'critical' => $allLowStock->filter(fn($i) => $i->available_stock > 0 && $i->available_stock <= ($i->min_stock_level * 0.5))->count(),
            'warning' => $allLowStock->filter(fn($i) => $i->available_stock > ($i->min_stock_level * 0.5))->count(),
        ];
    }

    /**
     * Get items expiring soon
     */
    public static function getExpiringItems(int $days = 30, int $limit = 50): Collection
    {
        return \Modules\Inventory\Models\Lot::where('status', 'ACTIVE')
            ->whereNotNull('expiry_date')
            ->whereDate('expiry_date', '<=', now()->addDays($days))
            ->whereDate('expiry_date', '>=', now())
            ->with(['product:id,name,sku', 'variation:id,sku,variation_name'])
            ->orderBy('expiry_date')
            ->take($limit)
            ->get()
            ->map(function ($lot) {
                return (object)[
                    'id' => $lot->id,
                    'product_id' => $lot->product_id,
                    'variation_id' => $lot->variation_id,
                    'product_name' => $lot->product?->name,
                    'sku' => $lot->variation?->sku ?? $lot->product?->sku,
                    'lot_no' => $lot->lot_no,
                    'expiry_date' => $lot->expiry_date,
                    'days_until_expiry' => now()->diffInDays($lot->expiry_date, false),
                    'received_qty' => $lot->received_qty,
                ];
            });
    }
}
