<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        if (!Schema::hasTable('project_time_logs')) {
            Schema::create('project_time_logs', function (Blueprint $table) {
                $table->id();
                
                // ========== CORE FIELDS ==========
                $table->unsignedBigInteger('task_id');
                $table->unsignedBigInteger('admin_id');
                $table->timestamp('start_time');
                $table->timestamp('end_time')->nullable();
                $table->integer('duration_minutes')->default(0);
                $table->text('description')->nullable();
                
                // ========== BILLING FIELDS ==========
                $table->boolean('is_billable')->default(true);
                $table->boolean('is_billed')->default(false);
                $table->boolean('is_running')->default(false);
                $table->decimal('hourly_rate', 12, 2)->nullable();
                $table->unsignedBigInteger('invoice_id')->nullable();
                
                // ========== GPS & LOCATION TRACKING (NEW) ==========
                $table->string('ip_address', 45)->nullable()->comment('IPv4 or IPv6 address');
                $table->decimal('latitude', 10, 8)->nullable()->comment('GPS latitude (e.g., 12.97160840)');
                $table->decimal('longitude', 11, 8)->nullable()->comment('GPS longitude (e.g., 77.59456510)');
                $table->decimal('location_accuracy', 8, 2)->nullable()->comment('GPS accuracy in meters');
                $table->string('location_address', 500)->nullable()->comment('Reverse geocoded address');
                
                // ========== END LOCATION TRACKING ==========
                $table->decimal('end_latitude', 10, 8)->nullable()->comment('GPS latitude at stop');
                $table->decimal('end_longitude', 11, 8)->nullable()->comment('GPS longitude at stop');
                $table->decimal('end_location_accuracy', 8, 2)->nullable()->comment('GPS accuracy in meters at stop');
                $table->string('end_location_address', 500)->nullable()->comment('Reverse geocoded address at stop');
                $table->string('end_ip_address', 45)->nullable()->comment('IP address at stop');
                
                // ========== PAUSE TRACKING ==========
                $table->boolean('is_paused')->default(false);
                $table->integer('total_pause_seconds')->default(0);
                
                // ========== TIMESTAMPS ==========
                $table->timestamps();
            });

            // ========== INDEXES FOR PERFORMANCE ==========
            Schema::table('project_time_logs', function (Blueprint $table) {
                // Core indexes
                $table->index('task_id');
                $table->index('admin_id');
                $table->index('is_running');
                $table->index('is_billable');
                $table->index('is_billed');
                $table->index('invoice_id');
                
                // Composite indexes for common queries
                $table->index(['task_id', 'admin_id'], 'idx_task_admin');
                $table->index(['admin_id', 'start_time'], 'idx_admin_starttime');
                $table->index(['is_running', 'admin_id'], 'idx_running_admin');
                
                // GPS search indexes (for location-based queries)
                $table->index(['latitude', 'longitude'], 'idx_gps_coordinates');
                $table->index(['end_latitude', 'end_longitude'], 'idx_end_gps_coordinates');
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('project_time_logs');
    }
};